<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\BusinessCentral\Plugin;

use Ecomero\BusinessCentral\Helper\Settings;
use Ecomero\BusinessCentral\Model\BusinessCentral;
use Ecomero\ErpCore\Model\Attribute;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\Interceptor;

class ErpPlugin
{
    protected $businessCentral;
    protected $settings;
    protected $capability;

    const ERP_NAME = "BusinessCentral";

    public function __construct(
        BusinessCentral $businessCentral,
        Settings $settings,
        Capability $capability
    ) {
        $this->businessCentral = $businessCentral;
        $this->settings = $settings;
        $this->capability = $capability;

        $this->capability->setName(ErpPlugin::ERP_NAME);
        $this->capability->setCapability(Capability::PRODUCT_IMPORT, true);
        $this->capability->setCapability(Capability::PRODUCT_PICTURE_EXPORT, true);
        $this->capability->setCapability(Capability::ORDER_EXPORT, true);
        $this->capability->setCapability(Capability::ORDER_IMPORT, true);
        $this->capability->setCapability(Capability::CREDIT_MEMO_EXPORT, true);
        $this->capability->setCapability(Capability::INVOICE_EXPORT, true);
        $this->capability->setCapability(Capability::INVOICE_IMPORT, true);
        $this->capability->setCapability(Capability::SHIPMENT_IMPORT, true);
    }

    public function isEnabled(
        Interceptor $subject
    ) : bool {
        return $this->settings->isEnabled($subject->getWebsite());
    }

    public function afterIsErpEnabled(
        Interceptor $subject,
        string $result
    ) : string {
        if ($this->isEnabled($subject)) {
            if ($result !== '') {
                $result = $result . ', ';
            }
            $result = $result . ErpPlugin::ERP_NAME;
        }
        return $result;
    }

    public function afterGetItemList(
        Interceptor $subject,
        array $result,
        String $category,
        bool $force
    ) : array {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        $rc = $this->businessCentral->getItemList($category);
        foreach ($rc->value as $item) {
            $item->commonName = $item->description;
            $result[] = $item;
        }
        return $result;
    }

    public function afterGetItemPrices(
        Interceptor $subject,
        array $result,
        String $category
    ) : array {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        $rc = $this->businessCentral->getItemPrices($category);
        foreach ($rc->value as $item) {
            $result[] = $item;
        }
        return $result;
    }

    public function afterGetItemAttributes(
        Interceptor $subject,
        array $result,
        String $category
    ) : array {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        $rc = $this->businessCentral->getItemAttributes($category);
        foreach ($rc->value as $item) {
            $result[] = $item;
        }
        return $result;
    }

    public function afterCreateDocument(
        Interceptor $subject,
        object $result,
        string $docType,
        string $customerId,
        string $currency,
        ?string $vatNumber,
        ?string $billingCompanyName,
        ?string $billingCustomerReference,
        ?string $billingTelephone,
        ?string $billingFax,
        ?string $billingStreet,
        ?string $billingCity,
        ?string $billingPostCode,
        ?string $billingCountry,
        ?string $billingRegion,
        ?string $shippingCompanyName,
        ?string $shippingCustomerReference,
        ?string $shippingTelephone,
        ?string $shippingFax,
        ?string $shippingStreet,
        ?string $shippingCity,
        ?string $shippingPostCode,
        ?string $shippingCountry,
        ?string $shippingRegion,
        ?string $shippingAgent,
        ?string $shippingService,
        string $magentoOrderId,
        string $salesPerson,
        ?string $profitCenter,
        ?string $wayOfDelivery
    ) : object {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        $rc = $this->businessCentral->createDocument(
            $docType,
            $customerId,
            $currency,
            $shippingCustomerReference,
            $shippingStreet,
            $shippingCity,
            $shippingPostCode,
            $shippingCountry,
            $shippingRegion,
            $shippingAgent,
            $shippingService,
            $magentoOrderId,
            $salesPerson
        );

        $tmpRc = (array)$rc;
        $tmpRc['customerNumber'] = $rc->billCustomerId;
        $tmpRc['number'] = $rc->no;

        return (object)$tmpRc;
    }

    public function afterUpdateDocument(
        Interceptor $subject,
        object $result,
        string $docType,
        string $erpId,
        string $magentoOrderId
    )  : object {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        $rc = $this->businessCentral->updateDocument(
            $docType,
            $erpId,
            $magentoOrderId
        );

        return $rc;
    }

    public function afterAddDocumentItem(
        Interceptor $subject,
        object $result,
        string $docType,
        string $orderId,
        string $erpItemId,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        float $discountAmount,
        float $discountPercent
    ) : object {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        $rc = $this->businessCentral->addDocumentItem(
            $docType,
            $orderId,
            $erpItemId,
            $price,
            $qtyOrdered,
            $vatCode,
            $discountAmount,
            $discountPercent
        );

        return $rc;
    }

    public function afterAddChargeItem(
        Interceptor $subject,
        object $result,
        string $docType,
        string $orderId,
        string $chargingSKU,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        string $invoiceNo,
        int $numOrderLines
    ) : object {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        $rc = $this->businessCentral->addChargeItem(
            $docType,
            $orderId,
            $chargingSKU,
            $price,
            $qtyOrdered,
            $vatCode,
            $invoiceNo,
            $numOrderLines
        );

        return $rc;
    }

    public function afterGetDocumentFromExternalId(
        Interceptor $subject,
        object $result,
        string $docType,
        string $magentoOrderId
    ) : object {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        $tmpResult = (array)$result;
        $rc = $this->businessCentral->getDocumentFromExternalId($docType, $magentoOrderId);
        if (count($rc->value) == 1) {
            $tmpResult = $rc->value[0];
            return (object)$tmpResult;
        } else {
            return $result;
        }
    }

    public function afterGetCustomerFromEmail(
        Interceptor $subject,
        object $result,
        string $email
    ) : object {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        $tmpResult = (array)$result;
        $rc = $this->businessCentral->getCustomerFromEmail($email);
        foreach ($rc->value as $item) {
            $newItem = [];
            $oldItem = $item;

            foreach ($oldItem as $key => $val) {
                if ($key === 'address') {
                    $newItem['street'] = $val->street;
                    $newItem['city'] = $val->city;
                    $newItem['countryCode'] = $val->countryLetterCode;
                    $newItem['postalCode'] = $val->postalCode;
                    $newItem['state'] = $val->state;
                } elseif ($key === 'displayName') {
                    $newItem['name'] = $val;
                } elseif ($key === 'phoneNumber') {
                    $newItem['phone'] = $val;
                } else {
                    $newItem[$key] = $val;
                }
            }

            $tmpResult[] = (object)$newItem;
        }
        return (object)$tmpResult;
    }

    public function afterCreateCustomer(
        Interceptor $subject,
        object $result,
        ?string $company,
        ?string $firstName,
        ?string $lastName,
        ?string $street,
        ?string $city,
        ?string $postCode,
        ?string $email,
        ?string $phone,
        ?string $country,
        ?string $region
    ) : object {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        $tmpResult = (array)$result;
        $rc = $this->businessCentral->createCustomer(
            $firstName,
            $lastName,
            $street,
            $city,
            $postCode,
            $email,
            $phone,
            $country,
            $region
        );
        return $rc;
    }

    public function afterGetSalesPersons(
        Interceptor $subject,
        array $result
    ) : array {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        $rc = $this->businessCentral->getSalesPersons();
        foreach ($rc->value as $item) {
            $result[] = $item;
        }
        return $result;
    }

    public function afterGetShippingAgents(
        Interceptor $subject,
        array $result
    ) : array {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        $rc = $this->businessCentral->getShippingAgents();
        foreach ($rc->value as $item) {
            $result[] = $item;
        }
        return $result;
    }

    public function afterGetAggregatedFreightOptions(
        Interceptor $subject,
        array $result
    ) : array {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        $rc = $this->businessCentral->getAggregatedFreightOptions();
        foreach ($rc as $item) {
            $result[] = $item;
        }
        return $result;
    }

    public function afterGetShippment(
        Interceptor $subject,
        array $result,
        string $orderNo
    ) : array {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        $rc = $this->businessCentral->getShippment($orderNo);
        foreach ($rc->value as $item) {
            $result[] = $item;
        }
        return $result;
    }

    public function afterGetInvoiceFromOrder(
        Interceptor $subject,
        object $result,
        string $orderNo
    ) : object {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        $rc = $this->businessCentral->getInvoiceFromOrder($orderNo);
        if (!empty($rc->value)) {
            return $rc->value[0];
        }

        return $result;
    }

    public function afterGetInvoiceOrderLines(
        Interceptor $subject,
        array $result,
        string $invoiceNo
    ) : array {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        $rc = $this->businessCentral->getInvoiceOrderLines($invoiceNo);
        foreach ($rc->value as $item) {
            $result[] = $item;
        }
        return $result;
    }

    public function afterCreateCreditMemo(
        Interceptor $subject,
        object $result,
        string $creditMemoDate,
        string $invoiceNumber,
        string $customerNumber
    ) : object {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        return $this->businessCentral->createCreditMemo(
            $creditMemoDate,
            $invoiceNumber,
            $customerNumber,
            $this->settings->getSalesPerson($subject->getWebsite())
        );
    }

    public function afterAddCreditMemoItem(
        Interceptor $subject,
        object $result,
        string $creditMemoId,
        string $bcItemId,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        float $discountAmount,
        float $discountPercent
    ) : object {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        return $this->businessCentral->addCreditMemoItem(
            $creditMemoId,
            $bcItemId,
            $price,
            $qtyOrdered,
            $vatCode,
            $discountAmount,
            $discountPercent
        );
    }

    public function afterAddCreditMemoCharge(
        Interceptor $subject,
        object $result,
        string $creditMemoId,
        string $chargingSKU,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        string $invoiceNo,
        int $numOrderLines
    ) : object {
        if (!$this->isEnabled($subject)) {
            return $result;
        }

        return $this->businessCentral->addCreditMemoCharge(
            $creditMemoId,
            $chargingSKU,
            $price,
            $qtyOrdered,
            $vatCode,
            $invoiceNo,
            $numOrderLines
        );
    }

    public function afterAddPicture(
        Interceptor $subject,
        int $result,
        string $itemId,
        string $imageUrl
    ) : int {
        return $this->businessCentral->addPicture(
            $itemId,
            $imageUrl
        );
    }

    public function afterGetCapabilities(
        Interceptor $subject,
        array $result
    ) : array {
        $result[] = $this->capability;

        return $result;
    }

    public function afterCreateCommonName(
        Interceptor $subject,
        bool $result,
        \Ecomero\ErpCore\Model\ResourceModel\Item\Collection $itemCollection
    ) : bool {
        $commonNameAttributes = [];
        if ($this->settings->getCommonNameAttributes($subject->getWebsite())) {
            $commonNameAttributes = explode(',', $this->settings->getCommonNameAttributes($subject->getWebsite()));
        }

        $configurableAttributes = '';
        if ($this->settings->getConfigurableAttributes($subject->getWebsite())) {
            $configurableAttributes = explode(',', $this->settings->getConfigurableAttributes($subject->getWebsite()));
        }

        foreach ($itemCollection->getItems() as $item) {
            $item->commonName = '';
            foreach ($commonNameAttributes as $attribute) {
                $attribute = strtolower($attribute);
                if (key_exists($attribute, $item->attributes)) {
                    if ($item->commonName === '') {
                        $item->commonName = $item->attributes[ $attribute ]->value;
                    } else {
                        $item->commonName = $item->commonName . '-' . $item->attributes[ $attribute ]->value;
                    }
                }
            }

            if ($configurableAttributes !== null && $configurableAttributes !== '') {

                // Clear usedInConfiguration since it may have been set as deault to true by erp core
                foreach ($item->attributes as $key => $value) {
                    $item->attributes[$key]->usedInConfiguration = Attribute::NO;
                }

                // If attribute is a configurable product attribute, then set flag
                foreach ($configurableAttributes as $attribute) {
                    $attribute = strtolower($attribute);
                    if (key_exists($attribute, $item->attributes)) {
                        $item->attributes[ $attribute ]->usedInConfiguration = Attribute::YES;
                    }
                }
            }
        }
        return true;
    }
}
