<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\BusinessCentral\Model;

use Ecomero\BusinessCentral\Helper\Settings;
use Ecomero\ErpCore\Model\Attribute;
use Ecomero\ErpCore\Model\Capability;

class BusinessCentralAdapter extends \Ecomero\ErpCore\Model\ErpAdapter implements \Ecomero\ErpCore\Model\ErpAdapterInterface
{
    protected $businessCentral;
    protected $settings;
    protected $capability;

    const ERP_NAME = "BusinessCentral";

    public function __construct(
        BusinessCentralAPI $businessCentral,
        Settings $settings,
        Capability $capability,
        \Magento\Store\Model\StoreManagerInterface $storeManager
    ) {
        parent::__construct($storeManager);

        $this->businessCentral = $businessCentral;
        $this->settings = $settings;
        $this->capability = $capability;

        $this->capability->setName(Self::ERP_NAME);
        $this->capability->setCapability(Capability::PRODUCT_IMPORT, true);
        $this->capability->setCapability(Capability::PRODUCT_PICTURE_EXPORT, true);
        $this->capability->setCapability(Capability::ORDER_EXPORT, true);
        $this->capability->setCapability(Capability::ORDER_IMPORT, true);
        $this->capability->setCapability(Capability::CREDIT_MEMO_EXPORT, true);
        $this->capability->setCapability(Capability::INVOICE_EXPORT, true);
        $this->capability->setCapability(Capability::INVOICE_IMPORT, true);
        $this->capability->setCapability(Capability::SHIPMENT_IMPORT, true);
        $this->capability->setCapability(Capability::RETURN_IMPORT, true);
    }

    public function isErpEnabled() : string
    {
        if ($this->settings->isEnabled($this->getWebsite())) {
            return Self::ERP_NAME;
        }
        return '';
    }

    public function getItemList(
        String $category,
        bool $force
    ) : array {
        $result = [];
        $rc = $this->businessCentral->getItemList($category);
        foreach ($rc->value as $item) {
            $item->commonName = $item->description;
            $result[] = $item;
        }
        return $result;
    }

    public function getItemPrices(
        String $category
    ) : array {
        $result = [];
        $rc = $this->businessCentral->getItemPrices($category);
        foreach ($rc->value as $item) {
            $result[] = $item;
        }
        return $result;
    }

    public function getItemAttributes(
        String $category
    ) : array {
        $result = [];
        $rc = $this->businessCentral->getItemAttributes($category);
        foreach ($rc->value as $item) {
            $result[] = $item;
        }
        return $result;
    }

    public function createDocument(
        string $docType,
        string $customerId,
        string $currency,
        array $billingAddress,
        array $shippingAddress,
        ?string $shippingAgent,
        ?string $shippingService,
        ?string $shippingPickupLocationId,
        string $magentoOrderId,
        string $salesPerson,
        ?string $profitCenter,
        ?string $termsOfDelivery,
        ?string $wayOfDelivery,
        ?string $customerCategory,
        ?string $customerDistrict
    ) : object {
        $rc = $this->businessCentral->createDocument(
            $docType,
            $customerId,
            $currency,
            $shippingAddress['firstname'] . ' ' . $shippingAddress['lastname'],
            $shippingAddress['street'],
            $shippingAddress['city'],
            $shippingAddress['postcode'],
            $shippingAddress['country_id'],
            $shippingAddress['region'],
            $shippingAgent,
            $shippingService,
            $shippingPickupLocationId,
            $magentoOrderId,
            $salesPerson
        );

        $tmpRc = (array)$rc;
        $tmpRc['customerNumber'] = $rc->billCustomerId;
        $tmpRc['number'] = $rc->no;

        return (object)$tmpRc;
    }

    public function updateDocument(
        string $docType,
        string $erpId,
        string $magentoOrderId
    )  : object {
        $rc = $this->businessCentral->updateDocument(
            $docType,
            $erpId,
            $magentoOrderId
        );

        return $rc;
    }

    public function addDocumentItem(
        string $docType,
        string $orderId,
        string $erpItemId,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        float $discountAmount,
        float $discountPercent,
        ?string $comment
    ) : void {
        $this->businessCentral->addDocumentItem(
            $docType,
            $orderId,
            $erpItemId,
            $price,
            $qtyOrdered,
            $vatCode,
            $discountAmount,
            $discountPercent
        );

        if ($comment) {
            $comment = str_replace("\n", " - ", $comment);
            $this->businessCentral->addDocumentComment(
                $docType,
                $orderId,
                $comment
            );
        }
    }

    public function addChargeItem(
        string $docType,
        string $orderId,
        string $chargingSKU,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        string $invoiceNo,
        int $numOrderLines
    ) : void {
        $this->businessCentral->addChargeItem(
            $docType,
            $orderId,
            $chargingSKU,
            $price,
            $qtyOrdered,
            $vatCode,
            $invoiceNo,
            $numOrderLines
        );
    }

    public function getDocumentFromExternalId(
        string $docType,
        string $magentoOrderId
    ) : object {
        $tmpResult = [];
        $rc = $this->businessCentral->getDocumentFromExternalId($docType, $magentoOrderId);
        if (count($rc->value) == 1) {
            $tmpResult = $rc->value[0];
            return (object)$tmpResult;
        } else {
            return (object)[];
        }
    }

    public function getCustomerFromEmail(
        string $email
    ) : object {
        $tmpResult = [];
        $rc = $this->businessCentral->getCustomerFromEmail($email);
        foreach ($rc->value as $item) {
            $newItem = [];
            $oldItem = $item;

            foreach ($oldItem as $key => $val) {
                if ($key === 'address') {
                    $newItem['street'] = $val->street;
                    $newItem['city'] = $val->city;
                    $newItem['countryCode'] = $val->countryLetterCode;
                    $newItem['postalCode'] = $val->postalCode;
                    $newItem['state'] = $val->state;
                } elseif ($key === 'displayName') {
                    $newItem['name'] = $val;
                } elseif ($key === 'phoneNumber') {
                    $newItem['phone'] = $val;
                } else {
                    $newItem[$key] = $val;
                }
            }

            $tmpResult[] = (object)$newItem;
        }
        return (object)$tmpResult;
    }

    public function createCustomer(
        ?string $company,
        ?string $firstName,
        ?string $lastName,
        ?string $street,
        ?string $city,
        ?string $postCode,
        ?string $email,
        ?string $phone,
        ?string $country,
        ?string $region
    ) : object {
        $rc = $this->businessCentral->createCustomer(
            $firstName,
            $lastName,
            $street,
            $city,
            $postCode,
            $email,
            $phone,
            $country,
            $region
        );
        return $rc;
    }

    public function getSalesPersons() : array
    {
        $result = [];
        $rc = $this->businessCentral->getSalesPersons();
        foreach ($rc->value as $item) {
            $result[] = $item;
        }
        return $result;
    }

    public function getShippingAgents() : array
    {
        $result = [];
        $rc = $this->businessCentral->getShippingAgents();
        foreach ($rc->value as $item) {
            $result[] = $item;
        }
        return $result;
    }

    public function getAggregatedFreightOptions() : array
    {
        $result = [];
        $rc = $this->businessCentral->getAggregatedFreightOptions();
        foreach ($rc as $item) {
            $result[] = $item;
        }
        return $result;
    }

    public function getNonInventoryItems() : array
    {
        return $this->businessCentral->getItemListNonInventory();
    }

    public function getShippment(
        string $orderNo
    ) : array {
        $result = [];
        $rc = $this->businessCentral->getShippment($orderNo);
        foreach ($rc->value as $item) {
            $result[] = $item;
        }
        return $result;
    }

    public function getInvoiceFromOrder(
        string $orderNo
    ) : object {
        $result = [];
        $rc = $this->businessCentral->getInvoiceFromOrder($orderNo);
        if (!empty($rc->value)) {
            return $rc->value[0];
        }
        return (object)$result;
    }

    public function getInvoiceOrderLines(
        string $invoiceNo
    ) : array {
        $result = [];
        $rc = $this->businessCentral->getInvoiceOrderLines($invoiceNo);
        foreach ($rc->value as $item) {
            $result[] = $item;
        }
        return $result;
    }

    public function createCreditMemo(
        string $creditMemoDate,
        string $orderNumber,
        string $invoiceNumber,
        string $customerNumber
    ) : object {
        $rc = $this->businessCentral->createCreditMemo(
            $creditMemoDate,
            $orderNumber,
            $invoiceNumber,
            $customerNumber,
            $this->settings->getSalesPerson($this->getWebsite())
        );

        if ($rc->id) {
            $this->businessCentral->addCreditMemoComment(
                $rc->id,
                'Credit for invoice : ' . $invoiceNumber
            );
        }

        return $rc;
    }

    public function addCreditMemoItem(
        string $creditMemoId,
        string $bcItemId,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        float $discountAmount,
        float $discountPercent
    ) : void {
        $this->businessCentral->addCreditMemoItem(
            $creditMemoId,
            $bcItemId,
            $price,
            $qtyOrdered,
            $vatCode,
            $discountAmount,
            $discountPercent
        );
    }

    public function addCreditMemoCharge(
        string $creditMemoId,
        string $chargingSKU,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        string $invoiceNo,
        int $numOrderLines
    ) : void {
        $this->businessCentral->addCreditMemoCharge(
            $creditMemoId,
            $chargingSKU,
            $price,
            $qtyOrdered,
            $vatCode,
            $invoiceNo,
            $numOrderLines
        );
    }

    public function postCreditMemo(string $creditMemoId) : void
    {
        $this->businessCentral->postCreditMemo(
            $creditMemoId
        );
    }

    public function addPicture(
        string $itemId,
        string $imageUrl
    ) : int {
        return $this->businessCentral->addPicture(
            $itemId,
            $imageUrl
        );
    }

    public function getCapabilities() : array
    {
        $result[] = $this->capability;
        return $result;
    }

    public function createCommonName(
        \Ecomero\ErpCore\Model\ResourceModel\Item\Collection $itemCollection
    ) : bool {
        $commonNameAttributes = [];
        if ($this->settings->getCommonNameAttributes($this->getWebsite())) {
            $commonNameAttributes = explode(',', $this->settings->getCommonNameAttributes($this->getWebsite()));
        }

        $configurableAttributes = '';
        if ($this->settings->getConfigurableAttributes($this->getWebsite())) {
            $configurableAttributes = explode(',', $this->settings->getConfigurableAttributes($this->getWebsite()));
        }

        foreach ($itemCollection->getItems() as $item) {
            $item->commonName = '';
            foreach ($commonNameAttributes as $attribute) {
                $attribute = strtolower($attribute);
                if (key_exists($attribute, $item->attributes)) {
                    if ($item->commonName === '') {
                        $item->commonName = $item->attributes[ $attribute ]->value;
                    } else {
                        $item->commonName = $item->commonName . '-' . $item->attributes[ $attribute ]->value;
                    }
                }
            }

            if ($configurableAttributes !== null && $configurableAttributes !== '') {

                // Clear usedInConfiguration since it may have been set as deault to true by erp core
                foreach ($item->attributes as $key => $value) {
                    $item->attributes[$key]->usedInConfiguration = Attribute::NO;
                }

                // If attribute is a configurable product attribute, then set flag
                foreach ($configurableAttributes as $attribute) {
                    $attribute = strtolower($attribute);
                    if (key_exists($attribute, $item->attributes)) {
                        $item->attributes[ $attribute ]->usedInConfiguration = Attribute::YES;
                    }
                }
            }
        }
        return true;
    }

    public function getReturns() : array
    {
        $result = [];
        $rc = $this->businessCentral->getReturns();
        foreach ($rc->value as $item) {
            $result[] = $item;
        }
        return $result;
    }
}
