<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\BusinessCentral\Model;

use Ecomero\BusinessCentral\Helper\Settings;
use Magento\Framework\Filesystem\Driver\File;
use Zend\Http\Client;
use Zend\Http\Request;

class BusinessCentralAPI
{
    protected $_zendClient;
    protected $settings;
    protected $companyNameIds;
    protected $fileSystem;

    public function __construct(
        Client $zendClient,
        Settings $settings,
        File $fileSystem
    ) {
        $this->_zendClient = $zendClient;
        $this->settings = $settings;
        $this->companyNameIds = [];
        $this->fileSystem = $fileSystem;
    }

    private function getEndPoint()
    {
        $endpoint = '';
        if ($this->settings->getInstallation() == "cloud") {
            $endpoint = 'https://api.businesscentral.dynamics.com/v2.0/' .
                        $this->settings->getTenantId() .
                        '/' .
                        $this->settings->getEnvironment();
        } else {
            $endpoint = $this->settings->getAPIEndPoint();
        }

        return $endpoint;
    }

    public function getEndPointForAPI(string $api = '/api/v1.0') : string
    {
        return $this->getEndPoint() . $api . '/companies(' . $this->getCompanyId() . ')/';
    }

    private function setHeaders(Request $request) : void
    {
        $token = base64_encode($this->settings->getUserName() . ':' . $this->settings->getAPIKey());

        $request->getHeaders()->addHeaders([
            'User-Agent' => 'Magento',
            'Cache-Control' => 'no-cache',
            'Accept' => 'application/json',
            'Authorization' => ('Basic ' . $token),
            'Accept-Encoding' => 'gzip',
            'If-Match' => '*'

        ]);
    }

    private function sendRequest(
        string $endpoint,
        string $type = \Zend\Http\Request::METHOD_GET,
        string $payLoad = null,
        bool $ignoreResponse = false
    ) : object {
        $this->_zendClient->reset();
        $request = new Request();
        $request->setUri($this->urlEncode($endpoint));
        $request->setMethod($type);
        $this->setHeaders($request);

        if ($payLoad !== null) {
            $request->setContent($payLoad);
            $request->getHeaders()->addHeaders([
                'Content-Length' => strlen($payLoad),
                'Content-Type' => 'application/json'
            ]);
        }

        $this->_zendClient->send($request);
        $response = $this->_zendClient->getResponse();
        $json = (object)'';
        if ($ignoreResponse === false || $response->getStatusCode() > 299) {
            $json = json_decode($response->getBody());
        }

        if ($response->getStatusCode() > 299) {
            if ($json) {
                if (property_exists($json, "error")) {
                    throw new \RuntimeException('ERROR: ' . $json->error->message);
                }
            }
            throw new \RuntimeException('ERROR: Business Central replied with status code ' .
                                        $response->getStatusCode() .
                                        ' when calling ' .
                                        $endpoint);
        }
        return $json;
    }

    private function getCompanyId() : string
    {
        $companyName = $this->settings->getAPICompanyName();
        if (array_key_exists($companyName, $this->companyNameIds)) {
            return $this->companyNameIds[$companyName];
        }

        $json = $this->getCompanies();
        $companyNames = '';
        if ($json != null) {
            foreach ($json->value as $companyEntry) {
                $this->companyNameIds[$companyEntry->name] = $companyEntry->id;
                $companyNames = $companyNames . $companyEntry->name . ', ';
                if ($companyEntry->name == $companyName) {
                    return $companyEntry->id;
                }
            }
        }
        $companyNames = rtrim($companyNames, ', ');
        throw new \RuntimeException('ERROR: Company with the name ' . $companyName .
                                    ' was not found in Business Central. The following companies are available ' .
                                    $companyNames);
    }

    private function urlEncode(string $url) : string
    {
        $url = str_replace(" ", '%20', $url);
        $url = str_replace("'", '%27', $url);
        return $url;
    }

    private function getFromBusinessCentral(string $url) : object
    {
        $endpoint = $this->getEndPointForAPI('/api/eComero/magento/v1.0') . $url;
        return $this->sendRequest($endpoint);
    }

    public function getItemList(string $category) : object
    {
        $url = 'items?$filter=category eq \'' . $category . '\'';
        return $this->getFromBusinessCentral($url);
    }

    public function getAggregatedFreightOptions() : array
    {
        $rc = [];
        $itemFreightList = $this->getItemListNonInventory();
        $chargeFreightList = $this->getItemChargesList();

        foreach ($itemFreightList as $item) {
            $rc[] = [
                'id' => $item->id,
                'type' => 'non-inv',
                'priceIncludesVAT' => $item->priceIncludesVAT ? 'true' : 'false',
                'description' => $item->description . ' (Non-Inv Item)',
                'taxRate' => $item->taxRate
            ];
        }

        foreach ($chargeFreightList->value as $charge) {
            $rc[] = [
                'id' => $charge->sku,
                'type' => 'charge',
                'priceIncludesVAT' => 'true',
                'description' => $charge->description . ' (Charge Item)',
                'taxRate' => 25 // TODO: Hardcoded to 25%, charges are not used most of the times a SKU is used
            ];
        }

        return $rc;
    }

    public function getItemListNonInventory() : array
    {
        $rc = [];
        $url = 'items?$filter=itemType eq \'Non-Inventory\'';
        $noneInvItemList = $this->getFromBusinessCentral($url);

        foreach ($noneInvItemList->value as $item) {
            $rc[] = (object)[
                'id' => $item->id,
                'type' => 'non-inv',
                'priceIncludesVAT' => $item->priceIncludesVAT ? 'true' : 'false',
                'description' => $item->description . ' (Non-Inv Item)',
                'taxRate' => $item->taxRate
            ];
        }
        return $rc;
    }

    public function getItemPrices(string $category) : object
    {
        $url = 'itemPrices?$filter=category eq \'' . $category . '\'&$select=id,sku,currency,priceCurrency';
        return $this->getFromBusinessCentral($url);
    }

    public function getItemAttributes(string $category) : object
    {
        $url = 'itemAttributes?$filter=category eq \'' . $category . '\'&$select=id,sku,type,name,value';
        return $this->getFromBusinessCentral($url);
    }

    public function getItemChargesList() : object
    {
        $url = 'itemCharges';
        return $this->getFromBusinessCentral($url);
    }

    public function getShippment(string $orderNo) : object
    {
        $url = 'salesShipments?$filter=magentoNo eq \'' . $orderNo . '\'&$orderby=shippingNo';
        return $this->getFromBusinessCentral($url);
    }

    public function getSalesPersons() : object
    {
        $url = 'salesPersons?$filter=blocked eq false';
        return $this->getFromBusinessCentral($url);
    }

    public function getShippingAgents() : object
    {
        $url = 'shippingAgents';
        return $this->getFromBusinessCentral($url);
    }

    public function getCustomerFromEmail(string $email) : object
    {
        $endpoint = $this->getEndPointForAPI() . 'customers?$filter=email eq \'' . $email . '\'';
        return $this->sendRequest($endpoint);
    }

    public function getInvoiceFromOrder(string $orderNo) : object
    {
        $endpoint = $this->getEndPointForAPI() .
                    'salesInvoices?$filter=orderNumber eq \'' .
                    $orderNo . '\'&$select=id,number';
        return $this->sendRequest($endpoint);
    }

    public function getInvoiceOrderLines(string $invoiceNo) : object
    {
        $endpoint = $this->getEndPointForAPI() .
                    'salesInvoices(' . $invoiceNo . ')/salesInvoiceLines';
        return $this->sendRequest($endpoint);
    }

    public function getCompanies() : object
    {
        $endpoint =  $this->getEndPoint() . '/api/v1.0/companies';
        return $this->sendRequest($endpoint);
    }

    public function createCustomer(
        string $firstName,
        string $lastName,
        string $street,
        string $city,
        string $postCode,
        string $email,
        ?string $phone,
        ?string $country,
        ?string $region
    ) : object {
        $endpoint = $this->getEndPointForAPI() . 'customers';

        $post_data = json_encode([ 'displayName' => $firstName . ' ' . $lastName,
                                        'phoneNumber' => $phone ?? '',
                                        'email' => $email,
                                        'address' => [
                                            'street' => $street,
                                            'city' => $city,
                                            'state' => $region,
                                            'countryLetterCode' => $country ?? '',
                                            'postalCode' => $postCode ?? ''
                                        ]  ]);
        return $this->sendRequest($endpoint, \Zend\Http\Request::METHOD_POST, $post_data);
    }

    public function createDocument(
        string $docType,
        string $customerId,
        string $currency,
        string $customerName,
        string $street,
        string $city,
        string $postCode,
        ?string $country,
        ?string $region,
        ?string $shippingAgent,
        ?string $shippingService,
        ?string $shippingPickupLocationId,
        string $magentoOrderId,
        string $salesPerson
    ) : object {
        if ($currency === $this->settings->getLCY()) {
            // Do not set currency for the base currency
            $currency = '';
        }

        $endpoint = $this->getEndPointForAPI('/api/eComero/magento/v1.0') . 'itemSalesOrders';

        $post_data = json_encode([  'docType' => $docType,
                                    'sellCustomerId' => $customerId,
                                    'billCustomerId' => $customerId,
                                    'currencyCode' => $currency,
                                    'salesperson' => $salesPerson,
                                    'name' => $customerName,
                                    'street' => $street,
                                    'city' => $city,
                                    'state' => $region,
                                    'countryLetterCode' => $country,
                                    'postalCode' => $postCode,
                                    // 'shippingMethodCode' => 'CFR',
                                    'shippingAgentCode' => $shippingAgent,
                                    'shippingAgentServiceCode' => $shippingService,
                                    'shippingAdvice' => 'Complete', // "Partial"
                                    'pickupLocationId' => $shippingPickupLocationId ?? '',
                                    'externalDocument' => $magentoOrderId
                                  ]);
        return $this->sendRequest($endpoint, \Zend\Http\Request::METHOD_POST, $post_data);
    }

    public function updateDocument(
        string $docType,
        string $businessCentralId,
        string $magentoOrderId
    ) : object {
        $endpoint = $this->getEndPointForAPI() . 'sales' . $docType . 's(' . $businessCentralId . ')';
        $post_data = json_encode([ 'externalDocumentNumber' => $magentoOrderId ]);
        return $this->sendRequest($endpoint, \Zend\Http\Request::METHOD_PATCH, $post_data);
    }

    public function getDocumentFromExternalId(
        string $docType,
        string $magentoOrderId
    ) : object {
        $endpoint = $this->getEndPointForAPI() .
                    'sales' . $docType . 's?$filter=externalDocumentNumber eq \'' .
                    $magentoOrderId . '\'';
        return $this->sendRequest($endpoint);
    }

    public function addDocumentItem(
        string $docType,
        string $orderId,
        string $erpItemId,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        float $discountAmount,
        float $discountPercent
    ) : object {
        $endpoint = $this->getEndPointForAPI() . 'sales' . $docType . 's(' . $orderId . ')/sales' . $docType . 'Lines';

        $post_data = json_encode([ 'itemId' => $erpItemId,
                                        'lineType' =>  'Item',
                                        'quantity' => ((float)$qtyOrdered),
                                        'shipQuantity' => ((float)$qtyOrdered),
                                        'unitPrice' => ((float)$price),
                                        'discountAmount' => ((float)$discountAmount),
                                        'discountPercent' => ((float)$discountPercent),
                                        'taxCode' => $vatCode ]);
        return $this->sendRequest($endpoint, \Zend\Http\Request::METHOD_POST, $post_data);
    }

    public function addDocumentComment(
        string $docType,
        string $orderId,
        string $comment
    ) : object {
        $endpoint = $this->getEndPointForAPI() . 'sales' . $docType . 's(' . $orderId . ')/sales' . $docType . 'Lines';

        $post_data = json_encode([  'lineType' =>  'Comment',
                                    'description' => $comment]);

        return $this->sendRequest($endpoint, \Zend\Http\Request::METHOD_POST, $post_data);
    }

    public function addChargeItem(
        string $docType,
        string $orderId,
        string $chargingSKU,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        string $invoiceNo,
        int $numOrderLines
    ) : void {
        $endpoint = $this->getEndPointForAPI() . 'sales' . $docType . 's(' . $orderId . ')/sales' . $docType . 'Lines';

        $post_data = json_encode([ 'lineType' =>  'Charge',
                                        'description' => $chargingSKU,
                                        'quantity' => ((float)$qtyOrdered),
                                        'unitPrice' => ((float)$price),
                                        'taxCode' => $vatCode ]);
        $this->sendRequest($endpoint, \Zend\Http\Request::METHOD_POST, $post_data);

        $endpoint = $this->getEndPointForAPI('/api/eComero/magento/v1.0') . 'itemChargeAssignments';

        $post_data = json_encode([ 'docType' =>  $docType,
                                        'invoiceNo' =>  $invoiceNo,
                                        "docLineNo" => 10000 * ($numOrderLines + 1),
                                        "lineNo" => 10000,
                                        "appliesToDocLineNo" => 10000,
                                        "chargingSKU" => $chargingSKU,
                                        "qtyToAssign" => 1,
                                        "qtyAssigned" => 0,
                                        "unitCost" => ((float)$price),
                                        "amountToAssign" => ((float)$price),
                                        "appliesToDocType" => $docType,
                                        "appliesToDocNo" => $invoiceNo,
                                        "appliesToDocLineAmount" => 0 ]);

        $this->sendRequest($endpoint, \Zend\Http\Request::METHOD_POST, $post_data);
    }

    public function createCreditMemo(
        string $creditMemoDate,
        string $orderNumber,
        string $invoiceNumber,
        string $customerNumber,
        string $salesPerson,
        string $currencyCode
    ) : object {
        $endpoint = $this->getEndPointForAPI() . 'salesCreditMemos';
        $post_data = json_encode([  'creditMemoDate' => $creditMemoDate,
                                    'externalDocumentNumber' => $orderNumber,
        //                            'invoiceNumber' => $invoiceNumber,
                                    'customerNumber' => $customerNumber,
                                    'salesperson' => $salesPerson,
                                    'currencyCode' => $currencyCode ]);
        return $this->sendRequest($endpoint, \Zend\Http\Request::METHOD_POST, $post_data);
    }

    public function addCreditMemoComment(
        string $creditMemoId,
        string $comment
    ) : object {
        $endpoint = $this->getEndPointForAPI() . 'salesCreditMemos(' . $creditMemoId . ')/salesCreditMemoLines';

        $post_data = json_encode([  'lineType' =>  'Comment',
                                    'description' => $comment]);
        return $this->sendRequest($endpoint, \Zend\Http\Request::METHOD_POST, $post_data);
    }

    public function addCreditMemoItem(
        string $creditMemoId,
        string $bcItemId,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        float $discountAmount,
        float $discountPercent
    ) : object {
        $endpoint = $this->getEndPointForAPI() . 'salesCreditMemos(' . $creditMemoId . ')/salesCreditMemoLines';

        $post_data = json_encode([ 'itemId' => $bcItemId,
                                        'lineType' =>  'Item',
                                        'quantity' => ((float)$qtyOrdered),
                                        'unitPrice' => ((float)$price),
                                        'discountAmount' => ((float)$discountAmount),
                                        'discountPercent' => ((float)$discountPercent),
                                        'taxCode' => $vatCode ]);
        return $this->sendRequest($endpoint, \Zend\Http\Request::METHOD_POST, $post_data);
    }

    public function addCreditMemoCharge(
        string $creditMemoId,
        string $chargingSKU,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        string $invoiceNo,
        int $numOrderLines
    ) : void {
        $endpoint = $this->getEndPointForAPI() . 'salesCreditMemos(' . $creditMemoId . ')/salesCreditMemoLines';

        $post_data = json_encode([ 'lineType' =>  'Charge',
                                        'description' => $chargingSKU,
                                        'quantity' => ((float)$qtyOrdered),
                                        'unitPrice' => ((float)$price),
                                        'taxCode' => $vatCode ]);
        $this->sendRequest($endpoint, \Zend\Http\Request::METHOD_POST, $post_data);

        $endpoint = $this->getEndPointForAPI('/api/eComero/magento/v1.0') . 'itemChargeAssignments';

        $post_data = json_encode([ 'docType' =>  'Credit Memo',
                                        'invoiceNo' =>  $invoiceNo,
                                        "docLineNo" => 10000 * ($numOrderLines + 1),
                                        "lineNo" => 10000,
                                        "appliesToDocLineNo" => 10000,
                                        "chargingSKU" => $chargingSKU,
                                        "qtyToAssign" => 1,
                                        "qtyAssigned" => 0,
                                        "unitCost" => ((float)$price),
                                        "amountToAssign" => ((float)$price),
                                        "appliesToDocType" => "Credit Memo",
                                        "appliesToDocNo" => $invoiceNo,
                                        "appliesToDocLineAmount" => 0 ]);

        $this->sendRequest($endpoint, \Zend\Http\Request::METHOD_POST, $post_data);
    }

    public function postCreditMemo(
        string $creditMemoId
    ) : object {
        $endpoint = $this->getEndPointForAPI() . 'salesCreditMemos(' . $creditMemoId . ')/Microsoft.NAV.post';
        return $this->sendRequest($endpoint, \Zend\Http\Request::METHOD_POST, '', true);
    }

    public function addPicture(
        string $itemId,
        string $imageUrl
    ) : int {
        $token = base64_encode($this->settings->getUserName() . ':' . $this->settings->getAPIKey());
        $endpoint = $this->getEndPointForAPI() . 'items(' . $itemId . ')/picture(' . $itemId . ')/content';

        $this->_zendClient->reset();
        $request = new Request();
        $request->setUri($endpoint);
        $request->setMethod(\Zend\Http\Request::METHOD_PATCH);

        $request->getHeaders()->addHeaders([
            'User-Agent' => 'Magento',
            'Cache-Control' => 'no-cache',
            'Content-Type' => 'application/octet-stream',
            'If-Match' => '*',
            'Accept' => 'application/json',
            'Authorization' => ('Basic ' . $token),
            'Accept-Encoding' => 'gzip'
        ]);

        try {
            $content = $this->fileSystem->fileGetContents($imageUrl);
        } catch (\Magento\Framework\Exception\FileSystemException $exception) {
            $content = false;
        }
        if ($content) {
            $request->setContent($content);

            $this->_zendClient->send($request);
            $response = $this->_zendClient->getResponse();
            return $response->getStatusCode();
        }
        return 500;
    }

    public function getReturns() : object
    {
        $endpoint = $this->getEndPointForAPI('/api/eComero/inventory/v1.0') .
                    'returnItems?$filter=processed eq false';
        return $this->sendRequest($endpoint);
    }
}

//
//
// DIMENSIONS
//
//
/*

Hämta lista på dimensions

https://erp.closely-official.com/sandbox/BC365Test/api/v1.0/companies(383f5c6d-29b3-44a3-adc2-1caa18fe9a62)/dimensions


{
    "@odata.context": "http://localhost:17059/BC365Test/api/v1.0/$metadata#companies(383f5c6d-29b3-44a3-adc2-1caa18fe9a62)/dimensions",
    "value": [
        {
            "@odata.etag": "W/\"JzQ0OzhFSXlFdjlYL1VpK1VJbDhadDBRK2J3Slhybm1rakxMRnFVODljYlI0VEE9MTswMDsn\"",
            "id": "298bb0f3-76e7-ea11-80cb-000d3ab3f0e3",
            "code": "MARKNAD",
            "displayName": "Marknad",
            "lastModifiedDateTime": "2020-08-26T08:34:27.88Z"
        },
        {
            "@odata.etag": "W/\"JzQ0O3JYWVEyYVFTNFhGWE5CU3VON3AzWmt1Zk91R3VtMU1JQUdxSlYxQ3pTbms9MTswMDsn\"",
            "id": "33a36d9c-97e2-e911-bb16-001dd8b75be6",
            "code": "AVDELNING",
            "displayName": "Avdelning",
            "lastModifiedDateTime": "0001-01-01T00:00:00Z"
        }
    ]
}

*/

/*

Hämta värden för dimension

https://erp.closely-official.com/sandbox/BC365Test/api/v1.0/companies(383f5c6d-29b3-44a3-adc2-1caa18fe9a62)/dimensions(298bb0f3-76e7-ea11-80cb-000d3ab3f0e3)/dimensionValues


{
    "@odata.context": "http://localhost:17059/BC365Test/api/v1.0/$metadata#companies(383f5c6d-29b3-44a3-adc2-1caa18fe9a62)/dimensions(298bb0f3-76e7-ea11-80cb-000d3ab3f0e3)/dimensionValues",
    "value": [
        {
            "@odata.etag": "W/\"JzQ0O0lXRERJNGpGaTRxZVFhNnI1Wk9nN1BkTmxET29FN1hlN0M4N3BTKzhmYjA9MTswMDsn\"",
            "id": "d78c8afc-76e7-ea11-80cb-000d3ab3f0e3",
            "code": "DK",
            "displayName": "Denmark",
            "lastModifiedDateTime": "2020-08-26T08:34:42.81Z"
        },
        {
            "@odata.etag": "W/\"JzQ0O3FNZVExRUx2L1VJbkVJQ3M3eExOaWg4VDEwazBzcWQ2b1RWQ1R0MHR3Y3c9MTswMDsn\"",
            "id": "606b3299-7de7-ea11-80cb-000d3ab3f0e3",
            "code": "DE",
            "displayName": "Germany",
            "lastModifiedDateTime": "2020-08-26T09:22:02.537Z"
        },
        {
            "@odata.etag": "W/\"JzQ0O0JXSXdMVmYyd1JPTzY4S0g0bEVrbkhMT3dKeUgwMjJGeGVKM283b0wxVDA9MTswMDsn\"",
            "id": "626b3299-7de7-ea11-80cb-000d3ab3f0e3",
            "code": "EU",
            "displayName": "Europe",
            "lastModifiedDateTime": "2020-08-26T09:22:06.887Z"
        },
        {
            "@odata.etag": "W/\"JzQ0OzRTWWMwVzd6VndPR2NpVWxiRXZmQnFPcXJMS0RubGhWdlVURGFqWm1RcDQ9MTswMDsn\"",
            "id": "646b3299-7de7-ea11-80cb-000d3ab3f0e3",
            "code": "FR",
            "displayName": "France",
            "lastModifiedDateTime": "2020-08-26T09:22:10.75Z"
        },
        {
            "@odata.etag": "W/\"JzQ0O2hzYkVBaXB1eHlTb1FQb0w0MUdacENLdE5oK2RNSUdUKzBNaWNhbzlFOEk9MTswMDsn\"",
            "id": "76b891a0-7de7-ea11-80cb-000d3ab3f0e3",
            "code": "SE",
            "displayName": "Sweden",
            "lastModifiedDateTime": "2020-08-26T09:22:14.907Z"
        },
        {
            "@odata.etag": "W/\"JzQ0O0Q0dnpsaVdpOUo0Y3ZPRERza05sMzJvM3o3Skc1R1RNZFRob0FzMitVckk9MTswMDsn\"",
            "id": "47417ba7-7de7-ea11-80cb-000d3ab3f0e3",
            "code": "UK",
            "displayName": "United Kingdom",
            "lastModifiedDateTime": "2020-08-26T09:22:26.503Z"
        }
    ]
}

*/

/*
Sätta dimension på kund

PATCH
https://erp.closely-official.com/sandbox/BC365Test/api/v1.0/companies(383f5c6d-29b3-44a3-adc2-1caa18fe9a62)/customers(3c3f8442-56dd-ea11-80ca-000d3ab3f0e3)/defaultDimensions(3c3f8442-56dd-ea11-80ca-000d3ab3f0e3,298bb0f3-76e7-ea11-80cb-000d3ab3f0e3)

{
    "dimensionValueId": "646b3299-7de7-ea11-80cb-000d3ab3f0e3"
}
*/
