<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\BusinessCentral\Model\Erp;

use Ecomero\BusinessCentral\Helper\Settings;
use Ecomero\BusinessCentral\Model\BusinessCentralAdapter;
use Ecomero\BusinessCentral\Model\BusinessCentralAPI;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpCustomerInterface;
use Ecomero\ErpCore\Model\Order\ResponseFactory as OrderResponseFactory;
use Magento\Framework\Stdlib\ArrayManager;

class BcCustomer extends BusinessCentralAdapter implements ErpCustomerInterface
{
    public function __construct(
        BusinessCentralAPI $api,
        Settings $settings,
        Capability $capability,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        OrderResponseFactory $orderResponseFactory,
        ArrayManager $arrayManager
    ) {
        parent::__construct(
            $api,
            $settings,
            $capability,
            $storeManager,
            $orderResponseFactory,
            $arrayManager
        );
    }

    public function getCompanyList(): array
    {
        $companies = [];
        $endpoint = $this->api->getEndPointForAPI('/api/v2.0')."contacts?\$filter=businessRelation eq 'Customer'";

        $rc = $this->api->sendRequest($endpoint);

        $records = $this->arrayManager->get('value', $rc);

        foreach ($records as $companyRecord) {
            $type = $this->arrayManager->get('type', $companyRecord);

            if ('Company' === $type) {
                $customer = [];

                // Create company
                $company['id'] = $this->arrayManager->get('id', $companyRecord);
                $company['number'] = $this->arrayManager->get('number', $companyRecord);
                $company['name'] = $this->arrayManager->get('companyName', $companyRecord);
                $company['alternativeName'] = $this->arrayManager->get('displayName', $companyRecord);
                $company['email'] = $this->arrayManager->get('email', $companyRecord);
                $company['web'] = $this->arrayManager->get('website', $companyRecord);
                $company['street'] = $company['deliveryStreet'] = $this->arrayManager->get('addressLine1', $companyRecord);
                $company['city'] = $company['deliveryCity'] = $this->arrayManager->get('city', $companyRecord);
                $company['countryId'] = $this->arrayManager->get('country', $companyRecord);
                $company['postalCode'] = $company['deliveryPostalCode'] = $this->arrayManager->get('postalCode', $companyRecord);
                $company['telephone'] = $this->arrayManager->get('phoneNumber', $companyRecord);
                $company['orgNo'] = $this->arrayManager->get('taxRegistrationNumber', $companyRecord);
                $company['vatNo'] = $this->arrayManager->get('taxRegistrationNumber', $companyRecord);
                $company['priceList'] = '';
                $company['customers'] = [];
                // Add contact persons to the company
                foreach ($records as $personRecord) {
                    $type = $this->arrayManager->get('type', $personRecord);

                    if ('Person' === $type && $company['number'] === $this->arrayManager->get('companyNumber', $personRecord)) {
                        $customer = [];

                        $names = $this->splitName($this->arrayManager->get('displayName', $personRecord));
                        $customer['id'] = $this->arrayManager->get('id', $personRecord);
                        $customer['number'] = $this->arrayManager->get('number', $personRecord);
                        $customer['firstName'] = $names['firstName'];
                        $customer['lastName'] = $names['lastName'];
                        $customer['street'] = $this->arrayManager->get('addressLine1', $personRecord);
                        $customer['city'] = $this->arrayManager->get('city', $personRecord);
                        $customer['postalCode'] = $this->arrayManager->get('postalCode', $personRecord);
                        $customer['email'] = $this->arrayManager->get('email', $personRecord);
                        $customer['companyId'] = $this->arrayManager->get('companyNumber', $personRecord);
                        $customer['telephoneSwitch'] = $this->arrayManager->get('phoneNumber', $personRecord);
                        $customer['telephoneDirect'] = $this->arrayManager->get('', $personRecord);
                        $customer['telephoneMobile'] = $this->arrayManager->get('mobilePhoneNumber', $personRecord);
                        $customer['countryId'] = $this->arrayManager->get('country', $personRecord);

                        $company['customers'][] = $customer;
                    }
                }

                $companies[] = $company;
            }
        }

        return $companies;
    }

    public function getCustomerCategories(): array
    {
        return [];
    }

    public function getCustomerDistricts(): array
    {
        return [];
    }

    public function getCustomerFromEmail(
        string $email
    ): array {
        $tmpResult = [];

        $rc = $this->api->callStandardApi('customers?$filter=email eq \''.$email.'\'');

        foreach ($rc as $item) {
            $newItem = [];
            $oldItem = $item;

            foreach ($oldItem as $key => $val) {
                if ('address' === $key) {
                    $newItem['street'] = $this->arrayManager->get('street', $val);
                    $newItem['city'] = $this->arrayManager->get('city', $val);
                    $newItem['countryCode'] = $this->arrayManager->get('countryLetterCode', $val);
                    $newItem['postalCode'] = $this->arrayManager->get('postalCode', $val);
                    $newItem['state'] = $this->arrayManager->get('state', $val);
                } elseif ('displayName' === $key) {
                    $newItem['name'] = $val;
                } elseif ('phoneNumber' === $key) {
                    $newItem['phone'] = $val;
                } else {
                    $newItem[$key] = $val;
                }
            }

            $tmpResult[] = $newItem;
        }

        return $tmpResult;
    }

    public function createCustomer(
        ?string $company,
        ?string $firstName,
        ?string $lastName,
        ?string $street,
        ?string $city,
        ?string $postCode,
        ?string $email,
        ?string $phone,
        ?string $country,
        ?string $region
    ): array {
        $post_data = json_encode(['displayName' => $firstName.' '.$lastName,
            'phoneNumber' => $phone ?? '',
            'email' => $email,
            'address' => [
                'street' => $street,
                'city' => $city,
                'state' => $region,
                'countryLetterCode' => $country ?? '',
                'postalCode' => $postCode ?? '',
            ],  ]);

        return $this->api->callStandardApi('customers', $post_data);
    }

    private function splitName(string $name)
    {
        $name = trim($name);
        if (false === strpos($name, ' ')) {
            return ['firstName' => $name, 'lastName' => ''];
        }

        $parts = explode(' ', $name);
        $lastname = array_pop($parts);
        $firstname = implode(' ', $parts);

        return ['firstName' => $firstname, 'lastName' => $lastname];
    }
}
