<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\BusinessCentral\Model;

use Ecomero\BusinessCentral\Helper\Settings;
use Laminas\Http\Client;
use Laminas\Http\Request;
use Magento\Framework\Filesystem\Driver\File;
use Magento\Framework\Stdlib\ArrayManager;

class BusinessCentralAPI
{
    protected $zendClient;
    protected $settings;
    protected $companyNameIds;
    protected $fileSystem;
    protected $arrayManager;
    protected $token;

    public function __construct(
        Client $zendClient,
        Settings $settings,
        File $fileSystem,
        ArrayManager $arrayManager
    ) {
        $this->zendClient = $zendClient;
        $this->settings = $settings;
        $this->companyNameIds = [];
        $this->fileSystem = $fileSystem;
        $this->arrayManager = $arrayManager;
    }

    public function getEndPointForAPI(string $api = '/api/v2.0'): string
    {
        return $this->getEndPoint().$api.'/companies('.$this->getCompanyId().')/';
    }

    public function getCompanies(): array
    {
        $endpoint = $this->getEndPoint().'/api/v2.0/companies';

        $rc = $this->sendRequest($endpoint);

        return $this->arrayManager->get('value', $rc);
    }

    public function sendRequest(
        string $endpoint,
        string $type = \Laminas\Http\Request::METHOD_GET,
        string $payLoad = null,
        bool $ignoreResponse = false
    ): array {
        $this->zendClient->reset();
        $this->zendClient->setOptions(['timeout' => 60]);
        $request = new Request();
        $request->setUri($this->urlEncode($endpoint));
        $request->setMethod($type);
        $this->setHeaders($request);

        if (null !== $payLoad) {
            $request->setContent($payLoad);
            $request->getHeaders()->addHeaders([
                'Content-Length' => strlen($payLoad),
                'Content-Type' => 'application/json',
            ]);
        }

        $this->zendClient->send($request);
        $response = $this->zendClient->getResponse();
        $json = [];
        if (false === $ignoreResponse || $response->getStatusCode() > 299) {
            $json = json_decode($response->getBody(), true);
        }

        if ($response->getStatusCode() > 299) {
            if ($json) {
                $errorMessage = $this->arrayManager->get('error/message', $json);
                if ($errorMessage) {
                    throw new \RuntimeException("ERROR: {$errorMessage}");
                }
            }

            throw new \RuntimeException('ERROR: Business Central replied with status code '.
                                        $response->getStatusCode().
                                        ' when calling '.
                                        $endpoint);
        }

        return $json;
    }

    public function callEcomeroApi(string $url, string $type = \Laminas\Http\Request::METHOD_GET, string $payload = null, bool $ignoreResponse = false): ?array
    {
        $endpoint = $this->getEndPointForAPI('/api/eComero/magento/v1.0').$url;
        $rc = $this->sendRequest($endpoint, $type, $payload, $ignoreResponse);

        return $this->arrayManager->get('value', $rc) ?? $rc;
    }

    public function callWarehouseApi(string $url, string $type = \Laminas\Http\Request::METHOD_GET, string $payload = null, bool $ignoreResponse = false): ?array
    {
        $endpoint = $this->getEndPointForAPI('/api/eComero/inventory/v1.0').$url;
        $rc = $this->sendRequest($endpoint, $type, $payload, $ignoreResponse);

        return $this->arrayManager->get('value', $rc) ?? $rc;
    }

    public function callStandardApi(string $url, string $type = \Laminas\Http\Request::METHOD_GET, string $payload = null, bool $ignoreResponse = false): ?array
    {
        $endpoint = $this->getEndPointForAPI().$url;
        $rc = $this->sendRequest($endpoint, $type, $payload, $ignoreResponse);

        return $this->arrayManager->get('value', $rc) ?? $rc;
    }

    public function callStandardApiFileUpload(
        string $url,
        string $fileUrl
    ): int {
        $endpoint = $this->getEndPointForAPI().$url;

        $this->zendClient->reset();
        $this->zendClient->setOptions(['timeout' => 60]);
        $request = new Request();
        $request->setUri($endpoint);
        $request->setMethod(\Laminas\Http\Request::METHOD_PATCH);

        $this->setHeaders($request);
        $request->getHeaders()->addHeaders([
            'Content-Type' => 'application/octet-stream',
        ]);

        try {
            $content = $this->fileSystem->fileGetContents($fileUrl);
        } catch (\Magento\Framework\Exception\FileSystemException $exception) {
            $content = false;
        }
        if ($content) {
            $request->setContent($content);

            $this->zendClient->send($request);
            $response = $this->zendClient->getResponse();

            return $response->getStatusCode();
        }

        return 500;
    }

    private function getEndPoint()
    {
        $endpoint = '';
        if ('cloud' == $this->settings->getInstallation()) {
            $endpoint = 'https://api.businesscentral.dynamics.com/v2.0/'.
                        $this->settings->getTenantId().
                        '/'.
                        $this->settings->getEnvironment();
        } else {
            $endpoint = $this->settings->getAPIEndPoint();
        }

        return $endpoint;
    }

    private function setHeaders(Request $request): void
    {
        if ('oauth2' === $this->settings->getAuthMethod()) {
            $token = $this->requestOAuth2Token();

            $request->getHeaders()->addHeaders([
                'User-Agent' => 'Magento',
                'Cache-Control' => 'no-cache',
                'Accept' => 'application/json',
                'Authorization' => ('Bearer '.$token),
                'Accept-Encoding' => 'gzip',
                'If-Match' => '*',
            ]);
        } else {
            $token = base64_encode($this->settings->getUserName().':'.$this->settings->getAPIKey());

            $request->getHeaders()->addHeaders([
                'User-Agent' => 'Magento',
                'Cache-Control' => 'no-cache',
                'Accept' => 'application/json',
                'Authorization' => ('Basic '.$token),
                'Accept-Encoding' => 'gzip',
                'If-Match' => '*',
            ]);
        }
    }

    private function requestOAuth2Token(): string
    {
        if (null !== $this->token) {
            return $this->token;
        }

        $tenantId = $this->settings->getTenantId();
        $endpoint = 'https://login.windows.net/'.$tenantId.'/oauth2/token';

        $payload = 'resource=https%3A%2F%2Fapi.businesscentral.dynamics.com'.
                    '&client_id='.$this->settings->getClientId().
                    '&client_secret='.$this->settings->getSecret().
                    '&grant_type='.'client_credentials';

        $this->zendClient->reset();
        $request = new Request();
        $request->setUri($this->urlEncode($endpoint));
        $request->setMethod(\Laminas\Http\Request::METHOD_POST);

        $request->setContent($payload);
        $request->getHeaders()->addHeaders([
            'Content-Length' => strlen($payload),
            'Content-Type' => 'application/x-www-form-urlencoded',
            'User-Agent' => 'Magento',
            'Cache-Control' => 'no-cache',
            'Accept' => 'application/json',
            'Accept-Encoding' => 'gzip',
        ]);

        $this->zendClient->send($request);
        $response = $this->zendClient->getResponse();
        $json = [];
        if (200 == $response->getStatusCode()) {
            $json = json_decode($response->getBody(), true);

            return $this->arrayManager->get('access_token', $json);
        }

        return 'invalid-token';
    }

    private function getCompanyId(): string
    {
        $companyName = $this->settings->getAPICompanyName();
        if (array_key_exists($companyName, $this->companyNameIds)) {
            return $this->companyNameIds[$companyName];
        }

        $companies = $this->getCompanies();
        $companyNames = '';
        if (null != $companies) {
            foreach ($companies as $companyEntry) {
                $id = $this->arrayManager->get('id', $companyEntry);
                $name = $this->arrayManager->get('name', $companyEntry);
                $this->companyNameIds[$name] = $id;
                $companyNames = $companyNames.$name.', ';
                if ($name === $companyName) {
                    return $id;
                }
            }
        }
        $companyNames = rtrim($companyNames, ', ');

        throw new \RuntimeException('ERROR: Company with the name '.$companyName.
                                    ' was not found in Business Central. The following companies are available '.
                                    $companyNames);
    }

    private function urlEncode(string $url): string
    {
        $url = str_replace(' ', '%20', $url);
        $url = str_replace('å', '%C3%A5', $url);
        $url = str_replace('ä', '%C3%A4', $url);
        $url = str_replace('ö', '%C3%B6', $url);
        $url = str_replace('Å', '%C3%85', $url);
        $url = str_replace('Ä', '%C3%84', $url);
        $url = str_replace('Ö', '%C3%96', $url);

        return str_replace("'", '%27', $url);
    }
}
