<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\BusinessCentral\Model\Erp;

use Ecomero\BusinessCentral\Helper\ProductHelper;
use Ecomero\BusinessCentral\Helper\Settings;
use Ecomero\BusinessCentral\Model\BusinessCentralAdapter;
use Ecomero\BusinessCentral\Model\BusinessCentralAPI;
use Ecomero\ErpCore\Model\Attribute;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpCatalogInterface;
use Ecomero\ErpCore\Model\Order\ResponseFactory as OrderResponseFactory;
use Ecomero\ErpCore\Model\ResourceModel\Item\Collection;
use Magento\Framework\Stdlib\ArrayManager;

class BcCatalog extends BusinessCentralAdapter implements ErpCatalogInterface
{
    private $itemCache = [];
    private $productHelper;

    public function __construct(
        BusinessCentralAPI $api,
        Settings $settings,
        Capability $capability,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        OrderResponseFactory $orderResponseFactory,
        ArrayManager $arrayManager,
        ProductHelper $productHelper
    ) {
        parent::__construct(
            $api,
            $settings,
            $capability,
            $storeManager,
            $orderResponseFactory,
            $arrayManager
        );
        $this->productHelper = $productHelper;
    }

    public function getItemList(
        string $category,
        bool $force
    ): array {
        if (array_key_exists($category, $this->itemCache)) {
            return $this->itemCache[$category];
        }

        $url = 'items?$filter=category eq \''.$category.'\'';
        $rc = $this->api->callEcomeroApi($url);
        $this->itemCache[$category] = [];

        foreach ($rc as $item) {
            $item['commonName'] = $item['description'];
            $this->itemCache[$category][] = $item;
        }

        return $this->itemCache[$category];
    }

    public function getItemPriceLists(): array
    {
        $priceLists = [];
        $discountGroup = [];
        $discountPrice = [];

        $url = 'itemPrices?$filter=sourceType eq \'Customer\'';
        $rc = $this->api->callEcomeroApi($url);

        foreach ($rc as $item) {
            $customerId = $item['assignedTo'];

            // Init price list if new
            if (null === $this->arrayManager->get($customerId, $priceLists)) {
                $companyName = $this->productHelper->getCompanyName($customerId);
                if (null === $companyName) {
                    continue;
                }
                $priceLists[$customerId]['id'] = $customerId.'_';
                $priceLists[$customerId]['name'] = $companyName;
                $priceLists[$customerId]['prices'] = [];
            }

            if ('Item Discount Group' === $item['type']) {
                $discountGroup[$customerId][] = [
                    'sku' => '',
                    'price' => 0,
                    'currency' => '' !== $item['currency'] ?: 'SEK',
                    'from_date' => null,
                    'to_date' => null,
                    'article_type' => $item['assetNo'],
                    'thresholds' => [],
                    'discount' => $item['lineDiscount'],
                    'price_type' => 'R',
                ];
            } elseif ('Item' === $item['type']) {
                $discountPrice[$customerId][] = [
                    'sku' => $item['sku'],
                    'price' => $item['priceCurrency'],
                    'currency' => '' !== $item['currency'] ?: 'SEK',
                    'from_date' => null,
                    'to_date' => null,
                    'article_type' => '',
                    'thresholds' => [],
                    'discount' => 0,
                    'price_type' => '',
                ];
            }
        }

        // Prioritize price before group
        foreach ($discountGroup as $customerId => $discount) {
            $priceLists[$customerId]['prices'] = $discount;
        }

        foreach ($discountPrice as $customerId => $discount) {
            $priceLists[$customerId]['prices'] = array_merge($priceLists[$customerId]['prices'], $discount);
        }

        return $priceLists;
    }

    public function setItemImportCompleted(): void
    {
    }

    public function getSalesPriceLists(
    ): array {
        return $this->api->callEcomeroApi('salesPriceList');
    }

    public function getItemPrices(
        string $category
    ): array {
        $result = [];

        $priceListCode = $this->settings->getDefaultPriceList($this->getWebsite());

        $url = 'itemPrices?$filter=category eq \''.$category.'\'&$select=id,sku,currency,priceCurrency,priceListCode';
        $rc = $this->api->callEcomeroApi($url);

        foreach ($rc as $item) {
            if ($item['priceListCode'] === $priceListCode) {
                if ('' === $item['currency']) {
                    $item['currency'] = $this->settings->getLCY();
                }
                $result[] = $item;
            }
        }

        return $result;
    }

    public function getItemAttributes(
        string $category
    ): array {
        $result = [];

        $url = 'itemAttributes?$filter=category eq \''.$category.'\'&$select=id,sku,type,name,value';
        $rc = $this->api->callEcomeroApi($url);

        foreach ($rc as $item) {
            $result[] = $item;
        }

        // Special case, we treat discountGroup as an attribute to avoid code changes in the core module
        $items = $this->getItemList($category, false);
        foreach ($items as $item) {
            $result[] = $this->mapAttribute('discountGroup', 'article_type', $item);
        }

        return $result;
    }

    public function getNonInventoryItems(): array
    {
        $rc = [];
        $url = 'items?$filter=itemType eq \'Non-Inventory\'';
        $nonInvItems = $this->api->callEcomeroApi($url);

        foreach ($nonInvItems as $item) {
            $rc[] = [
                'id' => $item['id'],
                'type' => 'non-inv',
                'priceIncludesVAT' => $item['priceIncludesVAT'] ? 'true' : 'false',
                'description' => $item['description'].' (Non-Inv Item)',
                'taxRate' => $item['taxRate'],
            ];
        }

        return $rc;
    }

    public function addPicture(
        string $itemId,
        string $imageUrl
    ): int {
        return $this->api->callStandardApiFileUpload('items('.$itemId.')/picture('.$itemId.')/content', $imageUrl);
    }

    public function createCommonName(
        Collection $itemCollection
    ): bool {
        $commonNameAttributes = [];
        if ($this->settings->getCommonNameAttributes($this->getWebsite())) {
            $commonNameAttributes = explode(',', $this->settings->getCommonNameAttributes($this->getWebsite()));
        }

        $configurableAttributes = '';
        if ($this->settings->getConfigurableAttributes($this->getWebsite())) {
            $configurableAttributes = explode(',', $this->settings->getConfigurableAttributes($this->getWebsite()));
        }

        foreach ($itemCollection->getItems() as $item) {
            $item->commonName = '';
            foreach ($commonNameAttributes as $attribute) {
                $attribute = strtolower($attribute);
                if (key_exists($attribute, $item->attributes)) {
                    if ('' === $item->commonName) {
                        $item->commonName = $item->attributes[$attribute]->value;
                    } else {
                        $item->commonName = $item->commonName.'-'.$item->attributes[$attribute]->value;
                    }
                }
            }

            if (null !== $configurableAttributes && '' !== $configurableAttributes) {
                // Clear usedInConfiguration since it may have been set as deault to true by erp core
                foreach ($item->attributes as $key => $value) {
                    $item->attributes[$key]->usedInConfiguration = Attribute::NO;
                }

                // If attribute is a configurable product attribute, then set flag
                foreach ($configurableAttributes as $attribute) {
                    $attribute = strtolower($attribute);
                    if (key_exists($attribute, $item->attributes)) {
                        $item->attributes[$attribute]->usedInConfiguration = Attribute::YES;
                    }
                }
            }
        }

        return true;
    }

    private function mapAttribute(string $name, string $attributeCode, array $item, string $type = 'Text'): ?array
    {
        $attribute = null;
        if (array_key_exists($name, $item)) {
            $attribute = [];
            $attribute['sku'] = $item['sku'];
            $attribute['name'] = $attributeCode;
            $attribute['type'] = $type;
            $attribute['value'] = $item[$name];
        }

        return $attribute;
    }
}
