<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\Pyramid\Model\Erp;

use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpCatalogInterface;
use Ecomero\ErpCore\Model\Order\ResponseFactory as OrderResponseFactory;
use Ecomero\ErpCore\Model\ResourceModel\Item\Collection;
use Ecomero\Pyramid\Model\PyramidAdapter;
use Ecomero\Pyramid\Model\PyramidAPI;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\Store\Model\StoreManagerInterface;

class PyramidCatalog extends PyramidAdapter implements ErpCatalogInterface
{
    protected $itemListCache = [];

    public function __construct(
        PyramidAPI $pyramidAPI,
        Settings $settings,
        Capability $capability,
        ArrayManager $arrayManager,
        StoreManagerInterface $storeManager,
        OrderResponseFactory $orderResponseFactory
    ) {
        parent::__construct(
            $pyramidAPI,
            $settings,
            $capability,
            $arrayManager,
            $storeManager,
            $orderResponseFactory
        );
    }

    public function setItemImportCompleted(): void
    {
    }

    public function getItemPrices(string $category): array
    {
        return [];
    }

    public function getItemPriceLists(): array
    {
        $rc = [];
        $response = $this->pyramidAPI->get('command=pricelist');
        $combinations = $this->arrayManager->get('priceAgreements', $response);
        foreach ($combinations as $combination) {
            $pId = $combination['price_list_id'];
            $aId = $combination['agreement_id'];
            $priceList = $this->getPriceList($response, $pId);
            $agreementList = $this->getPriceList($response, $aId);

            if ($priceList || $agreementList) {
                // Found valid price list

                $pName = $this->arrayManager->get('name', $priceList) ?? '';
                $aName = $this->arrayManager->get('name', $agreementList) ?? '';

                $name = $this->getPriceListName($pName, $aName);
                $pPrices = $this->arrayManager->get('prices', $priceList) ?? [];
                $aPrices = $this->arrayManager->get('prices', $agreementList) ?? [];
                $paPrices = array_merge($pPrices, $aPrices);

                $combinedPrices = array_filter($paPrices, function ($price) {
                    return $price['sku'] || (
                        $price['article_type']
                                && self::DISCOUNT_BASE_PRICE == $price['discount_base']
                                && self::PRICE_TYPE_DISCOUNT == $price['price_type']
                    );
                });

                $priceList = [
                    'id' => $pId.'_'.$aId,
                    'name' => $name,
                    'prices' => $combinedPrices,
                ];

                $rc[] = $priceList;
            }
        }

        return $rc;
    }

    public function getItemAttributes(string $category): array
    {
        $result = [];
        foreach ($this->itemListCache as $item) {
            $result[] = $this->mapAttribute('category', 'article_type', $item, 'Option');
            $result[] = $this->mapAttribute('leadTime', 'lead_time', $item);
            $result[] = $this->mapAttribute('hsCode', 'hs_code', $item);
            $result[] = $this->mapAttribute('height', 'height', $item);
            $result[] = $this->mapAttribute('length', 'length', $item);
            $result[] = $this->mapAttribute('width', 'width', $item);
            $result[] = $this->mapAttribute('grossWeight', 'gross_weight', $item);
            $result[] = $this->mapAttribute('packageSize', 'package_size', $item);

            $formattedDescriptions = $this->arrayManager->get('formattedDescription', $item);
            foreach ($formattedDescriptions as $formattedDescription) {
                $attribute = [];
                $attribute['sku'] = $item['sku'];
                $attribute['locale'] = $formattedDescription['language'];
                $attribute['name'] = 'description';
                $attribute['type'] = 'Text';
                $attribute['value'] = strip_tags($formattedDescription['description']);
                $result[] = $attribute;
            }
        }

        return array_filter($result);
    }

    public function getNonInventoryItems(): array
    {
        return [];
    }

    public function addPicture(string $itemId, string $imageUrl): int
    {
        return 0;
    }

    public function createCommonName(Collection $itemCollection): bool
    {
        return false;
    }

    public function getItemList(
        string $category,
        bool $force
    ): array {
        $pageSize = 0; // Disable paging, not supported on Zen V12 (only from V14)
        $row = 0;

        do {
            $endRow = $row + $pageSize;
            $this->logger->info("  Importing product {$row} to {$endRow}");
            $queryParam = "command=article&filter={$category}&pageRow={$row}&pageSize={$pageSize}&includeDescriptions=true";
            $pageData = $this->pyramidAPI->get($queryParam);
            $this->itemListCache = array_merge($this->itemListCache, $pageData);
            $row += count($pageData);
        } while (count($pageData) === $pageSize && 0 !== $pageSize);

        foreach ($this->itemListCache as &$item) {
            $item['commonName'] = $this->getCommonName($item);
        }

        return $this->itemListCache;
    }

    private function mapAttribute(string $name, string $attributeCode, array $item, string $type = 'Text'): ?array
    {
        $attribute = null;
        if (array_key_exists($name, $item)) {
            $attribute = [];
            $attribute['sku'] = $item['sku'];
            $attribute['name'] = $attributeCode;
            $attribute['type'] = $type;
            $attribute['value'] = $item[$name];
        }

        return $attribute;
    }

    private function getPriceListName(string $pName, string $aName): string
    {
        $name = '';

        if ($pName && $aName) {
            $name = "{$pName}, {$aName}";
        } elseif ($pName) {
            $name = $pName;
        } else {
            $name = $aName;
        }

        return $name;
    }

    private function getPriceList(array $response, string $priceListId): array
    {
        $priceLists = $this->arrayManager->get('priceLists', $response);
        foreach ($priceLists as $priceList) {
            if ($priceList['id'] === $priceListId) {
                return $priceList;
            }
        }

        return [];
    }

    private function getCommonName(array $item): ?string
    {
        return $this->arrayManager->get('sku', $item);
    }

    public function getErpExtVersion() : string
    {
        return '---';
    }
}
