<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\VismaSPCS\Model;

use Ecomero\VismaSPCS\Helper\Settings;
use Laminas\Http\Client;
use Laminas\Http\Request;
use Magento\Framework\Filesystem\Driver\File;

class VismaAPI
{
    protected $_zendClient;
    protected $settings;
    protected $companyNameIds;
    protected $fileSystem;

    public function __construct(
        Client $zendClient,
        Settings $settings,
        File $fileSystem
    ) {
        $this->_zendClient = $zendClient;
        $this->settings = $settings;
        $this->companyNameIds = [];
        $this->fileSystem = $fileSystem;
    }

    public function getItemList(string $category, string $modifiedAfter): array
    {
        $url = 'ArticlesSQL?company='.$this->settings->getCompany().'&filter='.$category.'&modifiedAfter='.$modifiedAfter;
        $endpoint = $this->getEndPoint().$url;

        return $this->sendRequest($endpoint);
    }

    public function createDocument(
        string $docType,
        array $orderData
    ): array {
        $url = $docType;
        $endpoint = $this->getEndPoint().$url;

        $orderData['company'] = $this->settings->getCompany();
        $post_data = json_encode($orderData);

        return $this->sendRequest($endpoint, \Laminas\Http\Request::METHOD_POST, $post_data);
    }

    public function getDocumentFromExternalId(
        string $docType,
        string $magentoOrderId
    ): array {
        $url = $docType.'?company='.$this->settings->getCompany().'&externalDocumentNumber='.$magentoOrderId;
        $endpoint = $this->getEndPoint().$url;

        return $this->sendRequest($endpoint);
    }

    public function getSalesPersons(): array
    {
        $url = 'sellers?company='.$this->settings->getCompany();
        $endpoint = $this->getEndPoint().$url;

        return $this->sendRequest($endpoint);
    }

    public function getProfitCenters(): array
    {
        $url = 'profitcenters?company='.$this->settings->getCompany();
        $endpoint = $this->getEndPoint().$url;

        return $this->sendRequest($endpoint);
    }

    public function getTermsOfDeliveries(): array
    {
        $url = 'TermsOfDeliveries?company='.$this->settings->getCompany();
        $endpoint = $this->getEndPoint().$url;

        return $this->sendRequest($endpoint);
    }

    public function getCustomerCategories(): array
    {
        $url = 'CustomerCategories?company='.$this->settings->getCompany();
        $endpoint = $this->getEndPoint().$url;

        return $this->sendRequest($endpoint);
    }

    public function getCustomerDistricts(): array
    {
        $url = 'CustomerDistricts?company='.$this->settings->getCompany();
        $endpoint = $this->getEndPoint().$url;

        return $this->sendRequest($endpoint);
    }

    public function getWayOfDeliveries(): array
    {
        $url = 'WayOfDeliveries?company='.$this->settings->getCompany();
        $endpoint = $this->getEndPoint().$url;

        return $this->sendRequest($endpoint);
    }

    public function getCustomerFromEmail(string $email): array
    {
        $url = 'customer?company='.$this->settings->getCompany().'&email='.$email;
        $endpoint = $this->getEndPoint().$url;

        return $this->sendRequest($endpoint);
    }

    public function createCustomer(
        string $company,
        string $firstName,
        string $lastName,
        string $street,
        string $city,
        string $postCode,
        string $email,
        ?string $phone,
        ?string $country,
        ?string $region
    ): array {
        $endpoint = $this->getEndPoint().'customer';

        $post_data = json_encode(['company' => $this->settings->getCompany(),
            'customer_company' => $company,
            'name' => $firstName.' '.$lastName,
            'phone' => $phone ?? '',
            'email' => $email,
            'street' => $street,
            'city' => $city,
            'state' => $region ?? '',
            'countryCode' => $country ?? '',
            'postalCode' => $postCode,
        ]);

        return $this->sendRequest($endpoint, \Laminas\Http\Request::METHOD_POST, $post_data);
    }

    public function getShippingAgents(): array
    {
        $url = 'dispatchers?company='.$this->settings->getCompany();
        $endpoint = $this->getEndPoint().$url;

        return $this->sendRequest($endpoint);
    }

    public function getItemListNonInventory(): array
    {
        $rc = [];

        $url = 'ArticlesSQLNonInv?company='.$this->settings->getCompany();
        $endpoint = $this->getEndPoint().$url;
        $noneInvItemList = $this->sendRequest($endpoint);

        foreach ($noneInvItemList as $item) {
            $rc[] = (object) [
                'id' => $item->id,
                'type' => 'non-inv',
                'priceIncludesVAT' => $item->priceIncludesVAT ? 'true' : 'false',
                'description' => $item->description.' (Non-Inv Item)',
                'taxRate' => $item->taxRate,
            ];
        }

        return $rc;
    }

    /*
    public function getAggregatedFreightOptions(): array
    {
        $result = [];
        $tmpItem['id'] = 'VISMA_FREIGHT';
        $tmpItem['type'] = 'charge';
        $tmpItem['priceIncludesVAT'] = 'false';
        $tmpItem['description'] = 'SPCS Frakt';
        $tmpItem['taxRate'] = 0;
        $result[] = $tmpItem;

        return $result;
    }
*/

    private function getEndPoint(): string
    {
        return $this->settings->getAPIEndPoint().'/api/';
    }

    private function setHeaders(Request $request): void
    {
        $request->getHeaders()->addHeaders([
            'API-Key' => $this->settings->getAPIKey(),
            'User-Agent' => 'Magento',
            'Cache-Control' => 'no-cache',
            'Accept' => 'application/json',
            'Accept-Encoding' => 'gzip',
            'If-Match' => '*',
        ]);
    }

    private function sendRequest(
        string $endpoint,
        string $type = \Laminas\Http\Request::METHOD_GET,
        string $payLoad = null,
        bool $ignoreResponse = false
    ): array {
        $this->_zendClient->reset();
        $request = new Request();
        $request->setUri($this->urlEncode($endpoint));
        $request->setMethod($type);
        $this->setHeaders($request);

        if (null !== $payLoad) {
            $request->setContent($payLoad);
            $request->getHeaders()->addHeaders([
                'Content-Length' => strlen($payLoad),
                'Content-Type' => 'application/json',
            ]);
        }

        $this->_zendClient->send($request);
        $response = $this->_zendClient->getResponse();
        $json = '';
        if (false === $ignoreResponse || $response->getStatusCode() > 299) {
            $json = json_decode($response->getBody());
        }

        if ($response->getStatusCode() > 299) {
            if ($json) {
                if (is_array($json) && 1 == count($json)) {
                    if (property_exists($json[0], 'errorCode')
                        && property_exists($json[0], 'errorMessage')) {
                        throw new \RuntimeException('ERROR: '.$json[0]->errorMessage.' ('.$json[0]->errorCode.')');
                    }
                }
            }

            throw new \RuntimeException('ERROR: Visma SPCS replied with status code '.
                                        $response->getStatusCode().
                                        ' when calling '.
                                        $endpoint);
        }

        return (array) $json;
    }

    private function urlEncode(string $url): string
    {
        $url = str_replace(' ', '%20', $url);
        $url = str_replace("'", '%27', $url);
        $url = str_replace('å', '%C3%A5', $url);
        $url = str_replace('ä', '%C3%A4', $url);
        $url = str_replace('ö', '%C3%B6', $url);
        $url = str_replace('Å', '%C3%85', $url);
        $url = str_replace('Ä', '%C3%84', $url);

        return str_replace('Ö', '%C3%96', $url);
    }
}
