<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\VismaSPCS\Model;

use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpAdapter;
use Ecomero\ErpCore\Model\Erp\ErpCatalogInterface;
use Ecomero\ErpCore\Model\Erp\ErpCustomerInterface;
use Ecomero\ErpCore\Model\Erp\ErpOrderInterface;
use Ecomero\ErpCore\Model\Order\Response as OrderResponse;
use Ecomero\ErpCore\Model\Order\ResponseFactory as OrderResponseFactory;
use Ecomero\ErpCore\Model\ResourceModel\Item\Collection;
use Ecomero\VismaSPCS\Helper\FieldMapper;
use Ecomero\VismaSPCS\Helper\Settings;
use Magento\Catalog\Model\ResourceModel\Product;
use Magento\Store\Model\StoreManagerInterface;

class VismaAdapter extends ErpAdapter implements ErpOrderInterface, ErpCatalogInterface, ErpCustomerInterface
{
    public const ERP_NAME = 'VismaSPCS';
    public const MAX_COMMENT_LEN = 60;
    protected $vismaSPCS;
    protected $itemListCache;
    protected $settings;
    protected $currentOrder;
    protected $capability;
    protected $fieldMapper;
    protected $productRepository;
    protected $orderResponseFactory;

    public function __construct(
        VismaAPI $vismaSPCS,
        Settings $settings,
        Capability $capability,
        FieldMapper $fieldMapper,
        Product $productRepository,
        StoreManagerInterface $storeManager,
        OrderResponseFactory $orderResponseFactory
    ) {
        parent::__construct($storeManager);

        $this->vismaSPCS = $vismaSPCS;
        $this->settings = $settings;
        $this->capability = $capability;
        $this->fieldMapper = $fieldMapper;
        $this->productRepository = $productRepository;
        $this->orderResponseFactory = $orderResponseFactory;

        $this->capability->setName(self::ERP_NAME);
        $this->capability->setCapability(Capability::PRODUCT_IMPORT, true);
        $this->capability->setCapability(Capability::PRODUCT_PICTURE_EXPORT, false);
        $this->capability->setCapability(Capability::ORDER_EXPORT, true);
        $this->capability->setCapability(Capability::ORDER_IMPORT, false);
        $this->capability->setCapability(Capability::CREDIT_MEMO_EXPORT, false);
        $this->capability->setCapability(Capability::INVOICE_EXPORT, false);
        $this->capability->setCapability(Capability::INVOICE_IMPORT, false);
        $this->capability->setCapability(Capability::SHIPMENT_IMPORT, false);
        $this->capability->setCapability(Capability::RETURN_IMPORT, false);
    }

    public function getCompanyList(): array
    {
        return [];
    }

    public function addPicture(string $itemId, string $imageUrl): int
    {
        return 0;
    }

    public function createCommonName(Collection $itemCollection): bool
    {
        return false;
    }

    public function postOrder(string $salesOrderId): void
    {
    }

    public function isErpEnabled(): string
    {
        if ($this->settings->isEnabled($this->getWebsite())) {
            return self::ERP_NAME;
        }

        return '';
    }

    public function getItemList(
        string $category,
        bool $force
    ): array {
        $result = [];

        $modifiedDateFilter = $this->settings->getArticleModifiedDate($this->getWebsite());
        if ('' === $modifiedDateFilter || $force) {
            $modifiedDateFilter = '1980-01-01';
        }
        $this->itemListCache = $this->vismaSPCS->getItemList($category, $modifiedDateFilter);

        $webshopItems = [];
        foreach ($this->itemListCache as $item) {
            if ($this->isProductForImport($item)) {
                if ('' !== $item->remark1) {
                    $item->description = $item->remark1;
                }
                $item->commonName = $this->getCommonName($item);
                $itemArray = (array) $item;
                $itemArray['isEnabledInErp'] = $item->isWebshopArticle;
                $result[] = (object) $itemArray;
                $webshopItems[] = (object) $itemArray;
            }
        }
        $this->itemListCache = $webshopItems;

        return $result;
    }

    public function getItemPriceLists(): array
    {
        return [];
    }

    public function setItemImportCompleted(): void
    {
        $this->settings->setArticleModifiedDate($this->getWebsite(), date('Y-m-d'));
    }

    public function getItemPrices(string $category): array
    {
        return [];
    }

    public function getItemAttributes(string $category): array
    {
        $result = [];
        $rc = [];
        $template = $this->settings->getArticleTemplate($this->getWebSite());
        $attribMappingTbl = $this->settings->getAttributeMapping($this->getWebSite());

        foreach ($this->itemListCache as $item) {
            $matches = [];
            $matches = \Ecomero\VismaSPCS\Helper\StringUtility::getPartsFromString($template, $item->sku);

            foreach ($matches as $group => $part) {
                $attribCode = $this->getMapping($group, $part, $attribMappingTbl);
                if (null == $attribCode) {
                    throw new \RuntimeException('ERROR: No article mapping for group '.$group);
                }
                if ('erp_common_name' !== $attribCode) {
                    $rc[] = (object) [
                        'id' => $item->sku,
                        'sku' => $item->sku,
                        'type' => 'Option',
                        'value' => $part,
                        'name' => $attribCode,
                    ];
                }
            }
        }

        foreach ($rc as $item) {
            $result[] = $item;
        }

        return $result;
    }

    public function createDocument(
        string $docType,
        string $customerId,
        ?string $companyId,
        string $currency,
        \Magento\Sales\Model\Order\Address $billingAddress,
        \Magento\Sales\Model\Order\Address $shippingAddress,
        ?string $shippingAgent,
        ?string $shippingService,
        ?string $shippingPickupLocationId,
        string $magentoOrderId,
        string $salesPerson,
        ?string $profitCenter,
        ?string $termsOfDelivery,
        ?string $wayOfDelivery,
        ?string $customerCategory,
        ?string $customerDistrict
    ): OrderResponse {
        $this->fieldMapper->init($magentoOrderId, $billingAddress, $shippingAddress, $this->getWebsite());

        $this->currentOrder['customerNumber'] = $customerId;
        $this->currentOrder['currencyCode'] = $currency;
        $this->currentOrder['salesperson'] = $salesPerson;
        $this->currentOrder['profitCenter'] = $profitCenter;
        $this->currentOrder['termsOfDelivery'] = $termsOfDelivery;
        $this->currentOrder['wayOfDelivery'] = $wayOfDelivery;
        $this->currentOrder['customerCategory'] = $customerCategory;
        $this->currentOrder['customerDistrict'] = $customerDistrict;

        $this->currentOrder['vatNumber'] = $this->fieldMapper->getValueForField('VAT');

        $this->currentOrder['name'] = $this->fieldMapper->getValueForField('BILLING_NAME');
        $this->currentOrder['reference'] = $this->fieldMapper->getValueForField('BILLING_REFERENCE');
        $this->currentOrder['telephone'] = $this->fieldMapper->getValueForField('BILLING_PHONE');
        $this->currentOrder['telephone2'] = $this->fieldMapper->getValueForField('BILLING_PHONE2');
        $this->currentOrder['telephone3'] = $this->fieldMapper->getValueForField('BILLING_PHONE3');
        $this->currentOrder['mobile'] = $this->fieldMapper->getValueForField('BILLING_MOBILE');
        $this->currentOrder['fax'] = $this->fieldMapper->getValueForField('BILLING_FAX');
        $this->currentOrder['street'] = $this->fieldMapper->getValueForField('BILLING_STREET1');
        $this->currentOrder['street2'] = $this->fieldMapper->getValueForField('BILLING_STREET2');
        $this->currentOrder['city'] = $this->fieldMapper->getValueForField('BILLING_CITY');
        $this->currentOrder['countryLetterCode'] = $this->fieldMapper->getValueForField('BILLING_COUNTRY_CODE');
        $this->currentOrder['postalCode'] = $this->fieldMapper->getValueForField('BILLING_POSTAL_CODE');
        $this->currentOrder['GLN'] = $this->fieldMapper->getValueForField('BILLING_GLN');
        $this->currentOrder['visitingAddress'] = $this->fieldMapper->getValueForField('BILLING_VISIT_ADDRESS');

        $this->currentOrder['deliveryName'] = $this->fieldMapper->getValueForField('DELIVERY_NAME');
        $this->currentOrder['deliveryTelephone'] = $this->fieldMapper->getValueForField('DELIVERY_PHONE');
        $this->currentOrder['deliveryTelephone2'] = $this->fieldMapper->getValueForField('DELIVERY_PHONE2');
        $this->currentOrder['deliveryTelephone3'] = $this->fieldMapper->getValueForField('DELIVERY_PHONE3');
        $this->currentOrder['deliveryFax'] = $this->fieldMapper->getValueForField('DELIVERY_FAX');
        $this->currentOrder['deliveryStreet'] = $this->fieldMapper->getValueForField('DELIVERY_STREET1');
        $this->currentOrder['deliveryStreet2'] = $this->fieldMapper->getValueForField('DELIVERY_STREET2');
        $this->currentOrder['deliveryCity'] = $this->fieldMapper->getValueForField('DELIVERY_CITY');
        $this->currentOrder['deliveryCountryLetterCode'] = $this->fieldMapper->getValueForField('DELIVERY_COUNTRY_CODE');
        $this->currentOrder['deliveryPostalCode'] = $this->fieldMapper->getValueForField('DELIVERY_POSTAL_CODE');
        $this->currentOrder['deliveryGLN'] = $this->fieldMapper->getValueForField('DELIVERY_GLN');
        $this->currentOrder['deliveryVisitingAddress'] = $this->fieldMapper->getValueForField('DELIVERY_VISIT_ADDRESS');

        $this->currentOrder['shippingAgentCode'] = $shippingAgent;
        $this->currentOrder['shippingAgentServiceCode'] = $shippingService;
        $this->currentOrder['shippingAdvice'] = '';
        $this->currentOrder['shippingAmount'] = 0;
        $this->currentOrder['externalDocument'] = $magentoOrderId;
        $this->currentOrder['orderLines'] = [];
        $this->currentOrder['number'] = '';
        $this->currentOrder['id'] = '';

        return $this->orderResponseFactory->create([
            'response' => $this->currentOrder,
        ]);
    }

    public function addDocumentItem(
        string $docType,
        string $orderId,
        string $erpItemId,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        float $discountAmount,
        float $discountPercent,
        ?string $comment
    ): void {
        $discountIsAmount = false;
        $discount = 0;
        if ($discountAmount > 0) {
            $discountIsAmount = true;
            $discount = $discountAmount;
        } else {
            $discountIsAmount = false;
            $discount = $discountPercent;
        }

        if (null !== $comment) {
            $finalComment = '';
            $commentLines = explode("\n", $comment);
            foreach ($commentLines as $line) {
                while (mb_strlen($line) > self::MAX_COMMENT_LEN) {
                    $finalComment .= mb_substr($line, 0, self::MAX_COMMENT_LEN)."\n";
                    $line = mb_substr($line, self::MAX_COMMENT_LEN, mb_strlen($line));
                }
                $finalComment .= $line."\n";
            }
            $comment = trim($finalComment);
        }

        $this->currentOrder['orderLines'][] = [
            'articleNumber' => $erpItemId,
            'qty' => $qtyOrdered,
            'price' => $price,
            'discount' => $discount,
            'discountIsAmount' => $discountIsAmount,
            'amount' => $price * $qtyOrdered,
            'vatCode' => $vatCode,
            'comment' => $comment ?? '',
        ];
    }

    public function updateDocument(
        string $docType,
        string $erpId,
        string $magentoOrderId
    ): OrderResponse {
        $this->currentOrder['externalDocument'] = $magentoOrderId;

        $response = $this->vismaSPCS->createDocument($docType, $this->currentOrder);

        return $this->orderResponseFactory->create([
            'response' => $response,
        ]);
    }

    public function getDocumentFromExternalId(
        string $docType,
        string $magentoOrderId
    ): object {
        $tmpResult = [];

        if (0 === strpos($magentoOrderId, 'PENDING(')) {
            $magentoOrderId = str_replace('PENDING(', '', $magentoOrderId);
            $magentoOrderId = rtrim($magentoOrderId, ')');
        }

        $rc = $this->vismaSPCS->getDocumentFromExternalId($docType, $magentoOrderId);
        if (1 == count($rc)) {
            $tmpResult = $rc[0];

            return (object) $tmpResult;
        }

        return (object) [];
    }

    public function getCustomerFromEmail(
        string $email
    ): array {
        return $this->vismaSPCS->getCustomerFromEmail($email);
    }

    public function createCustomer(
        ?string $company,
        ?string $firstName,
        ?string $lastName,
        ?string $street,
        ?string $city,
        ?string $postCode,
        ?string $email,
        ?string $phone,
        ?string $country,
        ?string $region
    ): array {
        return $this->vismaSPCS->createCustomer(
            $company ?? '',
            $firstName,
            $lastName,
            $street,
            $city,
            $postCode,
            $email,
            $phone,
            $country,
            $region
        );
    }

    public function getSalesPersons(): array
    {
        return $this->vismaSPCS->getSalesPersons();
    }

    public function getProfitCenters(): array
    {
        return $this->vismaSPCS->getProfitCenters();
    }

    public function getTermsOfDeliveries(): array
    {
        return $this->vismaSPCS->getTermsOfDeliveries();
    }

    public function getWayOfDeliveries(): array
    {
        return $this->vismaSPCS->getWayOfDeliveries();
    }

    public function getCustomerCategories(): array
    {
        return $this->vismaSPCS->getCustomerCategories();
    }

    public function getCustomerDistricts(): array
    {
        return $this->vismaSPCS->getCustomerDistricts();
    }

    public function getWayOfDelivery(
        string $carrier,
        string $method
    ): string {
        if ('visma' === $carrier) {
            return $method;
        }

        return $this->settings->getWayOfDelivery($this->getWebsite());
    }

    public function getShippingAgents(): array
    {
        $result = [];
        $rc = $this->vismaSPCS->getShippingAgents();
        foreach ($rc as $item) {
            $tmpItem = (array) $item;
            $tmpItem['code'] = $item->id;
            $tmpItem['serviceCode'] = $item->name;
            $result[] = (object) $tmpItem;
        }

        return $result;
    }

    public function getCapabilities(): array
    {
        $result[] = $this->capability;

        return $result;
    }

    public function getReturns(): array
    {
        return [];
    }

    public function getNonInventoryItems(): array
    {
        return $this->vismaSPCS->getItemListNonInventory();
    }

    protected function isProductForImport($item): bool
    {
        $productId = $this->productRepository->getIdBySku($item->sku);

        // If product already exist, then import and toggle status
        if ($productId) {
            return true;
        }

        // If product does not exist but web shop article is selected, then import
        if (true == $item->isWebshopArticle) {
            return true;
        }

        return false;
    }

    private function getCommonName($item): ?string
    {
        $template = $this->settings->getArticleTemplate($this->getWebSite());
        $attribMappingTbl = $this->settings->getAttributeMapping($this->getWebSite());

        $matches = [];
        $matches = \Ecomero\VismaSPCS\Helper\StringUtility::getPartsFromString($template, $item->sku);

        foreach ($matches as $group => $part) {
            $attribCode = $this->getMapping($group, $part, $attribMappingTbl);
            if (null == $attribCode) {
                throw new \RuntimeException('ERROR: No article mapping for group '.$group);
            }
            if ('erp_common_name' === $attribCode) {
                return $part;
            }
        }

        return null;
    }

    private function getMapping(int $group, string $part, array $mappingTable): ?string
    {
        // Search for specific matches
        foreach ($mappingTable as $mapping) {
            $mappingGroup = (int) $mapping['erp_sku_group'];
            $attribCode = $mapping['erp_sku_attribute'];
            $pattern = $mapping['erp_sku_pattern'];
            if ($mappingGroup === $group && $part === $pattern) {
                return $attribCode;
            }
        }

        // Search for wildcard matches
        foreach ($mappingTable as $mapping) {
            $mappingGroup = (int) $mapping['erp_sku_group'];
            $attribCode = $mapping['erp_sku_attribute'];
            $pattern = $mapping['erp_sku_pattern'];
            if ($mappingGroup === $group && fnmatch($pattern, $part, FNM_CASEFOLD)) {
                return $attribCode;
            }
        }

        return null;
    }
}
