<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\BusinessCentral\Model\Erp;

use Ecomero\BusinessCentral\Helper\Settings;
use Ecomero\BusinessCentral\Model\BusinessCentralAdapter;
use Ecomero\BusinessCentral\Model\BusinessCentralAPI;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpOrderInterface;
use Ecomero\ErpCore\Model\Order\Response as OrderResponse;
use Ecomero\ErpCore\Model\Order\ResponseFactory as OrderResponseFactory;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\Sales\Model\Order\Address;

class BcOrder extends BusinessCentralAdapter implements ErpOrderInterface
{
    private $lineNo = 0;

    public function __construct(
        BusinessCentralAPI $api,
        Settings $settings,
        Capability $capability,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        OrderResponseFactory $orderResponseFactory,
        ArrayManager $arrayManager
    ) {
        parent::__construct(
            $api,
            $settings,
            $capability,
            $storeManager,
            $orderResponseFactory,
            $arrayManager
        );
    }

    public function getSalesPersons(): array
    {
        $result = [];
        $url = 'salesPersons?$filter=blocked eq false';
        $rc = $this->api->callEcomeroApi($url);

        foreach ($rc as $item) {
            $result[] = $item;
        }

        return $result;
    }

    public function getPaymentMethods(): array
    {
        $result = [];
        $url = 'paymentMethods';
        $rc = $this->api->callStandardApi($url);

        foreach ($rc as $item) {
            $result[] = $item;
        }

        return $result;
    }

    public function getPaymentTerms(): array
    {
        $result = [];
        $url = 'paymentTerms';
        $rc = $this->api->callStandardApi($url);

        foreach ($rc as $item) {
            $result[] = $item;
        }

        return $result;
    }

    public function getDimensions(): array
    {
        $result = [];
        $url = 'dimensionValues';
        $rc = $this->api->callStandardApi($url);

        foreach ($rc as $item) {
            $result[] = $item;
        }

        return $result;
    }

    public function getProfitCenters(): array
    {
        return [];
    }

    public function getDocumentFromExternalId(
        string $docType,
        string $magentoOrderId
    ): object {
        $tmpResult = [];
        $rc = $this->api->callStandardApi('sales'.$docType.'s?$filter=externalDocumentNumber eq \''.$magentoOrderId.'\'');
        if (1 == count($rc)) {
            $tmpResult = $rc[0];

            return (object) $tmpResult;
        }

        return (object) [];
    }

    public function createDocument(
        string $docType,
        ?string $customerId,
        ?string $companyId,
        string $currency,
        Address $billingAddress,
        Address $shippingAddress,
        ?string $shippingAgent,
        ?string $shippingService,
        ?string $shippingPickupLocationId,
        ?string $shippingTmsReference,
        string $magentoOrderId,
        ?string $extPONumber,
        ?string $extGoodsMarking,
        ?string $extInvoiceMarking,
        string $salesPerson,
        ?string $profitCenter,
        ?string $termsOfDelivery,
        ?string $wayOfDelivery,
        ?string $customerCategory,
        ?string $customerDistrict,
        ?string $paymentMethod,
        ?string $paymentTerms
    ): OrderResponse {
        $this->lineNo = 1000;

        if ($currency === $this->settings->getLCY()) {
            // Do not set currency for the base currency
            $currency = '';
        }

        $contactName = $shippingAddress['firstname'].' '.$shippingAddress['lastname'];
        $companyName = $shippingAddress['company'];

        $doc = ['docType' => $docType,
            'sellCustomerId' => $companyId ?? $customerId,
            'billCustomerId' => $companyId ?? $customerId,
            'currencyCode' => $currency,
            'salesperson' => $salesPerson,
            'name' => $companyName ?? $contactName,
            'contact' => $contactName,
            'street' => $shippingAddress['street'],
            'city' => $shippingAddress['city'],
            'state' => $shippingAddress['region'],
            'phone' => $shippingAddress['telephone'],
            'countryLetterCode' => $shippingAddress['country_id'],
            'postalCode' => $shippingAddress['postcode'],
            'shippingAgentCode' => $shippingAgent,
            'shippingAgentServiceCode' => $shippingService,
            // 'shippingAdvice' => 'Complete', // "Partial" -- May be changed to a system setting in the future?
            'pickupLocationId' => $shippingPickupLocationId ?? '',
            'tmsReference' => $shippingTmsReference ?? '',
            'externalDocument' => $magentoOrderId,
        ];

        if ('-' !== $paymentMethod) {
            $doc['paymentMethod'] = $paymentMethod;
        }

        if ('-' !== $paymentTerms) {
            $doc['paymentTerms'] = $paymentTerms;
        }

        $payload = json_encode($doc);

        $rc = $this->api->callEcomeroApi('itemSalesOrders', \Laminas\Http\Request::METHOD_POST, $payload);

        if ($this->settings->isDimensionsEnabled()) {
            $erpId = $this->arrayManager->get('id', $rc);
            $payload = json_encode([
                'dim1' => $this->settings->getDimension1(),
                'dim2' => $this->settings->getDimension2(),
            ]);
            $rc = $this->api->callEcomeroApi('itemSalesOrders('.$erpId.')', \Laminas\Http\Request::METHOD_PATCH, $payload);
        }

        return $this->orderResponseFactory->create([
            'response' => [
                'customer_number' => $this->arrayManager->get('billCustomerId', $rc),
                'number' => $this->arrayManager->get('number', $rc),
                'id' => $this->arrayManager->get('id', $rc),
            ],
        ]);
    }

    public function updateDocument(
        string $docType,
        string $erpId,
        string $magentoOrderId
    ): OrderResponse {
        $payload = json_encode([
            'externalDocumentNumber' => $magentoOrderId,
        ]);
        $rc = $this->api->callStandardApi('sales'.$docType.'s('.$erpId.')', \Laminas\Http\Request::METHOD_PATCH, $payload);

        return $this->orderResponseFactory->create([
            'response' => [
                'customer_number' => $this->arrayManager->get('customerNumber', $rc),
                'number' => $this->arrayManager->get('number', $rc),
                'id' => $this->arrayManager->get('id', $rc),
            ],
        ]);
    }

    public function addDocumentItem(
        string $docType,
        string $orderId,
        string $erpItemId,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        float $discountAmount,
        float $discountPercent,
        ?string $comment,
        ?string $locationId
    ): void {
        $payload = json_encode(['itemId' => $erpItemId,
            'lineType' => 'Item',
            'docNo' => $orderId,
            'lineNo' => $this->lineNo,
            'docType' => $docType,
            'quantity' => ((float) $qtyOrdered),
            'unitPrice' => ((float) $price),
            'discountAmount' => ((float) $discountAmount),
            'discountPercent' => ((float) $discountPercent),
            'location' => $locationId,
            'taxCode' => $vatCode, ]);

        $this->api->callEcomeroApi('salesLines', \Laminas\Http\Request::METHOD_POST, $payload);
        $this->lineNo += 1000;

        if ($comment) {
            $payload = json_encode([
                'lineType' => 'Comment',
                'description' => str_replace("\n", ' - ', $comment),
            ]);

            $this->api->callEcomeroApi('salesLines', \Laminas\Http\Request::METHOD_POST, $payload);
            $this->lineNo += 1000;
        }
    }

    public function postOrder(string $salesOrderId): void
    {
        $this->api->callStandardApi('salesOrders('.$salesOrderId.')/Microsoft.NAV.shipAndInvoice', \Laminas\Http\Request::METHOD_POST, '', true);
    }
}
