<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\EcomNext\Model;

use Ecomero\ErpCore\Helper\Settings;
use Laminas\Http\Client;
use Laminas\Http\Request;
use Magento\Framework\Stdlib\ArrayManager;

class EcomNextAPI
{
    protected $client;
    protected $arrayManager;
    protected $settings;

    public function __construct(
        Client $client,
        Settings $settings,
        ArrayManager $arrayManager
    ) {
        $this->client = $client;
        $this->settings = $settings;
        $this->arrayManager = $arrayManager;
    }

    public function post(array $data): array
    {
        $app = $this->settings->getAppName();
        $rc = $this->sendRequest("https://connect.ecomero.com/api/queue/{$app}?max_poll_time=20", $data);

        $result = $this->arrayManager->get('result', $rc);

        $resultArray = json_decode($result, true);

        if ($msg = $this->arrayManager->get('Message', $resultArray)) {
            throw new \Laminas\Http\Exception\RuntimeException($msg);
        }

        return $resultArray;
    }

    private function sendRequest(
        string $endpoint,
        array $payLoad
    ): array {
        $this->client->reset();
        $this->client->setOptions(['timeout' => 60]);
        $request = new Request();
        $request->setUri($this->urlEncode($endpoint));
        $request->setMethod(\Laminas\Http\Request::METHOD_POST);
        $this->setHeaders($request);

        $encodedPayoad = json_encode($payLoad);
        $request->setContent($encodedPayoad);
        $request->getHeaders()->addHeaders([
            'Content-Length' => strlen($encodedPayoad),
            'Content-Type' => 'application/json',
        ]);

        $this->client->send($request);
        $response = $this->client->getResponse();

        if (200 !== $response->getStatusCode()) {
            throw new \Laminas\Http\Exception\RuntimeException($response->getReasonPhrase());
        }

        return json_decode($response->getBody(), true);
    }

    private function setHeaders(Request $request): void
    {
        $request->getHeaders()->addHeaders([
            'Authorization' => 'Bearer '.$this->settings->getAppSecret(),
            'User-Agent' => 'Magento',
            'Cache-Control' => 'no-cache',
            'Accept' => 'application/json',
        ]);
    }

    private function urlEncode(string $url): string
    {
        $url = str_replace(' ', '%20', $url);
        $url = str_replace("'", '%27', $url);
        $url = str_replace('å', '%C3%A5', $url);
        $url = str_replace('ä', '%C3%A4', $url);
        $url = str_replace('ö', '%C3%B6', $url);
        $url = str_replace('Å', '%C3%85', $url);
        $url = str_replace('Ä', '%C3%84', $url);

        return str_replace('Ö', '%C3%96', $url);
    }
}
