<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\EcomNext\Model\Erp;

use Ecomero\EcomNext\Model\EcomNextAdapter;
use Ecomero\EcomNext\Model\EcomNextAPI;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpOrderInterface;
use Ecomero\ErpCore\Model\Order\Response as OrderResponse;
use Ecomero\ErpCore\Model\Order\ResponseFactory as OrderResponseFactory;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\Sales\Model\Order\Address;

class Order extends EcomNextAdapter implements ErpOrderInterface
{
    protected $orderResponseFactory;

    public function __construct(
        EcomNextAPI $api,
        Settings $settings,
        Capability $capability,
        \Magento\Store\Model\StoreManagerInterface $storeManager,
        OrderResponseFactory $orderResponseFactory,
        ArrayManager $arrayManager
    ) {
        parent::__construct(
            $api,
            $settings,
            $capability,
            $arrayManager,
            $storeManager,
        );

        $this->orderResponseFactory = $orderResponseFactory;
    }

    public function getSalesPersons(): array
    {
        $result = $this->api->post([
            'command' => 'sales-person',
        ]);

        return $this->arrayManager->get('Items', $result);
    }

    public function getProfitCenters(): array
    {
        return [];
    }

    public function getDocumentFromExternalId(
        string $docType,
        string $magentoOrderId
    ): object {
        $tmpResult = [];

        if (0 === strpos($magentoOrderId, 'PENDING(')) {
            $magentoOrderId = str_replace('PENDING(', '', $magentoOrderId);
            $magentoOrderId = rtrim($magentoOrderId, ')');
        }

        $rc = $this->api->post([
            'command' => 'order-getbyid',
            'order-id' => $magentoOrderId,
        ]);
        if (1 == count($rc)) {
            return (object) $rc;
        }

        return (object) [];
    }

    public function createDocument(
        string $docType,
        string $customerId,
        ?string $companyId,
        string $currency,
        Address $billingAddress,
        Address $shippingAddress,
        ?string $shippingAgent,
        ?string $shippingService,
        ?string $shippingPickupLocationId,
        string $magentoOrderId,
        string $salesPerson,
        ?string $profitCenter,
        ?string $termsOfDelivery,
        ?string $wayOfDelivery,
        ?string $customerCategory,
        ?string $customerDistrict
    ): OrderResponse {
        if ($currency === $this->settings->getLCY()) {
            // Do not set currency for the base currency
            $currency = '';
        }

        $this->currentOrder['salesperson'] = $salesPerson;
        $this->currentOrder['profitCenter'] = $profitCenter;
        $this->currentOrder['termsOfDelivery'] = $termsOfDelivery;
        $this->currentOrder['wayOfDelivery'] = $wayOfDelivery;
        $this->currentOrder['customerCategory'] = $customerCategory;
        $this->currentOrder['customerDistrict'] = $customerDistrict;

        $this->currentOrder['customerNumber'] = $customerId;
        $this->currentOrder['companyNumber'] = $companyId;
        $this->currentOrder['currencyCode'] = $currency;
        $this->currentOrder['externalDocument'] = $magentoOrderId;
        $this->currentOrder['vatNumber'] = $billingAddress->getVatId();

        // Billing address
        $name = $billingAddress->getFirstname().' '.$billingAddress->getLastname();

        $this->currentOrder['name'] = $billingAddress->getCompany() ?? $name;
        $this->currentOrder['reference'] = $name;
        $this->currentOrder['telephone'] = $billingAddress->getTelephone();
        $this->currentOrder['street'] = '';
        $this->currentOrder['street2'] = '';

        if (count($billingAddress->getStreet()) >= 1) {
            $this->currentOrder['street'] = $billingAddress->getStreetLine(1);
        }
        if (count($billingAddress->getStreet()) >= 2) {
            $this->currentOrder['street2'] = $billingAddress->getStreetLine(2);
        }
        $this->currentOrder['city'] = $billingAddress->getCity();
        $this->currentOrder['countryLetterCode'] = $billingAddress->getCountryId();
        $this->currentOrder['postalCode'] = $billingAddress->getPostcode();
        $this->currentOrder['GLN'] = '';

        // Delivery address
        $this->currentOrder['deliveryName'] = $shippingAddress->getCompany() ?? $name;
        $this->currentOrder['deliveryTelephone'] = $shippingAddress->getTelephone();
        $this->currentOrder['deliveryStreet'] = '';
        $this->currentOrder['deliveryStreet2'] = '';

        if (count($shippingAddress->getStreet()) >= 1) {
            $this->currentOrder['deliveryStreet'] = $shippingAddress->getStreetLine(1);
        }
        if (count($shippingAddress->getStreet()) >= 2) {
            $this->currentOrder['deliveryStreet2'] = $shippingAddress->getStreetLine(2);
        }
        $this->currentOrder['deliveryCity'] = $shippingAddress->getCity();
        $this->currentOrder['deliveryCountryLetterCode'] = $shippingAddress->getCountryId();
        $this->currentOrder['deliveryPostalCode'] = $shippingAddress->getPostcode();
        $this->currentOrder['deliveryGLN'] = '';

        $this->currentOrder['orderLines'] = [];
        $this->currentOrder['number'] = '';
        $this->currentOrder['id'] = '';

        return $this->orderResponseFactory->create([
            'response' => $this->currentOrder,
        ]);
    }

    public function updateDocument(
        string $docType,
        string $erpId,
        string $magentoOrderId
    ): OrderResponse {
        $this->currentOrder['externalDocument'] = $magentoOrderId;

        $response = $this->api->post([
            'command' => 'order-create',
            'data' => json_encode($this->currentOrder),
        ]);

        return $this->orderResponseFactory->create([
            'response' => $response,
        ]);
    }

    public function addDocumentItem(
        string $docType,
        string $orderId,
        string $erpItemId,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        float $discountAmount,
        float $discountPercent,
        ?string $comment
    ): void {
        $discountIsAmount = false;
        $discount = 0;
        if ($discountAmount > 0) {
            $discountIsAmount = true;
            $discount = $discountAmount;
        } else {
            $discountIsAmount = false;
            $discount = $discountPercent;
        }

        if (null !== $comment) {
            $finalComment = '';
            $commentLines = explode("\n", $comment);
            foreach ($commentLines as $line) {
                while (mb_strlen($line) > self::MAX_COMMENT_LEN) {
                    $finalComment .= mb_substr($line, 0, self::MAX_COMMENT_LEN)."\n";
                    $line = mb_substr($line, self::MAX_COMMENT_LEN, mb_strlen($line));
                }
                $finalComment .= $line."\n";
            }
            $comment = trim($finalComment);
        }

        $this->currentOrder['orderLines'][] = [
            'articleNumber' => $erpItemId,
            'qty' => $qtyOrdered,
            'price' => $price,
            'discount' => $discount,
            'discountIsAmount' => $discountIsAmount,
            'amount' => $price * $qtyOrdered,
            'vatCode' => $vatCode,
            'comment' => $comment ?? '',
        ];
    }

    public function addChargeItem(
        string $docType,
        string $orderId,
        string $chargingSKU,
        float $price,
        float $qtyOrdered,
        string $vatCode,
        string $invoiceNo,
        int $numOrderLines
    ): void {
        $this->currentOrder['shippingAmount'] = $price * $qtyOrdered;
        $this->currentOrder['shippingVATCode'] = $vatCode;
    }

    public function postOrder(string $salesOrderId): void
    {
    }
}
