<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpB2b\Model\Company;

use Ecomero\ErpB2b\Api\Data\CompanyErpInterfaceFactory;
use Ecomero\ErpB2b\Helper\CustomerGroupHelper;
use Ecomero\ErpCore\Helper\CronLock;
use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpCustomerInterface;
use Magento\Company\Api\CompanyRepositoryInterface;
use Magento\Company\Api\Data\CompanyCustomerInterfaceFactory;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Api\Data\AddressInterfaceFactory;
use Magento\Customer\Api\Data\CustomerInterfaceFactory;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\Store\Model\StoreManagerInterface;

class Import extends \Ecomero\ErpCore\Model\Executor
{
    protected $customerGroupHelper;
    protected $settings;
    protected $storeManager;
    protected $erp;
    protected $arrayManager;
    protected $customerRepository;
    protected $customerFactory;
    protected $companyRepository;
    protected $companyFactory;
    protected $companyCustomerFactory;
    protected $searchCriteriaBuilder;
    protected $addressFactory;

    public function __construct(
        CustomerGroupHelper $customerGroupHelper,
        ErpCustomerInterface $erp,
        ErpLogger $logger,
        Notification $notification,
        Settings $settings,
        CronLock $cronLock,
        StoreManagerInterface $storeManager,
        ArrayManager $arrayManager,
        CustomerRepositoryInterface $customerRepository,
        CustomerInterfaceFactory $customerFactory,
        CompanyRepositoryInterface $companyRepository,
        CompanyErpInterfaceFactory $companyFactory,
        CompanyCustomerInterfaceFactory $companyCustomerFactory,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        AddressInterfaceFactory $addressFactory
    ) {
        parent::__construct(
            $logger,
            $cronLock,
            $notification,
            $erp
        );

        $this->customerGroupHelper = $customerGroupHelper;
        $this->erp = $erp;
        $this->settings = $settings;
        $this->storeManager = $storeManager;
        $this->arrayManager = $arrayManager;
        $this->customerRepository = $customerRepository;
        $this->customerFactory = $customerFactory;
        $this->companyRepository = $companyRepository;
        $this->companyFactory = $companyFactory;
        $this->companyCustomerFactory = $companyCustomerFactory;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->addressFactory = $addressFactory;
    }

    protected function run(): string
    {
        return $this->import();
    }

    protected function getServiceDescription(): string
    {
        return 'company import';
    }

    protected function getCapability(): string
    {
        return Capability::COMPANY_IMPORT;
    }

    private function import(): string
    {
        $errorMessage = '';

        \Ecomero\ErpB2b\Rewrite\Magento\Company\Model\Email\Sender::setEnableNotifications(false);

        try {
            $companies = $this->erp->getCompanyList();
            $maxCompany = count($companies);
            $pos = 0;
            foreach ($companies as $company) {
                ++$pos;

                $customers = $this->arrayManager->get('customers', $company);
                if (null === $customers || 0 === count($customers)) {
                    $this->logger->info("{$pos}/{$maxCompany} [SKIPPING] company {$company['name']} ({$company['id']}), no customer assigned in ERP");

                    continue;
                }
                $this->logger->info("{$pos}/{$maxCompany} [IMPORTING] company {$company['name']} ({$company['id']})");
                $this->createCustomers($customers);

                $superUser = $this->getSuperUser($customers);
                if (null === $superUser) {
                    $this->logger->error("Cannot import company, no customer for company {$company['name']} in Magento (no email in ERP?)");

                    continue;
                }

                $companyDto = $this->getCompanyFromErpId($company['id']);
                if ($companyDto) {
                    $this->logger->info("--> Company {$company['name']} already in Magento");
                } else {
                    $companyDto = $this->createCompany($company, $superUser);
                }
                $this->assignPriceList($companyDto, $company['priceList'].'_'.$company['agreement']);
                $this->assignCustomers((int) $companyDto->getId(), $customers);
            }
        } catch (\Laminas\Http\Exception\RuntimeException $exception) {
            $errorMessage = $errorMessage.$exception->getMessage()."\n";
            $this->logger->error($errorMessage);
        } catch (\Magento\Framework\Exception\LocalizedException $exception) {
            $errorMessage = $errorMessage.$exception->getMessage()."\n";
            $this->logger->error($errorMessage);
        } catch (\RuntimeException $exception) {
            $errorMessage = $errorMessage.$exception->getMessage()."\n";
            $this->logger->error($errorMessage);
        }

        \Ecomero\ErpB2b\Rewrite\Magento\Company\Model\Email\Sender::setEnableNotifications(true);

        return $errorMessage;
    }

    private function createCompany(array $company, \Magento\Customer\Api\Data\CustomerInterface $superUser): \Magento\Company\Api\Data\CompanyInterface
    {
        $customerGroup = $this->customerGroupHelper->getCustomerGroupFromPriceListId($company['priceList']);

        $companyDto = $this->companyFactory->create();
        $companyDto->setCustomerGroupId($customerGroup);
        $companyDto->setSuperUserId($superUser->getId());
        $companyDto->setCompanyName($company['name']);
        $companyDto->setCompanyEmail($company['email'] ?: $superUser->getEmail());
        $companyDto->setStreet($company['street'] ?: ($company['deliveryStreet'] ?: '-'));
        $companyDto->setCity($company['city'] ?: ($company['deliveryCity'] ?: '-'));
        $companyDto->setCountryId($company['countryId'] ?: 'SE');
        $companyDto->setPostcode($company['postalCode'] ?: ($company['deliveryPostalCode'] ?: '-'));
        $companyDto->setTelephone($company['telephone'] ?: '-');
        $companyDto->setVatTaxId($company['orgNo']);
        $companyDto->setErpCompanyNo($company['id']);

        try {
            $companyDto = $this->companyRepository->save($companyDto);
        } catch (\Magento\Framework\Exception\InputException $e) {
            $this->logger->error($e->getMessage());
            foreach ($e->getErrors() as $error) {
                $this->logger->error($error->getMessage());
            }
        }

        return $companyDto;
    }

    private function createCustomers(array $customers): void
    {
        foreach ($customers as $customer) {
            $customerEmail = $this->arrayManager->get('email', $customer);
            if ($customerEmail) {
                try {
                    $customer = $this->customerRepository->get($customerEmail);
                } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
                    try {
                        $newCustomer = $this->customerFactory->create();
                        $newCustomer->setEmail($customerEmail);
                        $newCustomer->setFirstname($customer['firstName'] ?: '-');
                        $newCustomer->setLastname($customer['lastName'] ?: '-');
                        $newCustomer = $this->customerRepository->save($newCustomer);
                    } catch (\Magento\Framework\Exception\InputException $e) {
                        $this->logger->error($e->getMessage()." {$customerEmail}");
                        foreach ($e->getErrors() as $error) {
                            $this->logger->error($error->getMessage());
                        }
                    }

                    try {
                        $address = $this->addressFactory->create();
                        $address->setFirstname($customer['firstName']);
                        $address->setLastname($customer['lastName']);
                        $address->setCountryId($customer['countryId'] ?: 'SE');
                        $address->setStreet([$customer['street']]);
                        $address->setCity($customer['city']);
                        $address->setPostcode($customer['postalCode']);
                        $telephone = $customer['telephoneMobile'] ?: ($customer['telephoneDirect'] ?: $customer['telephoneSwitch']);
                        $address->setTelephone($telephone);
                        $address->setIsDefaultBilling(true);
                        $address->setIsDefaultShipping(true);
                        $newCustomer->setAddresses([$address]);
                        $this->customerRepository->save($newCustomer);
                    } catch (\Magento\Framework\Exception\InputException $e) {
                        // If we cannot store the address, just ignore it, it is not important
                    }
                }
            }
        }
    }

    private function getSuperUser(array $customers): ?\Magento\Customer\Api\Data\CustomerInterface
    {
        foreach ($customers as $customer) {
            $customerEmail = $this->arrayManager->get('email', $customer);
            if ($customerEmail) {
                try {
                    $customer = $this->customerRepository->get($customerEmail);
                } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
                    $customer = null;
                }
                if ($customer) {
                    return $customer;
                }
            }
        }

        return null;
    }

    private function getCompanyFromErpId(string $erpId): ?\Magento\Company\Api\Data\CompanyInterface
    {
        $filter = $this->searchCriteriaBuilder->addFilter('erp_company_no', $erpId)->create();
        $collection = $this->companyRepository->getList($filter);
        foreach ($collection->getItems() as $item) {
            return $item;
        }

        return null;
    }

    private function assignCustomers(int $companyId, array $customers)
    {
        foreach ($customers as $customer) {
            $customerEmail = $this->arrayManager->get('email', $customer);
            if ($customerEmail) {
                try {
                    $customer = $this->customerRepository->get($customerEmail);
                    if ($customer) {
                        if ($customer->getExtensionAttributes()->getCompanyAttributes()
                            && $customer->getExtensionAttributes()->getCompanyAttributes()->getCompanyId()) {
                            $this->logger->info("--> {$customer->getEmail()} is already assigned to a company");

                            continue;
                        }
                        $attributes = $this->companyCustomerFactory->create();
                        $attributes->setCompanyId($companyId);
                        $attributes->setCustomerId($customer->getId());
                        $customer->getExtensionAttributes()->setCompanyAttributes($attributes);
                        $this->customerRepository->save($customer);
                    }
                } catch (\Exception $e) {
                    $this->logger->error("Cannot assign {$customerEmail} to company");
                }
            }
        }
    }

    private function assignPriceList(\Magento\Company\Api\Data\CompanyInterface $companyDto, string $priceListId)
    {
        try {
            $customerGroup = $this->customerGroupHelper->getCustomerGroupFromPriceListId($priceListId);
            $companyDto->setCustomerGroupId($customerGroup);
            $companyDto = $this->companyRepository->save($companyDto);
        } catch (\Magento\Framework\Exception\InputException $e) {
            $this->logger->error($e->getMessage());
            foreach ($e->getErrors() as $error) {
                $this->logger->error($error->getMessage());
            }
        }
    }
}
