<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpB2b\Model\Price;

use Ecomero\ErpB2b\Helper\CustomerGroupHelper;
use Ecomero\ErpB2b\Helper\ProductHelper;
use Ecomero\ErpCore\Helper\CronLock;
use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpCatalogInterface;
use Magento\Company\Api\CompanyRepositoryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\Store\Api\WebsiteRepositoryInterface;

class Import extends \Ecomero\ErpCore\Model\Executor
{
    protected $customerGroupHelper;
    protected $productHelper;
    protected $settings;
    protected $websiteRepository;
    protected $erp;
    protected $arrayManager;
    protected $companyRepository;
    protected $searchCriteriaBuilder;

    public function __construct(
        CustomerGroupHelper $customerGroupHelper,
        ProductHelper $productHelper,
        ErpCatalogInterface $erp,
        ErpLogger $logger,
        Notification $notification,
        Settings $settings,
        CronLock $cronLock,
        WebsiteRepositoryInterface $websiteRepository,
        ArrayManager $arrayManager,
        CompanyRepositoryInterface $companyRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder
    ) {
        parent::__construct(
            $logger,
            $cronLock,
            $notification,
            $erp
        );

        $this->customerGroupHelper = $customerGroupHelper;
        $this->productHelper = $productHelper;
        $this->erp = $erp;
        $this->settings = $settings;
        $this->websiteRepository = $websiteRepository;
        $this->arrayManager = $arrayManager;
        $this->companyRepository = $companyRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
    }

    protected function run(): string
    {
        return $this->import();
    }

    protected function getServiceDescription(): string
    {
        return 'price import';
    }

    protected function getCapability(): string
    {
        return Capability::PRICE_IMPORT;
    }

    private function import(): string
    {
        $errorMessage = '';

        try {
            $priceLists = $this->erp->getItemPriceLists();
            $this->customerGroupHelper->deleteImportedTierPrices();
            foreach ($priceLists as $priceList) {
                $name = $this->arrayManager->get('name', $priceList);
                $id = $this->arrayManager->get('id', $priceList);
                $prices = $this->arrayManager->get('prices', $priceList);
                $numPrices = count($prices);

                $this->logger->info("Process price list {$name} (id = {$id}, number of prices {$numPrices})");
                if (0 === $numPrices) {
                    continue;
                }
                $customerGroup = $this->customerGroupHelper->createIfNotExist($name, $id);
                $customerGroupId = $customerGroup->getId();
                $tierPrices = [];

                foreach ($prices as $priceData) {
                    $sku = $this->arrayManager->get('sku', $priceData);
                    $price = $this->arrayManager->get('price', $priceData);
                    $currency = $this->arrayManager->get('currency', $priceData) ?: 'SEK';
                    $from = $this->arrayManager->get('from_date', $priceData);
                    $to = $this->arrayManager->get('to_date', $priceData);

                    $articleType = $this->arrayManager->get('article_type', $priceData);
                    $thresholds = $this->arrayManager->get('thresholds', $priceData);
                    $discount = $this->arrayManager->get('discount', $priceData);
                    $priceType = $this->arrayManager->get('price_type', $priceData);
                    $numThresholds = count($thresholds);
                    $this->logger->info("   --> SKU: {$sku}, PRICE: {$price}, CURRENCY: {$currency}, DISCOUNT: {$discount}, ARTICLE_TYPE: {$articleType}, THRESHOLD: {$numThresholds}, FROM: {$from}, TO: {$to}");

                    try {
                        $websiteId = $this->getWebsiteForCurrency($currency);

                        $tierPrices = $this->getTierPriceData(
                            $tierPrices,
                            $sku,
                            (int) 1,
                            (float) (ErpCatalogInterface::PRICE_TYPE_DISCOUNT === $priceType) ? $discount : $price,
                            $priceType,
                            (int) $websiteId,
                            $articleType,
                            (int) $customerGroupId
                        );

                        foreach ($thresholds as $threshold) {
                            $qty = $this->arrayManager->get('threshold', $threshold);
                            $priceType = $this->arrayManager->get('price_type', $threshold);
                            $price = (ErpCatalogInterface::PRICE_TYPE_DISCOUNT === $priceType) ? $threshold['discount'] : $threshold['price'];
                            $tierPrices = $this->getTierPriceData(
                                $tierPrices,
                                $sku,
                                (int) $qty,
                                (float) $price,
                                $priceType,
                                (int) $websiteId,
                                $articleType,
                                (int) $customerGroupId
                            );
                        }
                    } catch (NoSuchEntityException $e) {
                        $this->logger->error("   --> [{$currency}] {$e->getMessage()}");
                    }
                }
                $maxPriceCount = count($tierPrices);
                $this->logger->info("   --> Updating {$maxPriceCount} tier prices");
                if ($tierPrices) {
                    $this->customerGroupHelper->updateTierPrices($tierPrices, (int) $customerGroupId);
                }

                $filter = $this->searchCriteriaBuilder->addFilter('erp_price_list_id', $id)->create();
                $collection = $this->companyRepository->getList($filter);
                foreach ($collection->getItems() as $company) {
                    $this->assignPriceList($company, (int) $customerGroupId);
                }
            }
        } catch (\Laminas\Http\Exception\RuntimeException $exception) {
            $errorMessage = $errorMessage.$exception->getMessage()."\n";
            $this->logger->error($errorMessage);
        } catch (\Magento\Framework\Exception\LocalizedException $exception) {
            $errorMessage = $errorMessage.$exception->getMessage()."\n";
            $this->logger->error($errorMessage);
        } catch (\RuntimeException $exception) {
            $errorMessage = $errorMessage.$exception->getMessage()."\n";
            $this->logger->error($errorMessage);
        }

        return $errorMessage;
    }

    private function getWebsiteForCurrency(string $currency): int
    {
        $websiteId = 0;
        foreach ($this->websiteRepository->getList() as $website) {
            foreach ($website->getStores() as $store) {
                if (strtolower($store->getCurrentCurrency()->getCode()) === strtolower($currency)) {
                    $websiteId = (int) $website->getId();
                }
            }
        }

        return $websiteId;
    }

    private function getTierPriceData(array $tierPrices, string $sku, int $qty, float $price, string $priceType, int $websiteId, string $articleType, int $customerGroupId): array
    {
        if (ErpCatalogInterface::PRICE_TYPE_DISCOUNT === $priceType && 0 == $price) {
            return $tierPrices;
        }

        if ($sku) {
            $productId = $this->productHelper->getRowId($sku);
            if ($productId) {
                $tierPrices[$sku][$qty] = [
                    'website_id' => $websiteId,
                    'qty' => (int) $qty,
                    ErpCatalogInterface::PRICE_TYPE_DISCOUNT === $priceType ? 'percentage_value' : 'value' => $price,
                    'customer_group_id' => $customerGroupId,
                    'row_id' => $productId,
                    'all_groups' => 0,
                ];
            }
        } else {
            $productCollection = $this->productHelper->getProductIds('article_type', $articleType);

            foreach ($productCollection as $item) {
                $sku = trim($this->arrayManager->get('sku', $item));
                $productId = $this->arrayManager->get('entity_id', $item);
                $tierPrices[$sku][$qty] = [
                    'website_id' => $websiteId,
                    'qty' => (int) $qty,
                    ErpCatalogInterface::PRICE_TYPE_DISCOUNT === $priceType ? 'percentage_value' : 'value' => $price,
                    'row_id' => $productId,
                    'customer_group_id' => $customerGroupId,
                    'all_groups' => 0,
                ];
            }
        }

        return $tierPrices;
    }

    private function assignPriceList(\Magento\Company\Api\Data\CompanyInterface $companyDto, int $customerGroupId)
    {
        try {
            $companyDto->setCustomerGroupId($customerGroupId);
            $companyDto = $this->companyRepository->save($companyDto);
        } catch (\Magento\Framework\Exception\InputException $e) {
            $this->logger->error($e->getMessage());
            foreach ($e->getErrors() as $error) {
                $this->logger->error($error->getMessage());
            }
        }
    }
}
