<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpB2b\Helper;

use Magento\Company\Api\Data\CompanyCustomerInterfaceFactory;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Api\Data\AddressInterfaceFactory;
use Magento\Customer\Api\Data\CustomerInterface;
use Magento\Customer\Api\Data\CustomerInterfaceFactory;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\ResourceConnection;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\Store\Model\ScopeInterface;
use Magento\Store\Model\StoreManagerInterface;

class CustomerHelper
{
    protected const DEFAULT_SITE = 1;

    protected $connection;
    protected $addressFactory;
    protected $customerRepository;
    protected $customerFactory;
    protected $companyCustomerFactory;
    protected $arrayManager;
    protected $storeManager;
    protected $scopeConfig;
    protected $logger;
    protected $data;

    public function __construct(
        ResourceConnection $connection,
        AddressInterfaceFactory $addressFactory,
        CustomerRepositoryInterface $customerRepository,
        CustomerInterfaceFactory $customerFactory,
        CompanyCustomerInterfaceFactory $companyCustomerFactory,
        StoreManagerInterface $storeManager,
        ScopeConfigInterface $scopeConfig,
        Data $data,
        ArrayManager $arrayManager
    ) {
        $this->connection = $connection;
        $this->addressFactory = $addressFactory;
        $this->customerRepository = $customerRepository;
        $this->customerFactory = $customerFactory;
        $this->companyCustomerFactory = $companyCustomerFactory;
        $this->storeManager = $storeManager;
        $this->scopeConfig = $scopeConfig;
        $this->arrayManager = $arrayManager;
        $this->data = $data;
    }

    public function setLogger($logger)
    {
        $this->logger = $logger;
    }

    public function createCustomers(array $customers, string $companyName): void
    {
        foreach ($customers as $customer) {
            $customerEmail = $this->arrayManager->get('email', $customer);
            if ($customerEmail) {
                try {
                    $existingCustomer = $this->customerRepository->get($customerEmail);
                    $this->updateErpContactNo($customer, (int)$existingCustomer->getId());
                } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
                    try {
                        $websiteId = $this->getWebsiteIdFromCountry($customer['countryId'] ?: 'SE');

                        $newCustomer = $this->customerFactory->create();
                        $newCustomer->setEmail($customerEmail);
                        $newCustomer->setFirstname($customer['firstName'] ?: '-');
                        $newCustomer->setLastname($customer['lastName'] ?: '-');
                        $newCustomer->setWebsiteId($websiteId);
                        $defaultStoreId = $this->getDefaultStoreIdByWebsite((int) $websiteId);
                        if ($defaultStoreId) {
                            $newCustomer->setStoreId($defaultStoreId);
                        }
                        $newCustomer = $this->customerRepository->save($newCustomer);
                        $this->updateErpContactNo($customer, (int)$newCustomer->getId());
                    } catch (\Magento\Framework\Exception\InputException $e) {
                        $this->logger->error($e->getMessage()." {$customerEmail}");
                        foreach ($e->getErrors() as $error) {
                            $this->logger->error($error->getMessage());
                        }
                    }

                    try {
                        $address = $this->addressFactory->create();
                        $address->setFirstname($customer['firstName']);
                        $address->setLastname($customer['lastName']);
                        $address->setCountryId($customer['countryId'] ?: 'SE');
                        $address->setStreet([$customer['street']]);
                        $address->setCity($customer['city']);
                        $address->setPostcode($customer['postalCode']);
                        $address->setCompany($companyName);
                        $telephone = $customer['telephoneMobile'] ?: ($customer['telephoneDirect'] ?: $customer['telephoneSwitch']);
                        $address->setTelephone($telephone);
                        $address->setIsDefaultBilling(true);
                        $address->setIsDefaultShipping(true);
                        $newCustomer->setAddresses([$address]);
                        $this->customerRepository->save($newCustomer);
                    } catch (\Magento\Framework\Exception\InputException $e) {
                        // If we cannot store the address, just ignore it, it is not important
                    }
                }
            }
        }
    }

    private function getDefaultStoreIdByWebsite(int $websiteId):?int
    {
        $stores = $this->storeManager->getWebsite($websiteId)->getStores();
        $result = [];
        foreach ($stores as $store) {
            $result[$store->getSortOrder()] = (int)$store->getId();
        }
        $lowestSortOrder = min(array_keys($result));

        return $result[$lowestSortOrder];
    }

    private function updateErpContactNo(array $customer, int $customerId) {
        if( $customerId) {
            $customerNumber = $this->arrayManager->get('number', $customer);
            $adapter = $this->connection->getConnection();
            $sqlStatement = <<<'SQL'
                UPDATE	customer_entity
                SET     erp_contact_no = ? 
                WHERE	entity_id = ?
SQL;

            $adapter->query($sqlStatement, [$customerNumber, $customerId]);    
        }
    }

    public function getSuperUser(array $customers, array $company): ?\Magento\Customer\Api\Data\CustomerInterface
    {
        $customerReference = $company['customerReference'] ?? '';

        $firstCustomer = null;
        foreach ($customers as $customerArray) {
            $customerEmail = $this->arrayManager->get('email', $customerArray);
            $customer = $this->getCustomerByEmail($customerEmail);
            if (!$customer) {
                continue;
            }
            if (!$firstCustomer) {
                $firstCustomer = $customer;
            }
            if ($this->isReferenceAndNameMatching($customerReference, $customer)) {
                return $customer;
            }
        }

        return $firstCustomer;
    }

    private function isReferenceAndNameMatching(string $customerReference, CustomerInterface $customer):bool
    {
        if ((int)strlen($customerReference) > 0) {
            $customerName = strtolower($customer->getFirstname() . ' ' . $customer->getLastname());
            if (strtolower($customerReference) === $customerName) {
                return true;
            }
        }
        return false;
    }

    private function getCustomerByEmail(string $email):?CustomerInterface
    {
        try {
            $customer = $this->customerRepository->get($email);
        } catch (\Magento\Framework\Exception\NoSuchEntityException $e) {
            $customer = null;
        }

        return $customer;
    }

    public function assignCustomers(int $companyId, array $customers)
    {
        foreach ($customers as $customer) {
            $customerEmail = $this->arrayManager->get('email', $customer);
            if ($customerEmail) {
                try {
                    $customer = $this->customerRepository->get($customerEmail);
                    if ($customer) {
                        if ($customer->getExtensionAttributes()->getCompanyAttributes()
                            && $customer->getExtensionAttributes()->getCompanyAttributes()->getCompanyId()) {
                            $this->logger->info("--> {$customer->getEmail()} is already assigned to a company");

                            continue;
                        }
                        $attributes = $this->companyCustomerFactory->create();
                        $attributes->setCompanyId($companyId);
                        $attributes->setCustomerId($customer->getId());
                        $attributes->setStatus($this->data->getIsActiveOnImport());
                        $customer->getExtensionAttributes()->setCompanyAttributes($attributes);
                        $this->customerRepository->save($customer);
                    }
                } catch (\Exception $e) {
                    $this->logger->error("Cannot assign {$customerEmail} to company");
                }
            }
        }
    }

    private function getWebsiteIdFromCountry(string $countryCode): int
    {
        foreach ($this->storeManager->getWebsites() as $website) {
            $websiteCountryCode = $this->data->getDefaultCountryByWebsite((int) $website->getId());
            if ($websiteCountryCode === $countryCode) {
                return (int) $website->getId();
            }
        }

        return $this->data->getDefaultLanguageWebsiteId();
    }
}
