<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpB2b\Model\Company;

use Ecomero\ErpB2b\Api\Data\CompanyErpInterfaceFactory;
use Ecomero\ErpB2b\Helper\CustomerGroupHelper;
use Ecomero\ErpB2b\Helper\CustomerHelper;
use Ecomero\ErpB2b\Helper\Data;
use Ecomero\ErpB2b\Helper\ProductHelper;
use Ecomero\ErpCore\Helper\CronLock;
use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpCustomerInterface;
use Magento\Company\Api\CompanyRepositoryInterface;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\InputException;
use Magento\Framework\Exception\LocalizedException;
use Magento\Framework\Exception\State\InputMismatchException;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\Setup\Exception;
use Magento\User\Model\ResourceModel\User;

class Import extends \Ecomero\ErpCore\Model\Executor
{
    protected $customerGroupHelper;
    protected $customerHelper;
    protected $productHelper;
    protected $settings;
    protected $erp;
    protected $arrayManager;
    protected $companyRepository;
    protected $companyFactory;
    protected $searchCriteriaBuilder;
    /**
     * @var User
     */
    private User $salesLoader;
    private Data $data;
    private CustomerRepositoryInterface $customerRepository;

    public function __construct(
        CustomerGroupHelper $customerGroupHelper,
        CustomerHelper $customerHelper,
        ProductHelper $productHelper,
        ErpCustomerInterface $erp,
        ErpLogger $logger,
        Data $data,
        CustomerRepositoryInterface $customerRepository,
        Notification $notification,
        Settings $settings,
        CronLock $cronLock,
        User $salesLoader,
        ArrayManager $arrayManager,
        CompanyRepositoryInterface $companyRepository,
        CompanyErpInterfaceFactory $companyFactory,
        SearchCriteriaBuilder $searchCriteriaBuilder
    ) {
        parent::__construct(
            $logger,
            $cronLock,
            $notification,
            $erp
        );

        $this->customerGroupHelper = $customerGroupHelper;
        $this->customerHelper = $customerHelper;
        $this->productHelper = $productHelper;
        $this->erp = $erp;
        $this->settings = $settings;
        $this->arrayManager = $arrayManager;
        $this->companyRepository = $companyRepository;
        $this->companyFactory = $companyFactory;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;

        $customerHelper->setLogger($logger);
        $this->salesLoader = $salesLoader;
        $this->data = $data;
        $this->customerRepository = $customerRepository;
    }

    protected function run(bool $force): string
    {
        return $this->import();
    }

    protected function getServiceDescription(): string
    {
        return 'company import';
    }

    protected function getCapability(): string
    {
        return Capability::COMPANY_IMPORT;
    }

    private function import(): string
    {
        $errorMessage = '';

        \Ecomero\ErpB2b\Rewrite\Magento\Company\Model\Email\Sender::setEnableNotifications(false);

        try {
            $companies = $this->erp->getCompanyList();
            $maxCompany = count($companies);
            $pos = 0;
            foreach ($companies as $company) {
                ++$pos;

                $customers = $this->arrayManager->get('customers', $company);
                if (null === $customers || 0 === count($customers)) {
                    $this->logger->info("{$pos}/{$maxCompany} [SKIPPING] company {$company['name']} ({$company['id']}), no customer assigned in ERP");

                    continue;
                }
                $this->logger->info("{$pos}/{$maxCompany} [IMPORTING] company {$company['name']} ({$company['id']})");
                $this->customerHelper->createCustomers($customers, $company['name']);

                $superUser = $this->customerHelper->getSuperUser($customers, $company);
                if (null === $superUser) {
                    $this->logger->error("Cannot import company, no customer for company {$company['name']} in Magento (no email in ERP?)");

                    continue;
                }
                $superUserByCustomerReference = $this->customerHelper->getSuperUserCustomerReference($customers, $company);
                if ($superUserByCustomerReference) {
                    $superUser = $superUserByCustomerReference;
                }
                $this->activateSuperUser($superUser);

                $companyDto = $this->getCompanyFromErpId($company['id']);
                if ($companyDto) {
                    $this->logger->info("--> Company {$company['name']} already in Magento");
                } else {
                    $companyDto = $this->companyFactory->create();
                }

                $companyDto = $this->createUpdateCompany($companyDto, $company, $superUser, $superUserByCustomerReference !== null);
                $superUserId = (int) $superUser->getId();
                $this->customerHelper->assignCustomers((int) $companyDto->getId(), $customers, $superUserId);
                $alternativeSkuNumbers = $this->arrayManager->get('articles', $company);
                if ($alternativeSkuNumbers && $companyDto->getId()) {
                    $this->productHelper->insertAlternativeSkus((int) $companyDto->getId(), $alternativeSkuNumbers);
                }
            }
        } catch (\Laminas\Http\Exception\RuntimeException $exception) {
            $errorMessage = $errorMessage.$exception->getMessage()."\n";
            $this->logger->error($errorMessage);
        } catch (\Magento\Framework\Exception\LocalizedException $exception) {
            $errorMessage = $errorMessage.$exception->getMessage()."\n";
            $this->logger->error($errorMessage);
        } catch (\RuntimeException $exception) {
            $errorMessage = $errorMessage.$exception->getMessage()."\n";
            $this->logger->error($errorMessage);
        }

        \Ecomero\ErpB2b\Rewrite\Magento\Company\Model\Email\Sender::setEnableNotifications(true);

        return $errorMessage;
    }

    private function getCompanyFromErpId(string $erpId): ?\Magento\Company\Api\Data\CompanyInterface
    {
        $filter = $this->searchCriteriaBuilder->addFilter('erp_company_no', $erpId)->create();
        $collection = $this->companyRepository->getList($filter);
        foreach ($collection->getItems() as $item) {
            return $item;
        }

        return null;
    }

    private function createUpdateCompany(
        \Magento\Company\Api\Data\CompanyInterface $companyDto,
        array $company,
        \Magento\Customer\Api\Data\CustomerInterface $superUser,
        $updateSuperUser = false
    ): \Magento\Company\Api\Data\CompanyInterface {
        if (null === $companyDto->getId()) {
            $companyDto->setCustomerGroupId(CustomerGroupHelper::CUSTOMER_GROUP_GENERAL);
            $companyDto->setSuperUserId((int)$superUser->getId());
        }
        if ($updateSuperUser && (int)$superUser->getId() !== (int) $companyDto->getSuperUserId()) {
            $companyDto->setSuperUserId((int)$superUser->getId());
        }

        $companyDto->setCompanyName($company['name']);
        $companyDto->setCompanyEmail($company['email'] ?: $superUser->getEmail());
        $companyDto->setStreet($company['street'] ?: ($company['deliveryStreet'] ?: '-'));
        $companyDto->setCity($company['city'] ?: ($company['deliveryCity'] ?: '-'));
        $companyDto->setCountryId($company['countryId'] ?: 'SE');
        $companyDto->setPostcode($company['postalCode'] ?: ($company['deliveryPostalCode'] ?: '-'));
        $companyDto->setTelephone($company['telephone'] ?: '-');
        $companyDto->setVatTaxId($company['orgNo']);
        $companyDto->setErpCompanyNo($company['id']);
        $companyDto->setErpPriceListId($company['priceList'].'_'.$company['agreement']);
        $terms = $this->arrayManager->get('terms', $company);
        if ($terms) {
            $companyDto->setErpTerms(json_encode($terms));
        }
        $salesPersonEmail = $this->arrayManager->get('salesPersonEmail', $company);
        if ($salesPersonEmail) {
            $salesRepresentative = $this->salesLoader->loadByUsername($salesPersonEmail);
            $userId = is_array($salesRepresentative) && isset($salesRepresentative['user_id']) ? $salesRepresentative['user_id']: 0;
            if ($userId) {
                $companyDto->setSalesRepresentativeId((int)$userId);
            }
        }

        try {
            $companyDto = $this->companyRepository->save($companyDto);
        } catch (\Magento\Framework\Exception\InputException $e) {
            $this->logger->error($e->getMessage());
            foreach ($e->getErrors() as $error) {
                $this->logger->error($error->getMessage());
            }
        }

        return $companyDto;
    }

    private function activateSuperUser(\Magento\Customer\Api\Data\CustomerInterface $superUser)
    {
        if (!$superUser->getExtensionAttributes()
            || !$superUser->getExtensionAttributes()->getCompanyAttributes()
        ) {
            return;
        }

        $status = (int) $superUser->getExtensionAttributes()->getCompanyAttributes()->getStatus();
        if (!$status) {
            $superUser->getExtensionAttributes()->getCompanyAttributes()->setStatus(1);
            $this->customerRepository->save($superUser);
        }
    }
}
