<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\ErpCore\Service;

use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp;
use Ecomero\ErpCore\Model\ResourceModel\Item\CollectionFactory;

use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product\Attribute\Source\Status;

use Magento\Catalog\Model\Product\Type;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Catalog\Model\ProductFactory;
use Magento\Catalog\Model\ResourceModel\Product;
use Magento\CatalogInventory\Api\StockRegistryInterface;
use Magento\ConfigurableProduct\Helper\Product\Options\Factory;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Framework\Exception\NoSuchEntityException;

use Magento\Store\Model\StoreManagerInterface;
use Symfony\Component\Console\Output\OutputInterface;

class ProductService
{
    protected $logger;
    protected $notification;
    protected $erp;
    protected $productFactory;
    protected $productRepository;
    protected $productResourceModel;
    protected $optionsFactory;
    protected $settings;
    protected $storeManager;
    protected $stockRegistry;
    protected $itemCollectionFactory;
    protected static $thisErrorHandler = null;

    public function __construct(
        ErpLogger $logger,
        Notification $notification,
        Erp $erp,
        ProductFactory $productFactory,
        ProductRepositoryInterface $productRepository,
        Product $productResourceModel,
        Factory $optionsFactory,
        Settings $settings,
        StoreManagerInterface $storeManager,
        StockRegistryInterface $stockRegistry,
        CollectionFactory $itemCollectionFactory
    ) {
        $this->logger = $logger;
        $this->notification = $notification;
        $this->erp = $erp;
        $this->productFactory = $productFactory;
        $this->productRepository = $productRepository;
        $this->productResourceModel = $productResourceModel;
        $this->optionsFactory = $optionsFactory;
        $this->settings = $settings;
        $this->storeManager = $storeManager;
        $this->stockRegistry = $stockRegistry;
        $this->itemCollectionFactory = $itemCollectionFactory;
    }

    private function initCommonProductAttributes($product, $item, $isNew) : bool
    {
        $isDirty = false;

        // Set attributes for all web sites
        $defaultStoreViewId = $this->storeManager->getDefaultStoreView()->getId();
        $product->setWebsiteIds($item->getWebsiteIds());
        $product->setStoreId($defaultStoreViewId);

        // Only set attribute, product category and metadata on new product.
        // It is possible that this data has been updated manually in Magento for existing products.
        if ($isNew) {
            $product->setAttributeSetId($item->attributeSet);
            $product->setCategoryIds($item->getCategoryIds());
            $itemName = $item->getName();
            if ($product->getTypeId() === Configurable::TYPE_CODE) {
                $itemName = $item->description;
            }
            $product->setMetaTitle($itemName);
            $product->setMetaKeyword($itemName);
            $product->setMetaDescription($itemName);
            $product->setName($itemName);
            $product->setPrice($item->getPrice());
            $isDirty = true;
        }

        if (((float)$product->getCost()) !== ((float)$item->cost)) {
            $product->setCost($item->cost);
            $isDirty = true;
        }

        if (((float)$product->getWeight()) !== ((float)$item->weight)) {
            $product->setWeight($item->weight);
            $isDirty = true;
        }

        return $isDirty;
    }

    public function getErpItemInfoFromSKU($sku) : array
    {
        $itemInfo = [];

        try {
            $product = $this->productRepository->get($sku);
            $bcAttrib = $product->getCustomAttribute('erp_id');
            if ($bcAttrib) {
                $itemInfo['erp_id'] = $bcAttrib->getValue();
            } else {
                throw new \RuntimeException('Error can not get Erp Article/Item Id from Magento ' .
                                            'product, is the attribute missing on the sku? (' . $sku . ')');
            }

            $bcAttrib = $product->getCustomAttribute('erp_tax_rate');
            if ($bcAttrib) {
                $itemInfo['erp_tax_rate'] = $bcAttrib->getValue();
            } else {
                throw new \RuntimeException('Error can not get Erp Article/Item VAT Rate from Magento' .
                                            ' product, is the attribute missing on the sku? (' . $sku . ')');
            }

            $bcAttrib = $product->getCustomAttribute('erp_tax_included');
            if ($bcAttrib) {
                $itemInfo['erp_tax_included'] = $bcAttrib->getValue();
            } else {
                throw new \RuntimeException('Error can not get Erp Article/Item VAT Included from Magento'
                . ' product, is the attribute missing on the sku? (' . $sku . ')');
            }

            return $itemInfo;
        } catch (NoSuchEntityException $e) {
            throw new \RuntimeException('Error can not get (or find) sku ' . $sku . ' in Magento');
        }
    }

    private function storeProductWithURLCheck(
        \Magento\Catalog\Model\Product $product,
        \Ecomero\ErpCore\Model\Item $item
    ) : void {
        try {
            $this->productRepository->save($product);
        } catch (\Magento\Framework\Exception\LocalizedException $exception) {
            // If two product in the same website ends up with the same URL key
            // then append the SKU and try to save the product again.
            if (strpos($exception->getMessage(), 'URL key') !== false) {
                $url = preg_replace('#[^0-9a-z]+#i', '-', $item->getName() . '-' . $product->getSku());
                $this->logger->debug('    ...url key exists, generating new key : ' . $url);
                $product->setUrlKey($url);
                $this->productRepository->save($product);
            } else {
                throw $exception;
            }
        }
    }

    private function createOrUpdateSimpleProducts(object $itemCollection) : void
    {
        $currentProduct = 0;
        $maxProduct = $itemCollection->count();
        $stores = $this->storeManager->getStores(false, true);

        foreach ($itemCollection->getItems() as $item) {
            $currentProduct ++;

            $skuDescription =   $item->sku . ', ' .
                                $item->getName() . ', ' .
                                $item->getPrice() . ', ' .
                                $item->attributes['erp_id']->value;
            $stockItem = null;
            try {
                $product = $this->productRepository->get($item->sku);
                $stockItem = $this->stockRegistry->getStockItemBySku($item->sku);
                $isNew = false;
                $this->logger->debug("  - " . $currentProduct . "/" . $maxProduct .
                                     " [" . $item->getCategoryLabel() . "]" .
                                     " Product already imported, updating (" . $skuDescription . ")");
            } catch (NoSuchEntityException $e) {
                $product = $this->productFactory->create();
                $product->setSku($item->sku);
                $product->setTypeId(Type::TYPE_SIMPLE);

                if ($item->isSimpleProduct()) {
                    $product->setVisibility(Visibility::VISIBILITY_BOTH);
                } else {
                    $product->setVisibility(Visibility::VISIBILITY_NOT_VISIBLE);
                }
                $product->setStatus(Status::STATUS_DISABLED);
                $isNew = true;
                $this->logger->debug("  - " . $currentProduct . "/" . $maxProduct .
                                     " [" . $item->getCategoryLabel() . "]" .
                                     " New product (" . $skuDescription . ")");
            }

            $isDirty = $this->initCommonProductAttributes($product, $item, $isNew);

            if ($stockItem == null || $stockItem->getQty() != $item->inventory) {
                $product->setStockData(['use_config_manage_stock' => 1,
                                        'qty' => $item->inventory, 'is_qty_decimal' => 0, 'is_in_stock' => 1]);
                $isDirty = true;
            }

            if ($isDirty) {
                $this->logger->debug("    ...found changes, saving product");
                $this->storeProductWithURLCheck($product, $item);
            }

            $product = $this->productRepository->get($item->sku);
            $item->magentoId = $product->getId();
            $item->associatedProductIds = [];

            foreach ($item->attributes as $attrib) {
                if ($attrib->assignAttributeToAttributeSet($product, 'Product Details')) {
                    $attrib->setAttribute($product);
                }
            }

            // Set product price for each website
            $lcy = strtolower($this->settings->getLCY());
            foreach ($stores as $storeName => $store) {
                $storeCurrency = $this->settings->getCurrency((int)$store->getId());
                $currency = strtolower($storeCurrency);
                if ($currency == "" || $currency == $lcy) {
                    $product->setPrice($item->getPrice());
                    $product->setStoreId($store->getId());
                    $this->productResourceModel->saveAttribute((Object)$product, 'price');
                } elseif (array_key_exists($currency, $item->prices)) {
                    $product->setPrice($item->prices[$currency]);
                    $product->setStoreId($store->getId());
                    $this->productResourceModel->saveAttribute((Object)$product, 'price');
                } else {
                    $this->logger->error("    ...no price in " . $currency . " found on item in Erp");
                }
            }
        }
    }

    private function createOrUpdateConfigurableProduct(object $item) : void
    {
        $configuarble_sku = $item->getCommmonName();

        if ($item->associatedProductIds == null) {
            $this->logger->error("  - No associated products found! (" . $configuarble_sku . ")");
            return;
        }

        if (count($item->associatedProductIds) == 1) {
            $this->logger->debug("  - Single product found '" . $item->getCommmonName() .
                                 "', do not create a matching configurable product");
            return;
        }

        try {
            $product = $this->productRepository->get($configuarble_sku);
            $isNew = false;
            $this->logger->debug("  - Configurable product already exists, updating (" . $configuarble_sku . ")");
        } catch (NoSuchEntityException $e) {
            $product = $this->productFactory->create();
            $product->setSku($configuarble_sku);
            $product->setTypeId(Configurable::TYPE_CODE);
            $product->setVisibility(Visibility::VISIBILITY_BOTH);
            $product->setStatus(Status::STATUS_ENABLED);
            $isNew = true;
            $this->logger->debug("  - New configurable product must be created (" . $configuarble_sku . ")");
        }

        $isDirty = $this->initCommonProductAttributes($product, $item, $isNew);

        if ($isDirty) {
            $this->logger->debug("    ...found changes, saving product");

            $product->setStockData(['use_config_manage_stock' => 1,
                                    'manage_stock' => 1,
                                    'is_qty_decimal' => 0,
                                    'is_in_stock' => 1]);

            // Super attribute
            $usedProductAttribArray = [];
            foreach ($item->attributes as $key => $value) {
                if ($value->usedInConfiguration === \Ecomero\ErpCore\Model\Attribute::YES) {
                    $attrib = $product->getResource()->getAttribute($key);
                    if ($attrib) {
                        $usedProductAttribArray[] = $attrib->getId();
                    }
                }
            }
            $product->getTypeInstance()->setUsedProductAttributeIds($usedProductAttribArray, $product);
            $configurableAttributesData = $product->getTypeInstance()->getConfigurableAttributesAsArray($product);
            $product->setCanSaveConfigurableAttributes(true);
            $product->setConfigurableAttributesData($configurableAttributesData);
            $configurableProductsData = [];
            $product->setConfigurableProductsData($configurableProductsData);

            $this->storeProductWithURLCheck($product, $item);
        }

        // Link simple products
        $usedProducts = $product->getTypeInstance()->getUsedProducts($product);
        $existingLinks = [];
        foreach ($usedProducts as $child) {
            $existingLinks[] = $child->getId();
        }

        // Only update if links are changed
        array_multisort($existingLinks);
        array_multisort($item->associatedProductIds);
        if (serialize($existingLinks) !== serialize($item->associatedProductIds)) {

            // Add missing products
            foreach ($usedProducts as $child) {
                if (!in_array($child->getId(), $item->associatedProductIds)) {
                    $item->associatedProductIds[] = $child->getId();
                }
            }

            $this->logger->debug("    ...found link changes, saving product");
            $product = $this->productRepository->get($configuarble_sku);
            $product->setAssociatedProductIds($item->associatedProductIds);
            $product->setCanSaveConfigurableAttributes(true);
            $this->storeProductWithURLCheck($product, $item);
        }
    }

    private function getUniqueProducts(object $itemCollection) : array
    {
        $uniqueProdArray = [];

        foreach ($itemCollection->getItems() as $item) {
            if ($item->getCommmonName() == "") {
                continue;
            }
            if (array_key_exists($item->getCommmonName(), $uniqueProdArray) == false) {
                $uniqueProdArray[$item->getCommmonName()] = $item;
            }
            if ($item->magentoId != null) {
                $uniqueProdArray[$item->getCommmonName()]->associatedProductIds[] = $item->magentoId;
            }
        }
        return $uniqueProdArray;
    }

    private function createProducts(object $itemCollection) : void
    {
        if ($itemCollection->count() > 0) {
            $this->createOrUpdateSimpleProducts($itemCollection);

            $uniqueProducts = $this->getUniqueProducts($itemCollection);
            foreach ($uniqueProducts as $item) {
                $this->createOrUpdateConfigurableProduct($item);
            }
        }
    }

    public function importCommandLine(OutputInterface $output, bool $forceUnlock) : void
    {
        $this->logger->info('Starting product import from command line');
        $this->logger->setOutput($output);
        $this->import($forceUnlock);
        set_error_handler(null);
    }

    public function importCron(bool $forceUnlock = false) : void
    {
        $this->logger->info('Starting product import from cron job');
        $this->import($forceUnlock);
        set_error_handler(null);
    }

    public function importWeb(bool $forceUnlock = false) : void
    {
        $this->logger->info('Starting product import from web');
        $this->import($forceUnlock);
        set_error_handler(null);
    }

    public static function erpErrorHandler($errno, $errstr, $errfile, $errline)
    {
        if (self::$thisErrorHandler) {
            self::$thisErrorHandler->logger->error($errstr);
        }

        return false;
    }

    private function import(bool $forceUnlock = false) : void
    {
        self::$thisErrorHandler = $this;
        set_error_handler('Ecomero\ErpCore\Service\ProductService::erpErrorHandler');

        if ($forceUnlock) {
            $this->settings->releaseCronLock();
        }

        if ($this->settings->requestCronLock($this->logger) == false) {
            return;
        }

        $capabilities = $this->erp->getCapabilities();
        foreach ($capabilities as $capability) {
            if (!$capability->isCapabilitySupported(Capability::PRODUCT_IMPORT)) {
                $this->logger->warning('Import of products are not supported by the ' . $capability->getName() . ' integration');
                $this->settings->releaseCronLock();
                return;
            }
        }

        $start_time = microtime(true);

        $errorMessage = '';
        try {
            $allWebsites = $this->storeManager->getWebsites(false, true);
            $itemCollection = $this->itemCollectionFactory->create();
            foreach ($allWebsites as $website) {
                $websiteId = (int)$website->getId();
                $websiteName = $this->storeManager->getWebsite($websiteId)->getName();

                $itemCategories = $this->settings->getCategoryMapping($websiteId);
                if ($itemCategories === null) {
                    continue;
                }
                $this->erp->setWebsite($websiteId);
                foreach ($itemCategories as $category) {
                    $categoryFilter = $category['category_filter'];
                    $categoryId = $category['product_category'];
                    $attribSetId = $category['attribute_set'];

                    $this->logger->info('Retrieving products for category "' . $categoryFilter .
                                        '" (Website : ' . $websiteName . ')');

                    $json = $this->erp->getItemList($categoryFilter, $forceUnlock);
                    $itemCollection->initItems($json, $websiteId, $categoryId, $categoryFilter, $attribSetId);

                    $json = $this->erp->getItemPrices($categoryFilter);
                    $itemCollection->initItemsPrices($json);

                    $json = $this->erp->getItemAttributes($categoryFilter);
                    $itemCollection->initItemsAttributes($json);
                }
                $this->erp->setItemImportCompleted();
            }
            $this->erp->createCommonName($itemCollection);
            $this->createProducts($itemCollection);
        } catch (\Zend\Http\Exception\RuntimeException $exception) {
            $errorMessage = $errorMessage . $exception->getMessage();
            $this->logger->error($exception->getMessage());
        } catch (\Magento\Framework\Exception\LocalizedException $exception) {
            $errorMessage = $errorMessage . $exception->getMessage();
            $this->logger->error($exception->getMessage());
        } catch (\RuntimeException $exception) {
            $errorMessage = $errorMessage . $exception->getMessage();
            $this->logger->error($exception->getMessage());
        }

        if ($errorMessage !== '' && $this->settings->stopOnError()) {
            $this->notification->notify($errorMessage);
            return;
        }
        $this->logger->info('Import completed in ' . (microtime(true) - $start_time) . ' sec');
        $this->settings->releaseCronLock();

        if ($errorMessage !== '') {
            $this->notification->notify($errorMessage);
        }
    }
}
