<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\ErpCore\Service;

use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\ErpAdapterInterface;

use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;

use Magento\Store\Model\StoreManagerInterface;
use Symfony\Component\Console\Output\OutputInterface;

class ImageService
{
    protected $logger;
    protected $erp;
    protected $settings;
    protected $productService;
    protected $productCollectionFactory;
    protected $mediaUrl;

    public function __construct(
        ErpLogger $logger,
        ErpAdapterInterface $erp,
        Settings $settings,
        ProductService $productService,
        CollectionFactory $productCollectionFactory,
        StoreManagerInterface $storeManager
    ) {
        $this->logger = $logger;
        $this->erp = $erp;
        $this->settings = $settings;
        $this->productService = $productService;
        $this->productCollectionFactory = $productCollectionFactory;

        /** @var $defaultStore \Magento\Store\Model\Store */
        $defaultStore = $storeManager->getStore(1);
        $this->mediaUrl = $defaultStore->getBaseUrl(\Magento\Framework\UrlInterface::URL_TYPE_MEDIA);
    }

    public function exportCommandLine(OutputInterface $output, bool $forceUnlock) : void
    {
        $this->logger->info('Starting product image export from command line');
        $this->logger->setOutput($output);
        $this->export($forceUnlock);
    }

    public function exportWeb(bool $forceUnlock = false) : void
    {
        $this->logger->info('Starting product image export from web');
        $this->export($forceUnlock);
    }

    private function export(bool $forceUnlock = false) : void
    {
        if ($forceUnlock) {
            $this->settings->releaseCronLock();
        }

        if ($this->settings->requestCronLock($this->logger) == false) {
            return;
        }

        $capabilities = $this->erp->getCapabilities();
        foreach ($capabilities as $capability) {
            if (!$capability->isCapabilitySupported(Capability::PRODUCT_PICTURE_EXPORT)) {
                $this->logger->warning('Export of product images is not supported by the ' . $capability->getName() . ' integration');
                $this->settings->releaseCronLock();
                return;
            }
        }

        try {
            $this->logger->debug('Exporting images');

            $productCollection = $this->productCollectionFactory->create();
            $productCollection->addAttributeToSelect('*');
            $productCollection->addAttributeToFilter('erp_id', ['like' => '%-%-%-%-%']);

            foreach ($productCollection as $product) {
                $bcAttrib = $product->getCustomAttribute('erp_id');
                $itemId = $bcAttrib->getValue();
                $imageUrl = $this->mediaUrl . "catalog/product" . $product->getImage();
                $rc = $this->erp->addPicture($itemId, $imageUrl);
                if ($rc) {
                    $this->logger->debug('  - Exporting image ' . $imageUrl .
                                         ' for "' . $product->getName() .
                                         '" (' . $itemId . ') status code ' . $rc);
                } else {
                    $this->logger->debug('  - Failed exporting image ' . $imageUrl .
                                         ' for "' . $product->getName() .
                                         '" (' . $itemId . ')');
                }
            }
        } catch (\Zend\Http\Exception\RuntimeException $exception) {
            $this->logger->error($exception->getMessage());
        } catch (\Magento\Framework\Exception\LocalizedException $exception) {
            $this->logger->error($exception->getMessage());
        }
        $this->logger->info('Export completed');
        $this->settings->releaseCronLock();
    }
}
