<?php declare(strict_types=1);

/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */

namespace Ecomero\ErpCore\Service;

use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Helper\TaxItemHelper;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\ErpAdapterInterface;

use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Sales\Api\CreditmemoRepositoryInterface;
use Magento\Sales\Model\ResourceModel\Order\Creditmemo\CollectionFactory;
use Symfony\Component\Console\Output\OutputInterface;

class ExportCreditMemoService
{
    protected $logger;
    protected $notification;
    protected $erp;
    protected $settings;
    protected $creditMemoCollectionFactory;
    protected $creditMemoRepository;
    protected $searchCriteriaBuilder;
    protected $productService;
    protected $taxItemHelper;

    public function __construct(
        ErpLogger $logger,
        Notification $notification,
        ErpAdapterInterface $erp,
        Settings $settings,
        CollectionFactory $creditMemoCollectionFactory,
        CreditmemoRepositoryInterface $creditMemoRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        ProductService $productService,
        TaxItemHelper $taxItemHelper
    ) {
        $this->logger = $logger;
        $this->notification = $notification;
        $this->erp = $erp;
        $this->settings = $settings;
        $this->creditMemoCollectionFactory = $creditMemoCollectionFactory;
        $this->creditMemoRepository = $creditMemoRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->productService = $productService;
        $this->taxItemHelper = $taxItemHelper;
    }

    public function exportCommandLine(OutputInterface $output, bool $forceUnlock) : void
    {
        $this->logger->info('Starting credit memo export from command line');
        $this->logger->setOutput($output);
        $this->export($forceUnlock);
    }

    public function exportCron(bool $forceUnlock = false) : void
    {
        $this->logger->info('Starting credit memo export from cron job');
        $this->export($forceUnlock);
    }

    public function exportWeb(bool $forceUnlock = false) : void
    {
        $this->logger->info('Starting credit memo export from web');
        $this->export($forceUnlock);
    }

    public function addCreditLines(object $creditMemoDetails, object $creditMemoResponse, string $creditMemoId) : int
    {
        $numOrderLines = 0;
        foreach ($creditMemoDetails->getAllItems() as $item) {
            if ($item->getOrderItem()->getParentItem()) {
                continue;
            }

            $bcItem = $this->productService->getErpItemInfoFromSKU($item->getSku());
            if ($bcItem['erp_id']) {
                $vatCode = $this->taxItemHelper->getTaxCodeForOrderLine(
                    $creditMemoId,
                    $item->getOrderItem()->getItemId()
                );
                $bcDefaultTaxRate = ((float)$bcItem['erp_tax_rate'] / 100);
                $itemPrice = $item->getPrice();
                if ($bcItem['erp_tax_included'] == '1') {
                    $itemPrice = $itemPrice + ($item->getPrice() * $bcDefaultTaxRate);
                }

                $this->logger->debug('  - Processing credit memo ' . $creditMemoResponse->number .
                                     ' (' . $creditMemoResponse->id . '), line item ' . $item->getName() .
                                     ', Price : ' . $item->getPriceInclTax() .
                                     ', Qty : ' . $item->getQtyOrdered() .
                                     ', VAT: ' . $vatCode);
                $rc = $this->erp->addCreditMemoItem(
                    $creditMemoResponse->id,
                    $bcItem['erp_id'],
                    (float)$itemPrice,
                    (float)$item->getQty(),
                    $vatCode,
                    (float)$item->getDiscountAmount(),
                    (float)$item->getDiscountPercent()
                );
                $numOrderLines ++;
            }
        }
        return $numOrderLines;
    }

    public function creditShippingCharges(object $creditMemoDetails, object $creditMemoResponse, int $numOrderLines) : void
    {
        $order = $creditMemoDetails->getOrder();
        $orderNo = $order['entity_id'];
        $vatCode = '';
        if ($this->settings->getUseShippingTaxFromOrder()) {
            $vatCode = $this->taxItemHelper->getTaxCodeForFirstProduct($orderNo);
        } else {
            $vatCode = $this->taxItemHelper->getShippingTaxCodeForOrder($orderNo);
        }

        $magentoShippingMethod = $order->getShippingDescription();
        $shippingAgentService = $this->settings->getShippingChargeItem($this->erp->getWebsite(), $magentoShippingMethod);
        $shippingAgentServiceArray = explode('#', $shippingAgentService);
        $shipmentItemId = '';
        $shipmentItemType = '';
        $shipmentItemTaxIncluded = false;
        if (count($shippingAgentServiceArray) === 4) {
            $shipmentItemId = $shippingAgentServiceArray[0];
            $shipmentItemType = $shippingAgentServiceArray[1];
            $shipmentItemTaxIncluded = (bool)($shippingAgentServiceArray[2] === 'true');
            // NOT USED - $shipmentItemTaxRate = $shippingAgentServiceArray[3];

            $shippingPrice = $creditMemoDetails->getBaseShippingAmount();
            if ($shipmentItemTaxIncluded) {
                $shippingPrice = $shippingPrice + $creditMemoDetails->getBaseShippingTaxAmount();
            }

            if ($shipmentItemType === 'charge') {
                $this->erp->addCreditMemoCharge(
                    $creditMemoResponse->id,
                    $shipmentItemId,
                    (float)$shippingPrice,
                    (float)1,
                    $vatCode,
                    $creditMemoResponse->number,
                    (int)$numOrderLines
                );
            } else {
                $this->erp->addCreditMemoItem(
                    $creditMemoResponse->id,
                    $shipmentItemId,
                    (float)$shippingPrice,
                    (float)1,
                    $vatCode,
                    (float)0,
                    (float)0
                );
            }
        } else {
            $this->logger->debug('  - *** No Shipping Agent Mapping found for ' . $magentoShippingMethod);
        }
    }

    public function export(bool $forceUnlock = false) : void
    {
        if ($forceUnlock) {
            $this->settings->releaseCronLock();
        }

        if ($this->settings->requestCronLock($this->logger) == false) {
            return;
        }

        $capabilities = $this->erp->getCapabilities();
        foreach ($capabilities as $capability) {
            if (!$capability->isCapabilitySupported(Capability::CREDIT_MEMO_EXPORT)) {
                $this->logger->warning('Export of credit memos is not supported by the ' . $capability->getName() . ' integration');
                $this->settings->releaseCronLock();
                return;
            }
        }

        $start_time = microtime(true);
        try {
            $creditMemoCollecion = $this->creditMemoCollectionFactory->create();

            $creditMemoCollecion->getSelect()
                ->joinLeft(
                    ['order' => $creditMemoCollecion->getTable('sales_order')],
                    'main_table.order_id = order.entity_id',
                    [   'erp_customer_no' => 'order.erp_customer_no',
                        'order_number' => 'order.increment_id']
                )
                ->joinLeft(
                    ['invoice' => $creditMemoCollecion->getTable('sales_invoice')],
                    'main_table.order_id = invoice.order_id',
                    [   'erp_invoice_no' => 'invoice.erp_invoice_no']
                )
                ->where('order.erp_customer_no IS NOT NULL AND ' .
                        'main_table.erp_creditmemo_no IS NULL');

            $creditMemoCollecion
                ->addFieldToSelect('entity_id')
                ->addFieldToSelect('created_at')
                ->addFieldToSelect('increment_id')
                ->addFieldToSelect('order_id');

            foreach ($creditMemoCollecion->getData() as $creditMemo) {
                $creditMemoDetails = $this->creditMemoRepository->get($creditMemo['entity_id']);

                $storeId = (int)$creditMemoDetails->getStoreId();
                $this->erp->setWebsiteFromStoredId($storeId);
                $this->taxItemHelper->setDefaultTaxCode($this->settings->getDefaultTaxCode($this->erp->getWebsite()));
                $activeErp = $this->erp->isErpEnabled();
                if ($activeErp === '') {
                    $this->logger->debug('Processing Magento credit memo ' . $creditMemo['increment_id'] . ' no ERP integration active for website');
                    continue;
                }

                $this->logger->debug('Processing Magento credit memo ' . $creditMemo['increment_id'] . ' for ' . $activeErp . ' integration');
                $bcCreditMemo = $this->erp->createCreditMemo(
                    substr($creditMemo['created_at'], 0, 10),
                    $creditMemo['order_number'],
                    $creditMemo['erp_invoice_no'],
                    $creditMemo['erp_customer_no']
                );

                $numOrderLines = $this->addCreditLines($creditMemoDetails, $bcCreditMemo, $creditMemo['entity_id']);

                // TODO: Option for posting?
                // $this->erp->postCreditMemo($bcCreditMemo->id);
                $this->creditShippingCharges($creditMemoDetails, $bcCreditMemo, $numOrderLines);

                /** @var $creditMemoDetails \Magento\Sales\Api\Data\CreditmemoInterface */
                $creditMemoDetails = $this->creditMemoRepository->get($creditMemo['entity_id']);
                $creditMemoDetails->setErpCreditmemoNo($bcCreditMemo->number);
                $this->creditMemoRepository->save($creditMemoDetails);
            }
        } catch (\Zend\Http\Exception\RuntimeException $exception) {
            $this->notification->notify($exception->getMessage());
            $this->logger->error($exception->getMessage());
        } catch (\Magento\Framework\Exception\LocalizedException $exception) {
            $this->notification->notify($exception->getMessage());
            $this->logger->error($exception->getMessage());
        } catch (\RuntimeException $exception) {
            $this->notification->notify($exception->getMessage());
            $this->logger->error($exception->getMessage());
        }
        $this->logger->info('Export completed in ' . (microtime(true) - $start_time) . ' sec');
        $this->settings->releaseCronLock();
    }
}
