<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\ErpCore\Service;

use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\ErpAdapterInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory;

abstract class OrderServiceAbstract
{
    protected $logger;
    protected $notification;
    protected $settings;
    protected $erp;
    protected $orderCollectionFactory;
    protected $orderRepository;
    protected $customerService;
    protected static $thisErrorHandler = null;

    public function __construct(
        ErpLogger $logger,
        Notification $notification,
        Settings $settings,
        ErpAdapterInterface $erp,
        CustomerService $customerService,
        CollectionFactory $orderCollectionFactory,
        OrderRepositoryInterface $orderRepository
    ) {
        $this->logger = $logger;
        $this->notification = $notification;
        $this->settings = $settings;
        $this->erp = $erp;
        $this->orderCollectionFactory = $orderCollectionFactory;
        $this->orderRepository = $orderRepository;
        $this->customerService = $customerService;
    }

    public function executeFromShell(\Symfony\Component\Console\Output\OutputInterface $output, bool $forceUnlock) : void
    {
        $this->logger->info('Starting ' . $this->getServiceDescription() . ' from command line');
        $this->logger->setOutput($output);
        $this->execute($forceUnlock);
        set_error_handler(null);
    }

    public function executeFromCron(bool $forceUnlock = false) : void
    {
        $this->logger->info('Starting ' . $this->getServiceDescription() . ' from cron job');
        $this->execute($forceUnlock);
        set_error_handler(null);
    }

    public function executeFromWeb(bool $forceUnlock = false) : void
    {
        $this->logger->info('Starting ' . $this->getServiceDescription() . ' from web');
        $this->execute($forceUnlock);
        set_error_handler(null);
    }

    private function handleException($order, $exception) : string
    {
        $errorMessage = $exception->getMessage();

        $this->logger->error($errorMessage);

        $orderStatusHistories = $order->getStatusHistories();
        $orderStatusHistory = reset($orderStatusHistories);
        if ($orderStatusHistory !== false &&
            $orderStatusHistory->getComment() === $errorMessage) {
            $orderStatusHistory->setCreatedAt(new \DateTime());
            $errorMessage = ''; // Prevent error to be sent as a notification again
        } else {
            $order->addStatusHistoryComment($errorMessage);
        }

        $order->setErpOrderNo('ERROR see history log');

        // Change status (not the state) on the order so it is not resent until manually handled
        if (substr($errorMessage, 0, 5) === "ERROR") {
            $orderStatusMarkSent = $this->settings->getOrderStatusSent();
            $order->setStatus($orderStatusMarkSent);
        }

        $order->save();

        return $errorMessage;
    }

    public static function erpErrorHandler($errno, $errstr, $errfile, $errline)
    {
        if (self::$thisErrorHandler) {
            self::$thisErrorHandler->logger->error($errstr);
        }

        return false;
    }

    public function execute(bool $forceUnlock = false) : void
    {
        self::$thisErrorHandler = $this;
        set_error_handler('Ecomero\ErpCore\Service\OrderServiceAbstract::erpErrorHandler');

        if ($forceUnlock) {
            $this->settings->releaseCronLock();
        }

        if ($this->settings->requestCronLock($this->logger) == false) {
            return;
        }

        $capabilities = $this->erp->getCapabilities();
        foreach ($capabilities as $capability) {
            if (!$capability->isCapabilitySupported($this->getCapability())) {
                $this->logger->warning('This operation is not supported by the ' . $capability->getName() . ' integration');
                $this->settings->releaseCronLock();
                return;
            }
        }

        $start_time = microtime(true);
        $orderCollecion = $this->orderCollectionFactory->create()
                                                        ->addFieldToSelect('entity_id')
                                                        ->addFieldToSelect('increment_id')
                                                        ->addFieldToSelect('customer_email')
                                                        ->addFieldToSelect('order_currency_code');

        $orderStatusMarkSent = $this->getOrderStatusFilter();

        $errorMessage = '';
        $orderCollecion->addAttributeToFilter('status', ['eq'=> $orderStatusMarkSent ]);
        foreach ($orderCollecion->getData() as $order) {
            try {
                $fullOrderObj = $this->orderRepository->get($order['entity_id']);
                $storeId = (int)$fullOrderObj->getStoreId();
                $this->erp->setWebsiteFromStoredId($storeId);
                $this->taxItemHelper->setDefaultTaxCode($this->settings->getDefaultTaxCode($this->erp->getWebsite()));
                $activeErp = $this->erp->isErpEnabled();
                if ($activeErp !== '') {
                    $this->logger->debug('Processing Magento order ' . $order['increment_id'] . ' for ' . $activeErp . ' integration');
                    $this->processorder($fullOrderObj);
                } else {
                    $this->logger->debug('Processing Magento order ' . $order['increment_id'] . ' no ERP integration active for website');
                }
            } catch (\Zend\Http\Exception\RuntimeException $exception) {
                $errorMessage = $errorMessage . $this->handleException($fullOrderObj, $exception);
            } catch (\Magento\Framework\Exception\LocalizedException $exception) {
                $errorMessage = $errorMessage . $this->handleException($fullOrderObj, $exception);
            } catch (\RuntimeException $exception) {
                $errorMessage = $errorMessage . $this->handleException($fullOrderObj, $exception);
            }

            if ($errorMessage !== '' && $this->settings->stopOnError()) {
                $this->notification->notify($errorMessage);
                return;
            }
        }
        $this->logger->info('Process completed in ' . (microtime(true) - $start_time) . ' sec');
        $this->settings->releaseCronLock();

        if ($errorMessage !== '') {
            $this->notification->notify($errorMessage);
        }
    }

    protected function isDocumentAlreadyInErp(string $magentoOrderId) : bool
    {
        $rc = $this->erp->getDocumentFromExternalId(
            $this->getDocumentType(),
            'PENDING(' . $magentoOrderId . ')'
        );
        if (property_exists($rc, "errorMessage")) {
            if ($rc->errorMessage !== null) {
                $this->logger->error($rc->errorMessage);
                return true;
            }
        }
        if (property_exists($rc, "number")) {
            $this->logger->error($this->getDocumentType() . " with Magento id " . $magentoOrderId . " already exists in Erp.");
            return true;
        }
        return false;
    }

    protected function createNewDocument(object $order, string $orderCurrency) : object
    {
        $billingAddress = $order->getBillingAddress()->getData();
        $bcCustomerId = $this->customerService->getCustomerId(
            $this->logger,
            $billingAddress['company'],
            $billingAddress['firstname'],
            $billingAddress['lastname'],
            $billingAddress['street'],
            $billingAddress['city'],
            $billingAddress['postcode'],
            $billingAddress['email'],
            $billingAddress['telephone'],
            $billingAddress['country_id'],
            $billingAddress['region']
        );

        $magentoShippingDescription = $order->getShippingDescription();
        $shippingAgentService = $this->settings->getShippingAgentService($this->erp->getWebsite(), $magentoShippingDescription);
        $shippingAgentServiceArray = explode('-', $shippingAgentService);
        $shippingAgent = '';
        $shippingService = '';
        if (count($shippingAgentServiceArray) == 2) {
            $shippingAgent = $shippingAgentServiceArray[0];
            $shippingService = $shippingAgentServiceArray[1];
        } else {
            $this->logger->debug('  - *** No Shipping Agent Mapping found for ' . $magentoShippingDescription);
        }

        $magentoShippingMethod = $order->getShippingMethod(true);
        $carrier = $magentoShippingMethod->getCarrierCode();
        $method = str_replace('shipping_', '', $magentoShippingMethod->getMethod());
        $wayOfDelivery = $this->erp->getWayOfDelivery($carrier, $method);

        $shippingAddress = $order->getShippingAddress()->getData();
        $shippingPickupLocationId = $order->getKssPickupLocationId(); // Klarna Shipping Service, field may not exist for non Klarna integrations

        if (isset($shippingPickupLocationId) && $shippingPickupLocationId !== '') {

            // Workaround : When pickup location is used by Klarna,
            // the shippingAddress is set to the address of the pickup location (this is not correct)
            $shippingAddress['company'] = $billingAddress['company'];
            $shippingAddress['firstname'] = $billingAddress['firstname'];
            $shippingAddress['lastname'] = $billingAddress['lastname'];
            $shippingAddress['telephone'] = $billingAddress['telephone'];
            $shippingAddress['fax'] = $billingAddress['fax'];
            $shippingAddress['street'] = $billingAddress['street'];
            $shippingAddress['city'] = $billingAddress['city'];
            $shippingAddress['postcode'] = $billingAddress['postcode'];
            $shippingAddress['country_id'] = $billingAddress['country_id'];
            $shippingAddress['region'] =  $billingAddress['region'];
        }

        $orderResponse = $this->erp->createDocument(
            $this->getDocumentType(),
            $bcCustomerId,
            $orderCurrency,
            $billingAddress,
            $shippingAddress,
            $shippingAgent,
            $shippingService,
            $shippingPickupLocationId,
            'PENDING(' . $order['increment_id'] . ')',
            $this->settings->getSalesPerson($this->erp->getWebsite()),
            $this->settings->getProfitCenter($this->erp->getWebsite()),
            $this->settings->getTermsOfDelivery($this->erp->getWebsite()),
            $wayOfDelivery,
            $this->settings->getCustomerCategory($this->erp->getWebsite()),
            $this->settings->getCustomerDistrict($this->erp->getWebsite())
        );

        return $orderResponse;
    }

    protected function addDocumentLines(object $orderDetails, object $orderResponse, string $orderNo) : int
    {
        $numOrderLines = 0;
        foreach ($orderDetails->getAllVisibleItems() as $item) {
            $bcItem = $this->productService->getErpItemInfoFromSKU($item->getSku());
            if ($bcItem['erp_id']) {
                $vatCode = $this->taxItemHelper->getTaxCodeForOrderLine($orderNo, $item->getItemId());
                $bcDefaultTaxRate = ((float)$bcItem['erp_tax_rate'] / 100);
                $itemPrice = (float)$item->getPrice();
                if ($bcItem['erp_tax_included'] == '1') {
                    $itemPrice = $itemPrice + ($item->getPrice() * $bcDefaultTaxRate);
                }

                $this->logger->debug('  - Processing line for document ' . $orderResponse->number .
                                     ' (' . $orderResponse->id . '), line item ' . $item->getName() .
                                     ', Price : ' . $item->getPriceInclTax() .
                                     ', Qty : ' . $item->getQtyOrdered() .
                                     ', VAT: ' . $vatCode);

                $itemComment = '';
                if ($options = $item->getData('product_options')) {
                    if (isset($options['options'])) {
                        foreach ($options['options'] as $optionValues) {
                            if ($optionValues['value']) {
                                if ($itemComment !== '') {
                                    $itemComment .= "\n";
                                }
                                $itemComment .= $optionValues['label'] . "\n" . $optionValues['value'];
                            }
                        }
                    }
                }

                $this->erp->addDocumentItem(
                    $this->getDocumentType(),
                    $orderResponse->id,
                    $bcItem['erp_id'],
                    (float)$itemPrice,
                    (float)$item->getQtyOrdered(),
                    $vatCode,
                    (float)$item->getDiscountAmount(),
                    (float)$item->getDiscountPercent(),
                    $itemComment
                );
                $numOrderLines ++;
            }
        }
        return $numOrderLines;
    }

    protected function addShippingCharges(object $order, object $orderResponse, string $orderNo, int $numOrderLines) : void
    {
        $vatCode = '';
        if ($this->settings->getUseShippingTaxFromOrder()) {
            $vatCode = $this->taxItemHelper->getTaxCodeForFirstProduct($orderNo);
        } else {
            $vatCode = $this->taxItemHelper->getShippingTaxCodeForOrder($orderNo);
        }

        $magentoShippingMethod = $order->getShippingDescription();
        $shippingAgentService = $this->settings->getShippingChargeItem($this->erp->getWebsite(), $magentoShippingMethod);
        $shippingAgentServiceArray = explode('#', $shippingAgentService);
        $shipmentItemId = '';
        $shipmentItemType = '';
        $shipmentItemTaxIncluded = false;
        if (count($shippingAgentServiceArray) == 4) {
            $shipmentItemId = $shippingAgentServiceArray[0];
            $shipmentItemType = $shippingAgentServiceArray[1];
            $shipmentItemTaxIncluded = (bool)($shippingAgentServiceArray[2] === 'true');
            $shipmentItemTaxRate = $shippingAgentServiceArray[3];

            $bcDefaultTaxRate = ((float)$shipmentItemTaxRate / 100);
            $shippingPrice = $order->getBaseShippingAmount();
            if ($shipmentItemTaxIncluded) {
                $shippingPrice = $shippingPrice + ($shippingPrice * $bcDefaultTaxRate);
            }

            if ($shipmentItemType === 'charge') {
                $this->erp->addChargeItem(
                    $this->getDocumentType(),
                    $orderResponse->id,
                    $shipmentItemId,
                    (float)$shippingPrice,
                    1,
                    $vatCode,
                    $orderResponse->number,
                    $numOrderLines
                );
            } else {
                $this->erp->addDocumentItem(
                    $this->getDocumentType(),
                    $orderResponse->id,
                    $shipmentItemId,
                    (float)$shippingPrice,
                    1,
                    $vatCode,
                    0,
                    0,
                    null
                );
            }
        } else {
            $this->logger->debug('  - *** No Shipping Charge Mapping found for ' . $magentoShippingMethod);
        }
    }

    abstract protected function processOrder(object $order) : void;

    abstract protected function getOrderStatusFilter() : string;

    abstract protected function getServiceDescription() : string;

    abstract protected function getDocumentType() : string;

    abstract protected function getCapability() : string;
}
