<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\ErpCore\Service;

use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Erp;

class CustomerService
{
    protected $logger;
    protected $erp;
    protected $settings;

    public function __construct(
        ErpLogger $logger,
        Erp $erp,
        Settings $settings
    ) {
        $this->logger = $logger;
        $this->erp = $erp;
        $this->settings = $settings;
    }

    public function getCustomerFingerPrint(
        string $name,
        string $street,
        string $city,
        string $postalcode,
        string $email
    ) : string {
        $customerInfo = $name . $street . $city . $postalcode . $email;
        $customerInfo = str_replace(' ', '', $customerInfo);
        $customerInfo = strtolower($customerInfo);
        $hash = sha1($customerInfo);
        return $hash;
    }

    public function getCustomerId(
        ErpLogger $logger,
        ?string $company,
        string $firstName,
        string $lastName,
        string $street,
        string $city,
        string $postCode,
        string $email,
        ?string $phone,
        ?string $country,
        ?string $region
    ) : string {
        $this->logger = $logger;

        if ($company) {
            $orderFingerPrint = $this->getCustomerFingerprint($company, $street, $city, $postCode, $email);
        } else {
            $orderFingerPrint = $this->getCustomerFingerprint($firstName . $lastName, $street, $city, $postCode, $email);
        }

        $json = $this->erp->getCustomerFromEmail($email);
        if ($json) {
            // We found a customer, let us check if it is the same customer
            foreach ($json as $item) {
                $bcFingerPrint = $this->getCustomerFingerprint(
                    $item->name,
                    $item->street,
                    $item->city,
                    $item->postalCode,
                    $item->email
                );

                if ($this->settings->getCustomerIdentificationMethod($this->erp->getWebsite()) === "fingerprint" &&
                    $orderFingerPrint ==  $bcFingerPrint) {
                    $this->logger->debug('  - Found customer in Erp with id ' . $item->id .
                                         ' and fingerprint ' . $bcFingerPrint);
                    return $item->number;
                }

                if ($this->settings->getCustomerIdentificationMethod($this->erp->getWebsite()) === "email" &&
                    $item->email === $email) {
                    $this->logger->debug('  - Found customer in Erp with id ' . $item->id .
                                         ' and email ' . $email);
                    return $item->number;
                }
            }
        }

        if ($this->settings->getAutoCreateCustomers($this->erp->getWebsite())) {
            $json = $this->erp->createCustomer(
                $company,
                $firstName,
                $lastName,
                $street,
                $city,
                $postCode,
                $email,
                $phone,
                $country,
                $region
            );

            if ($json) {
                $this->logger->debug('  - Creating new customer in Erp with id ' . $json->id);
                return $json->number;
            }
            throw new \RuntimeException('Error while creating customer in Erp');
        }

        throw new \RuntimeException('Customer not found in Erp (auto create customer is disabled in the store setup)');
    }
}
