<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\ErpCore\Test\Unit;

use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Helper\TaxItemHelper;
use Ecomero\ErpCore\Model\Erp;
use Ecomero\ErpCore\Service\CustomerService;
use Ecomero\ErpCore\Service\ProductService;

use Magento\Sales\Model\Order;
use Magento\Sales\Model\Order\Address;
use Magento\Sales\Model\Order\Item;
use Magento\Sales\Model\OrderRepository;
use Magento\Sales\Model\ResourceModel\Order\Collection;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory;

class ExportOrderServiceTest extends \PHPUnit\Framework\TestCase
{
    protected $objectManager;
    protected $exportOrderServiceClass;
    protected $settingsMock;
    protected $erpMock;

    private function createErpLoggerMock() : ErpLogger
    {
        /** @var ErpLogger&\PHPUnit\Framework\MockObject\MockObject $loggerMock */
        $loggerMock = $this->getMockBuilder(ErpLogger::class)
        ->disableOriginalConstructor()
        ->getMock();
        return $loggerMock;
    }

    private function createNotificationMock() : Notification
    {
        /** @var Notification&\PHPUnit\Framework\MockObject\MockObject $notificationMock */
        $notificationMock = $this->getMockBuilder(Notification::class)
                ->disableOriginalConstructor()
                ->setMethods(['notify'])
                ->getMock();
        return $notificationMock;
    }

    private function createErpMock() : Erp
    {
        /** @var Erp&\PHPUnit\Framework\MockObject\MockObject $erpMock */
        $erpMock = $this->getMockBuilder(Erp::class)
                ->disableOriginalConstructor()
                ->setMethods(['setErpOrderNo', 'setWebsiteFromStoredId', 'isErpEnabled', 'getWebsite', 'createDocument', 'updateDocument', 'getDocumentFromExternalId'])
                ->getMock();
        $orderRepsonceMock = new \stdClass();
        $orderRepsonceMock->id = '1';
        $orderRepsonceMock->number = '123';
        $orderRepsonceMock->customerNumber = '1';
        $erpMock->method('isErpEnabled')->willReturn('MockErp');
        $erpMock->method('getWebsite')->willReturn(1);
        $erpMock->method('createDocument')->willReturn($orderRepsonceMock);
        $erpMock->method('updateDocument')->willReturn($orderRepsonceMock);

        return $erpMock;
    }

    private function createSettingsMock() : Settings
    {
        /** @var Settings&\PHPUnit\Framework\MockObject\MockObject $settingsMock */
        $settingsMock = $this->getMockBuilder(Settings::class)
            ->disableOriginalConstructor()
            ->setMethods(['requestCronLock', 'getOrderStatusSend', 'getShippingAgentService', 'getSalesPerson', 'getUseShippingTaxFromOrder', 'getShippingChargeItem', 'getOrderStatusSent', 'releaseCronLock', 'stopOnError'])
            ->getMock();

        $settingsMock->method('getSalesPerson')->willReturn('TEST-SALES-PERSON');
        $settingsMock->method('getOrderStatusSend')->willReturn('Pending');
        $settingsMock->method('getOrderStatusSent')->willReturn('SentToErp');

        return $settingsMock;
    }

    private function createCustomerServiceMock() : CustomerService
    {
        /** @var CustomerService&\PHPUnit\Framework\MockObject\MockObject $customerServiceMock */
        $customerServiceMock = $this->getMockBuilder(CustomerService::class)
                ->disableOriginalConstructor()
                ->setMethods(['getCustomerId'])
                ->getMock();
        $customerServiceMock->method('getCustomerId')->willReturn('1');
        return $customerServiceMock;
    }

    private function createOrderCollectionMock() : CollectionFactory
    {
        $orderItemMock = [];
        $orderItemMock['entity_id'] = '1';
        $orderItemMock['increment_id'] = '1000000';
        $orderCollectionMock = $this->objectManager->getCollectionMock(Collection::class, [$orderItemMock]);
        $orderCollectionMock->method('getData')->willReturn([$orderItemMock]);
        $orderCollectionMock->method('addFieldToSelect')->willReturn($orderCollectionMock);

        /** @var CollectionFactory&\PHPUnit\Framework\MockObject\MockObject $orderCollectionFactory */
        $orderCollectionFactory = $this->getMockBuilder(CollectionFactory::class)
            ->disableOriginalConstructor()
            ->setMethods(['create', 'addFieldToSelect', 'addAttributeToFilter', 'getData'])
            ->getMock();

        $orderCollectionFactory->method('create')->willReturn($orderCollectionMock);
        return $orderCollectionFactory;
    }

    private function createOrderRepositoryMock() : OrderRepository
    {
        $orderItemMock = $this->getMockBuilder(Item::class)
        ->disableOriginalConstructor()
        ->getMock();

        $address = [];
        $address['firstname'] = 'Test';
        $address['lastname'] = 'Testersen';
        $address['street'] = 'Testroad';
        $address['city']  = 'Test City';
        $address['postcode']  = '99412';
        $address['email'] = 'tester@unittest.com';
        $address['telephone']  = '800-123 123';
        $address['country_id']  = 'Test Land';
        $address['region'] = 'Test Region';

        $addressMock = $this->getMockBuilder(Address::class)
            ->disableOriginalConstructor()
            ->setMethods(['getData'])
            ->getMock();
        $addressMock->method('getData')->willReturn($address);

        /** @var Order&\PHPUnit\Framework\MockObject\MockObject $orderMock */
        $orderMock = $this->getMockBuilder(Order::class)
        ->disableOriginalConstructor()
        ->setMethods(['getStoreId', 'getBillingAddress', 'getShippingAddress', 'getShippingDescription', 'getAllVisibleItems', 'addStatusToHistory', 'save', 'addStatusHistoryComment', 'getStatusHistories'])
        ->getMock();
        $orderMock->method('getStatusHistories')->willReturn([]);
        $orderMock->method('getStoreId')->willReturn(1);
        $orderMock->method('getShippingDescription')->willReturn('postnord');
        $orderMock->method('getAllVisibleItems')->willReturn([$orderItemMock]);
        $orderMock->method('getBillingAddress')->willReturn($addressMock);
        $orderMock->method('getShippingAddress')->willReturn($addressMock);
        $orderMock['order_currency_code'] = 'SEK';
        $orderMock['increment_id'] = '1000000';
        $orderMock['entity_id'] = '1';

        /** @var OrderRepository&\PHPUnit\Framework\MockObject\MockObject $orderRepositoryMock */
        $orderRepositoryMock = $this->getMockBuilder(OrderRepository::class)
                ->disableOriginalConstructor()
                ->setMethods(['get'])
                ->getMock();
        $orderRepositoryMock->method('get')->willReturn($orderMock);
        return $orderRepositoryMock;
    }

    private function createProductServiceMock() : ProductService
    {
        /** @var ProductService&\PHPUnit\Framework\MockObject\MockObject $productServiceMock */
        $productServiceMock = $this->getMockBuilder(ProductService::class)
            ->disableOriginalConstructor()
            ->setMethods(['getErpItemInfoFromSKU'])
            ->getMock();
        $erpItemInfoMock['erp_id'] = '100';
        $erpItemInfoMock['erp_tax_rate'] = 25;
        $erpItemInfoMock['erp_tax_included'] = 1;
        $productServiceMock->method('getErpItemInfoFromSKU')->willReturn($erpItemInfoMock);
        return $productServiceMock;
    }

    private function createTaxItemHelperMock() : TaxItemHelper
    {
        /** @var TaxItemHelper&\PHPUnit\Framework\MockObject\MockObject $taxItemHelperMock */
        $taxItemHelperMock = $this->getMockBuilder(TaxItemHelper::class)
            ->disableOriginalConstructor()
            ->setMethods(['getTaxCodeForOrderLine', 'getShippingTaxCodeForOrder', 'getTaxCodeForFirstProduct'])
            ->getMock();
        $taxItemHelperMock->method('getTaxCodeForOrderLine')->willReturn('1');
        $taxItemHelperMock->method('getShippingTaxCodeForOrder')->willReturn('MOMS25');
        $taxItemHelperMock->method('getTaxCodeForFirstProduct')->willReturn('MOMS25');
        return $taxItemHelperMock;
    }

    public function setUp() : void
    {
        $this->objectManager =  new \Magento\Framework\TestFramework\Unit\Helper\ObjectManager($this);

        $this->settingsMock = $this->createSettingsMock();
        $this->erpMock = $this->createErpMock();
        $this->exportOrderServiceClass = new \Ecomero\ErpCore\Service\ExportOrderService(
            $this->createErpLoggerMock(),
            $this->createNotificationMock(),
            $this->erpMock,
            $this->settingsMock,
            $this->createCustomerServiceMock(),
            $this->createOrderCollectionMock(),
            $this->createOrderRepositoryMock(),
            $this->createProductServiceMock(),
            $this->createTaxItemHelperMock()
        );
    }

    public function testExecuteFromCron() : void
    {
        $this->exportOrderServiceClass->executeFromCron();
    }

    public function testExecuteFromWeb() : void
    {
        $this->exportOrderServiceClass->executeFromWeb();
    }

    public function testExecuteFromShell() : void
    {
        /** @var \Symfony\Component\Console\Output\ConsoleOutput&\PHPUnit\Framework\MockObject\MockObject $outputMock */
        $outputMock = $this->getMockBuilder(\Symfony\Component\Console\Output\ConsoleOutput::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->exportOrderServiceClass->executeFromShell($outputMock, false);
    }

    public function testExecuteLocked() : void
    {
        $this->settingsMock->expects($this->once())
            ->method('requestCronLock')
            ->willReturn(false);
        $this->exportOrderServiceClass->execute();
    }

    public function testExecuteForceUnlock() : void
    {
        $this->settingsMock->expects($this->once())
            ->method('requestCronLock')
            ->willReturn(false);
        $this->exportOrderServiceClass->execute(true);
    }

    public function testExecuteShippingTaxFromOrder() : void
    {
        $this->settingsMock->expects($this->once())
            ->method('getUseShippingTaxFromOrder')
            ->willReturn(true);
        $this->settingsMock->expects($this->once())
            ->method('requestCronLock')
            ->willReturn(true);

        $this->exportOrderServiceClass->execute();
    }

    public function testExecuteShippingCharge() : void
    {
        $this->settingsMock->expects($this->once())
            ->method('getShippingChargeItem')
            ->willReturn('id#charge#true');
        $this->settingsMock->expects($this->once())
            ->method('requestCronLock')
            ->willReturn(true);

        $this->exportOrderServiceClass->execute();
    }

    public function testExecuteShippingItem() : void
    {
        $this->settingsMock->expects($this->once())
            ->method('getShippingChargeItem')
            ->willReturn('id#item#true');
        $this->settingsMock->expects($this->once())
            ->method('requestCronLock')
            ->willReturn(true);

        $this->exportOrderServiceClass->execute();
    }

    public function testExecuteShippingMissing() : void
    {
        $this->settingsMock->expects($this->once())
            ->method('getShippingChargeItem')
            ->willReturn('fail');
        $this->settingsMock->expects($this->once())
            ->method('requestCronLock')
            ->willReturn(true);

        $this->exportOrderServiceClass->execute();
    }

    public function testExecuteHttpException() : void
    {
        $this->settingsMock->expects($this->once())
            ->method('stopOnError')
            ->willReturn(false);
        $this->settingsMock->expects($this->once())
            ->method('requestCronLock')
            ->willReturn(true);

        $this->erpMock->expects($this->once())
            ->method('createDocument')
            ->will(
                $this->throwException(
                    new \Zend\Http\Exception\RuntimeException('ERROR: ' . 'create document failed!')
                )
            );

        $this->exportOrderServiceClass->execute();
    }

    public function testExecuteHttpExceptionStopOnError() : void
    {
        $this->settingsMock->expects($this->once())
            ->method('stopOnError')
            ->willReturn(true);
        $this->settingsMock->expects($this->once())
            ->method('requestCronLock')
            ->willReturn(true);

        $this->erpMock->expects($this->once())
            ->method('createDocument')
            ->will(
                $this->throwException(
                    new \Zend\Http\Exception\RuntimeException('ERROR: ' . 'create document failed!')
                )
            );

        $this->exportOrderServiceClass->execute();
    }

    public function testExecuteLocalizedException() : void
    {
        $this->settingsMock->expects($this->once())
            ->method('requestCronLock')
            ->willReturn(true);

        $this->erpMock->expects($this->once())
            ->method('createDocument')
            ->will(
                $this->throwException(
                    new \Magento\Framework\Exception\LocalizedException(new \Magento\Framework\Phrase('ERROR'))
                )
            );

        $this->exportOrderServiceClass->execute();
    }

    public function testExecuteRuntimeException() : void
    {
        $this->settingsMock->expects($this->once())
            ->method('requestCronLock')
            ->willReturn(true);

        $this->erpMock->expects($this->once())
            ->method('createDocument')
            ->will(
                $this->throwException(
                    new \RuntimeException('ERROR: ' . 'create document failed!')
                )
            );

        $this->exportOrderServiceClass->execute();
    }

    public function testExecuteDocumentExists() : void
    {
        $this->settingsMock->expects($this->once())
            ->method('requestCronLock')
            ->willReturn(true);

        $this->erpMock->expects($this->once())
            ->method('getDocumentFromExternalId')
            ->willReturn((object)['number' => '1']);

        $this->exportOrderServiceClass->execute();
    }

    public function testExecuteDocumentExistsError() : void
    {
        $this->settingsMock->expects($this->once())
            ->method('requestCronLock')
            ->willReturn(true);

        $this->erpMock->expects($this->once())
            ->method('getDocumentFromExternalId')
            ->willReturn((object)['errorMessage' => 'mock error']);

        $this->exportOrderServiceClass->execute();
    }

    public function testExecuteShipmentAgentService() : void
    {
        $this->settingsMock->expects($this->once())
            ->method('requestCronLock')
            ->willReturn(true);

        $this->settingsMock->expects($this->once())
            ->method('getShippingAgentService')
            ->willReturn('Postnord-Standard');

        $this->exportOrderServiceClass->execute();
    }
}
