<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\ErpCore\Service;

use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\ErpAdapterInterface;

use Magento\Store\Model\StoreManagerInterface;
use Symfony\Component\Console\Output\OutputInterface;

class ReturnService
{
    protected $logger;
    protected $notification;
    protected $erp;
    protected $settings;
    protected $storeManager;
    protected $productMetadata;
    protected static $thisErrorHandler = null;

    public function __construct(
        ErpLogger $logger,
        Notification $notification,
        ErpAdapterInterface $erp,
        Settings $settings,
        StoreManagerInterface $storeManager,
        \Magento\Framework\App\ProductMetadataInterface $productMetadata
    ) {
        $this->logger = $logger;
        $this->notification = $notification;
        $this->erp = $erp;
        $this->settings = $settings;
        $this->storeManager = $storeManager;
        $this->productMetadata = $productMetadata;
    }

    public function importCommandLine(OutputInterface $output, bool $forceUnlock) : void
    {
        $this->logger->info('Starting import of returns from command line');
        $this->logger->setOutput($output);
        $this->import($forceUnlock);
        set_error_handler(null);
    }

    public function importCron(bool $forceUnlock = false) : void
    {
        $this->logger->info('Starting import of returns from cron job');
        $this->import($forceUnlock);
        set_error_handler(null);
    }

    public function importWeb(bool $forceUnlock = false) : void
    {
        $this->logger->info('Starting import of returns from web');
        $this->import($forceUnlock);
        set_error_handler(null);
    }

    public static function erpErrorHandler($errno, $errstr, $errfile, $errline)
    {
        if (self::$thisErrorHandler) {
            self::$thisErrorHandler->logger->error($errstr);
        }

        return false;
    }

    private function import(bool $forceUnlock = false) : void
    {
        self::$thisErrorHandler = $this;
        set_error_handler('Ecomero\ErpCore\Service\ReturnService::erpErrorHandler');

        if ($this->productMetadata->getEdition() !== 'Enterprise') {
            $this->logger->error('Returns are not available in Magento ' . $this->productMetadata->getEdition() . ' edition.');
            return;
        }

        if ($forceUnlock) {
            $this->settings->releaseCronLock();
        }

        if ($this->settings->requestCronLock($this->logger) == false) {
            return;
        }

        $capabilities = $this->erp->getCapabilities();
        foreach ($capabilities as $capability) {
            if (!$capability->isCapabilitySupported(Capability::RETURN_IMPORT)) {
                $this->logger->warning('Import of returns are not supported by the ' . $capability->getName() . ' integration');
                $this->settings->releaseCronLock();
                return;
            }
        }

        $start_time = microtime(true);

        $errorMessage = '';
        try {
            $this->logger->info('Retrieving returns from ERP');
            $json = $this->erp->getReturns();
            foreach ($json as $item) {
                $this->logger->info('Importing return for ERP order ' . $item->erpOrderNumber . ', sku ' . $item->sku . ', qty ' . $item->qty);
            }
        } catch (\Zend\Http\Exception\RuntimeException $exception) {
            $errorMessage = $errorMessage . $exception->getMessage();
            $this->logger->error($exception->getMessage());
        } catch (\Magento\Framework\Exception\LocalizedException $exception) {
            $errorMessage = $errorMessage . $exception->getMessage();
            $this->logger->error($exception->getMessage());
        } catch (\RuntimeException $exception) {
            $errorMessage = $errorMessage . $exception->getMessage();
            $this->logger->error($exception->getMessage());
        }

        if ($errorMessage !== '' && $this->settings->stopOnError()) {
            $this->notification->notify($errorMessage);
            return;
        }
        $this->logger->info('Import completed in ' . (microtime(true) - $start_time) . ' sec');
        $this->settings->releaseCronLock();

        if ($errorMessage !== '') {
            $this->notification->notify($errorMessage);
        }
    }
}
