<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\ErpCore\Helper;

use Magento\Framework\Filesystem\Driver\File;
use Magento\Framework\Notification\NotifierInterface;
use Symfony\Component\Console\Output\OutputInterface;

class ErpLogger
{
    protected $logger;
    protected $output;
    protected $type;
    protected $notificationManager;
    protected $fileSystem;
    protected $settings;
    protected $logDirectory;

    const MAGENTO_LOG_ROOT = BP . '/var/log/';
    const ERP_LOG_ROOT = self::MAGENTO_LOG_ROOT . 'eComero';
    const FILE_PREFIX = 'erp-integration-';

    const RETENTION_1_DAY = 60 * 60 * 24 * 1;
    const RETENTION_7_DAY = 60 * 60 * 24 * 7;
    const RETENTION_14_DAY = 60 * 60 * 24 * 14;
    const RETENTION_30_DAY = 60 * 60 * 24 * 30;
    const RETENTION_90_DAY = 60 * 60 * 24 * 90;

    public function __construct(
        \Magento\Framework\App\State $state,
        NotifierInterface $notificationManager,
        File $fileSystem,
        Settings $settings,
        String $type = 'general'
    ) {
        try {
            $state->getAreaCode(\Magento\Framework\App\Area::AREA_ADMINHTML);
        } catch (\Magento\Framework\Exception\LocalizedException $e) {
            $state->setAreaCode(\Magento\Framework\App\Area::AREA_ADMINHTML);
        }
        $this->type = $type;
        $this->notificationManager = $notificationManager;
        $this->fileSystem = $fileSystem;
        $this->settings = $settings;
        $this->logDirectory = self::ERP_LOG_ROOT;

        $tmpStr = $this->settings->getLogDirectory();

        if ($tmpStr !== null && $tmpStr !== '') {
            if (substr($tmpStr, 0, 1) === "/") {
                // We have an absolute path
                $this->logDirectory = $this->settings->getLogDirectory();
            } else {
                // We have a relative path
                $this->logDirectory = BP . '/' . $this->settings->getLogDirectory();
            }
        }

        // Check if we have the normal log directory, to avoid problem during a fresh installation
        if ($this->fileSystem->isDirectory(self::MAGENTO_LOG_ROOT)) {
            // Check if our log directory exists
            if (!$this->fileSystem->isDirectory($this->logDirectory)) {
                $this->fileSystem->createDirectory($this->logDirectory);
            }
            $this->logger = new \Laminas\Log\Logger();
            $writer = new \Laminas\Log\Writer\Stream($this->getActiveLogFile());
            $this->logger->addWriter($writer);
        }
    }

    public function deleteOldLogfiles() : void
    {
        $timeLimit = 0;
        switch ($this->settings->getLogRetentionTime()) {
            case 'RETENTION_1_DAY':
                $timeLimit = self::RETENTION_1_DAY;
            break;
            case 'RETENTION_7_DAY':
                $timeLimit = self::RETENTION_7_DAY;
            break;
            case 'RETENTION_14_DAY':
                $timeLimit = self::RETENTION_14_DAY;
            break;
            case 'RETENTION_30_DAY':
                $timeLimit = self::RETENTION_30_DAY;
            break;
            case 'RETENTION_90_DAY':
                $timeLimit = self::RETENTION_90_DAY;
            break;
            case 'RETENTION_FOREVER':
                $timeLimit = 0;
            break;
        }

        foreach ($this->fileSystem->readDirectory($this->logDirectory) as $fileName) {
            if (strpos($fileName, $this->logDirectory . '/' . self::FILE_PREFIX) === 0) {
                $fileInfo = $this->fileSystem->stat($fileName);

                $mtime = $fileInfo['mtime'];
                $now   = time();
                if ($now - $mtime >= $timeLimit && $timeLimit !== 0) {
                    $this->fileSystem->deleteFile($fileName);
                }
            }
        }
    }

    public function getActiveLogFile() : string
    {
        return $this->logDirectory . '/' . self::FILE_PREFIX . $this->type . '-' . date("Ymd") . '.log';
    }

    public function setOutput(OutputInterface $output) : void
    {
        $this->output = $output;
    }

    public function emergency(string $message, array $context = []) : void
    {
        if ($this->output) {
            $this->output->writeln('<error>' . $message . '</error>');
        }
        $this->logger->emerg($message, $context);
    }

    public function alert(string $message, array $context = []) : void
    {
        if ($this->output) {
            $this->output->writeln('<error>' . $message . '</error>');
        }
        $this->logger->alert($message, $context);
    }

    public function critical(string $message, array $context = []) : void
    {
        if ($this->output) {
            $this->output->writeln('<error>' . $message . '</error>');
        }

        $this->logger->crit($message, $context);
    }

    public function error(string $message, array $context = []) : void
    {
        if ($this->output) {
            $this->output->writeln('<error>' . $message . '</error>');
        }

        $this->notificationManager->addCritical('ERP Integration', $message);
        $this->logger->debug('<error>' . $message . '</error>', $context);
    }

    public function warning(string $message, array $context = []) : void
    {
        if ($this->output) {
            $this->output->writeln('<error>' . $message . '</error>');
        }
        $this->logger->debug('<error>' . $message . '</error>', $context);
    }

    public function notice(string $message, array $context = []) : void
    {
        if ($this->output) {
            $this->output->writeln('<info>' . $message . '</info>');
        }
        $this->logger->notice('<info>' . $message . '</info>', $context);
    }

    public function info(string $message, array $context = []) : void
    {
        if ($this->output) {
            $this->output->writeln('<info>' . $message . '</info>');
        }
        $this->logger->info('<info>' . $message . '</info>', $context);
    }

    public function debug(string $message, array $context = []) : void
    {
        if ($this->output) {
            $this->output->writeln('<comment>' . $message . '</comment>');
        }
        $this->logger->debug($message, $context);
    }

    public function log(string $message, array $context = []) : void
    {
        if ($this->output) {
            $this->output->writeln($message);
        }
        $this->logger->log($message, $context);
    }
}
