<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\ErpCore\Service;

use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Helper\TaxItemHelper;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\ErpAdapterInterface;

use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order\Email\Sender\InvoiceSenderFactory;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory as OrderCollectionFactory;
use Magento\Sales\Model\ResourceModel\Order\Invoice\CollectionFactory as IvoiceCollectionFactory;
use Magento\Sales\Model\Service\InvoiceServiceFactory;

class ImportInvoiceService extends OrderServiceAbstract
{
    protected $invoiceServiceFactory;
    protected $shipmentFactory;
    protected $shipmentTrackFactory;
    protected $shipmentDocFactory;
    protected $shipmentSender;
    protected $invoiceCollectionFactory;
    protected $invoiceSenderFactory;
    protected $taxItemHelper;

    public function __construct(
        CustomerService $customerService,
        ErpLogger $logger,
        Notification $notification,
        ErpAdapterInterface $erp,
        Settings $settings,
        OrderCollectionFactory $orderCollectionFactory,
        OrderRepositoryInterface $orderRepository,
        InvoiceServiceFactory $invoiceServiceFactory,
        IvoiceCollectionFactory $invoiceCollectionFactory,
        InvoiceSenderFactory $invoiceSenderFactory,
        TaxItemHelper $taxItemHelper
    ) {
        parent::__construct(
            $logger,
            $notification,
            $settings,
            $erp,
            $customerService,
            $orderCollectionFactory,
            $orderRepository
        );

        $this->invoiceServiceFactory = $invoiceServiceFactory;
        $this->invoiceCollectionFactory = $invoiceCollectionFactory;
        $this->invoiceSenderFactory = $invoiceSenderFactory;
        $this->taxItemHelper = $taxItemHelper;
    }

    protected function getServiceDescription() : string
    {
        return 'invoice import';
    }

    protected function getDocumentType() : string
    {
        return 'Invoice';
    }

    protected function getCapability() : string
    {
        return Capability::INVOICE_IMPORT;
    }

    private function getOrderItemIdFromSku(object $order, string $sku, float $qty) : array
    {
        $itemsArray = [];
        foreach ($order->getAllItems() as $orderItem) {
            if ($orderItem->getSku() == $sku) {
                $itemsArray[$orderItem->getId()] = $qty;
            }
        }
        return $itemsArray;
    }

    private function isOrderAlreadyInMagento(string $invoiceNo) : bool
    {
        $invoiceCollecion = $this->invoiceCollectionFactory->create();
        $invoiceCollecion->addAttributeToFilter('erp_invoice_no', ['eq'=> $invoiceNo ]);
        foreach ($invoiceCollecion->getData() as $order) {
            return true;
        }
        return false;
    }

    protected function getOrderStatusFilter() : string
    {
        return $this->settings->getOrderStatusSent();
    }

    protected function processOrder(\Magento\Sales\Model\Order $order) : void
    {
        if (!$order['erp_order_no'] || strpos($order['erp_order_no'], 'ERROR') !== false) {
            return;
        }

        try {
            $invoiceNo = $this->erp->getInvoiceFromOrder($order['erp_order_no']);

            if ($order->canInvoice() && property_exists($invoiceNo, 'number')) {
                if ($this->isOrderAlreadyInMagento($invoiceNo->number)) {
                    $this->logger->debug('  - Invoice already imported for invoice : ' . $invoiceNo->number);
                    return;
                }

                if ($order->canShip()) {
                    $this->logger->debug('Found order that is not shipped ' . $order['erp_order_no']);
                    return;
                }

                $this->logger->debug('  - Creating invoice document for invoice : ' . $invoiceNo->number);

                $bcInvoiceLines = $this->erp->getInvoiceOrderLines($invoiceNo->id);

                $itemsArray = [];
                $shippingAmount = 0;
                $total = 0;
                $giftCardAmount = 0;
                foreach ($bcInvoiceLines as $bcInvoiceLine) {
                    if ($bcInvoiceLine->lineType === 'Charge') {
                        $shippingAmount += $bcInvoiceLine->amountIncludingTax;
                        $total += $bcInvoiceLine->amountIncludingTax;
                    } elseif ($bcInvoiceLine->lineType === 'Item') {
                        $items = $this->getOrderItemIdFromSku(
                            $order,
                            $bcInvoiceLine->lineDetails->number,
                            $bcInvoiceLine->quantity
                        );
                        foreach ($items as $key => $value) {
                            $itemsArray[$key] = $value;
                        }
                        if ($this->settings->isShippingChargeItem(
                            $this->erp->getWebsite(),
                            $bcInvoiceLine->itemId
                        )) {
                            $shippingAmount += $bcInvoiceLine->amountIncludingTax;
                        }

                        $total += $bcInvoiceLine->amountIncludingTax;
                    }
                }

                $invoice = $this->invoiceServiceFactory->create()->prepareInvoice($order, $itemsArray);

                $invoice->setShippingAmount($shippingAmount);

                $giftCardAmount = $order->getBaseGiftCardsAmount() ?? (float)0;
                $invoice->setBaseGiftCardsAmount($giftCardAmount);
                $invoice->setGiftCardsAmount($giftCardAmount);
                $invoice->setSubtotal($total);
                $invoice->setBaseSubtotal($total);
                $invoice->setGrandTotal($total);
                $invoice->setBaseGrandTotal($total);

                $invoice->setRequestedCaptureCase(\Magento\Sales\Model\Order\Invoice::CAPTURE_ONLINE);
                $invoice->setErpInvoiceNo($invoiceNo->number);
                $invoice->register();
                $invoice->save();
                $invoice->getOrder()->save();   // Order must be saved to update its status to closed

                if ($this->settings->sendInvoiceConfirmation()) {
                    $this->invoiceSenderFactory->create()->send($invoice);
                    $order->addStatusHistoryComment(
                        __('Notified customer about invoice #%1.', $invoice->getId())
                    )
                    ->setIsCustomerNotified(true)
                    ->save();
                }
            }
        } catch (\Throwable  $e) {
            $this->logger->critical($e->getMessage());
        }
    }
}
