<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\ErpCore\Service;

use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Helper\TaxItemHelper;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\ErpAdapterInterface;

use Magento\Framework\DB\TransactionFactory;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Api\ShipmentRepositoryInterface;
use Magento\Sales\Model\Convert\OrderFactory;
use Magento\Sales\Model\Order\Shipment\Sender\EmailSenderFactory as ShipmentSenderFactory;
use Magento\Sales\Model\Order\Shipment\TrackFactory;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory as OrderCollectionFactory;
use Magento\Sales\Model\ResourceModel\Order\Shipment\CollectionFactory as ShipmentCollectionFactory;

class ImportShipmentService extends OrderServiceAbstract
{
    protected $transactionFactory;
    protected $shipmentFactory;
    protected $shipmentTrackFactory;
    protected $shipmentDocFactory;
    protected $shipmentSenderFactory;
    protected $shipmentRepository;
    protected $erp;
    protected $taxItemHelper;

    public function __construct(
        CustomerService $customerService,
        ErpLogger $logger,
        Notification $notification,
        Settings $settings,
        ErpAdapterInterface $erp,
        OrderCollectionFactory $orderCollectionFactory,
        OrderRepositoryInterface $orderRepository,
        TransactionFactory $transactionFactory,
        OrderFactory $shipmentFactory,
        TrackFactory $shipmentTrackFactory,
        ShipmentCollectionFactory $shipmentDocFactory,
        ShipmentSenderFactory $shipmentSenderFactory,
        ShipmentRepositoryInterface $shipmentRepository,
        TaxItemHelper $taxItemHelper
    ) {
        parent::__construct(
            $logger,
            $notification,
            $settings,
            $erp,
            $customerService,
            $orderCollectionFactory,
            $orderRepository
        );

        $this->transactionFactory = $transactionFactory;
        $this->shipmentFactory = $shipmentFactory;
        $this->shipmentTrackFactory = $shipmentTrackFactory;
        $this->shipmentDocFactory = $shipmentDocFactory;
        $this->shipmentSenderFactory = $shipmentSenderFactory;
        $this->shipmentRepository = $shipmentRepository;
        $this->taxItemHelper = $taxItemHelper;
    }

    protected function getServiceDescription() : string
    {
        return 'shipment import';
    }

    protected function getDocumentType() : string
    {
        return 'Shipment';
    }

    protected function getCapability() : string
    {
        return Capability::SHIPMENT_IMPORT;
    }

    private function getQtyForSku(array $shippingLineItems, string  $sku) : int
    {
        $qty = 0;
        foreach ($shippingLineItems as $item) {
            if ($item->sku == $sku) {
                $qty += $item->qty;
            }
        }
        return $qty;
    }

    private function existShippingDoc(string $shippingNo) : bool
    {
        $shipmentDocCollecion = $this->shipmentDocFactory->create();
        $shipmentDocCollecion->addAttributeToFilter('erp_sales_shipment_no', ['eq'=> $shippingNo ]);
        foreach ($shipmentDocCollecion->getData() as $order) {
            return true;
        }
        return false;
    }

    private function isFreightItem(object $shippingLine) : bool
    {
        return $shippingLine->lineType == 'Charge (Item)';
    }

    private function createShippingDoc(array $shippingLineItems, object $order) : void
    {
        // Check if shipping document already is created
        if ($this->existShippingDoc($shippingLineItems[0]->shippingNo)) {
            return;
        }

        if ($order->canShip()) {
            $shipmentOrder =  $this->shipmentFactory->create();
            if ($shipment = $shipmentOrder->toShipment($order)) {
                $isAnythingToShip = false;
                foreach ($order->getAllItems() as $orderItem) {
                    // Check if order item has qty to ship or is virtual
                    $qtyShipped = $this->getQtyForSku($shippingLineItems, $orderItem->getSku());
                    if (!$orderItem->getQtyToShip() || $qtyShipped == 0 || $orderItem->getIsVirtual()) {
                        continue;
                    }
                    $isAnythingToShip = true;
                    $shipmentItem = $shipmentOrder->itemToShipmentItem($orderItem)->setQty($qtyShipped);
                    $shipment->addItem($shipmentItem);
                }

                if ($isAnythingToShip) {
                    $carrierCode = $order->getShippingMethod();

                    $this->logger->debug('  - Creating shipping document for shipment : ' .
                                        $shippingLineItems[0]->shippingNo .
                                        ', carrier : ' . $carrierCode .
                                        ', agent : ' . $shippingLineItems[0]->shippingAgent .
                                        ', tracking no. : ' . $shippingLineItems[0]->packageTrackingNo);

                    $shipment->register();

                    if ($shippingLineItems[0]->packageTrackingNo && $shippingLineItems[0]->shippingAgent) {
                        $track = $this->shipmentTrackFactory->create();
                        $track->setNumber($shippingLineItems[0]->packageTrackingNo);

                        $track->setCarrierCode($carrierCode);
                        $track->setTitle('Tracking Number');
                        $track->setDescription($shippingLineItems[0]->shippingAgent);

                        // If the warehouse is setting a carrier we can match, then use it
                        // otherwise use the carrier description from the Magento order
                        $url = $this->settings->getShippingTrackUrl(
                            $this->erp->getWebsite(),
                            $shippingLineItems[0]->shippingAgent
                        );
                        if ($url === '') {
                            $url = $this->settings->getShippingTrackUrl(
                                $this->erp->getWebsite(),
                                $order->getShippingDescription()
                            );
                            if ($url !== '') {
                                $track->setDescription($order->getShippingDescription());
                            }
                        }

                        $shipment->addTrack($track);
                    }
                    $shipment->setErpSalesShipmentNo($shippingLineItems[0]->shippingNo);
                    $this->shipmentRepository->save($shipment);

                    $transaction = $this->transactionFactory->create();
                    $transactionSave = $transaction->addObject($shipment)
                                                    ->addObject($order);
                    $transactionSave->save();

                    if ($this->settings->sendShipmentConfirmation()) {
                        $this->shipmentSender->create()->send($order, $shipment);
                        $order->addStatusHistoryComment(
                            __('Notified customer about shipment #%1.', $shipment->getId())
                        )
                        ->setIsCustomerNotified(true)
                        ->save();
                    }
                }
            }
        }
    }

    protected function getOrderStatusFilter() : string
    {
        return $this->settings->getOrderStatusSent();
    }

    protected function processOrder(\Magento\Sales\Model\Order $order) : void
    {
        $shippmentDetails = $this->erp->getShippment($order['increment_id']);

        $shippingLineItems = [];
        $activeShippingNo = null;
        foreach ($shippmentDetails as $shippingDoc) {
            if (!$this->isFreightItem($shippingDoc)) {
                if ($activeShippingNo === null) {
                    $activeShippingNo = $shippingDoc->shippingNo;
                }

                if ($activeShippingNo !== $shippingDoc->shippingNo) {
                    $this->createShippingDoc($shippingLineItems, $order);
                    $shippingLineItems = [];
                    $activeShippingNo = $shippingDoc->shippingNo;
                }

                $shippingLineItems[] = $shippingDoc;
            }
        }
        if (count($shippingLineItems) > 0) {
            $this->createShippingDoc($shippingLineItems, $order);
        }
    }
}
