<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\ErpCore\Helper;

use Magento\Config\Model\ResourceModel\Config\Data\CollectionFactory;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Config\Storage\WriterInterface;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\Encryption\EncryptorInterface;
use Magento\Framework\Serialize\Serializer\Json;

class Settings extends AbstractHelper
{
    public const DISCOUNT_LINE = 'LINE';
    public const DISCOUNT_TOTAL = 'TOTAL';

    protected $encryptor;
    protected $configWriter;
    protected $configCollectionFactory;
    protected $unserializer;

    public function __construct(
        Context $context,
        EncryptorInterface $encryptor,
        WriterInterface $configWriter,
        CollectionFactory $configCollectionFactory,
        Json $unserializer
    ) {
        parent::__construct($context);
        $this->encryptor = $encryptor;
        $this->configWriter = $configWriter;
        $this->configCollectionFactory = $configCollectionFactory;
        $this->unserializer = $unserializer;
    }

    public function isEnabled(int $websiteId) : bool
    {
        return $this->scopeConfig->getValue('erp_core/general/enabled', 'website', $websiteId) == '1';
    }

    public function getCustomerIdentificationMethod(int $websiteId) : string
    {
        return $this->scopeConfig->getValue('erp_core/order/customer_id_method', 'website', $websiteId);
    }

    public function getAutoCreateCustomers(int $websiteId) : bool
    {
        return $this->scopeConfig->getValue('erp_core/order/auto_create_customers', 'website', $websiteId) == '1';
    }

    public function getAllowNewProducts($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : bool
    {
        return $this->scopeConfig->getValue('erp_core/catalog/allow_new_products', $scope) == '1';
    }

    public function getCategoryMapping(int $websiteId) : ?array
    {
        $tableConfig =  $this->scopeConfig->getValue('erp_core/catalog/category_mapping', 'website', $websiteId);
        if ($tableConfig) {
            $tableConfigResults = $this->unserializer->unserialize($tableConfig);
            if (is_array($tableConfigResults)) {
                return $tableConfigResults;
            }
        }
        return null;
    }

    public function sendShipmentConfirmation($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : bool
    {
        return $this->scopeConfig->getValue('erp_core/shipping/send_shipment_confirmation', $scope) == '1';
    }

    public function getUseShippingTaxFromOrder($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : bool
    {
        return $this->scopeConfig->getValue('erp_core/shipping/shipping_tax_order', $scope) == '1';
    }

    public function getShippingChargeItem(int $websiteId, string $magentoShippingMethod) : string
    {
        $item = '';
        $tableConfig =  $this->scopeConfig->getValue('erp_core/shipping/shipping_agent', 'website', $websiteId);
        if ($tableConfig) {
            $tableConfigResults = $this->unserializer->unserialize($tableConfig);
            if (is_array($tableConfigResults)) {
                foreach ($tableConfigResults as $tableConfigResult) {
                    if (key_exists('freight_item', $tableConfigResult)) {
                        $item = $tableConfigResult['freight_item'];
                    }
                    $magentoShipping = $tableConfigResult['magento_shipping'];
                    if (stripos($magentoShippingMethod, $magentoShipping) !== false) {
                        return $item;
                    }
                }
            }
        }
        return $item;
    }

    public function getGiftCardItemId(int $websiteId) : string
    {
        $itemId = '';
        $itemConfig = $this->scopeConfig->getValue('erp_core/giftcard/item_id', 'website', $websiteId);
        if ($itemConfig) {
            $itemId = explode('#', $itemConfig)[0];
        }
        return $itemId;
    }

    public function isGiftCardItem(int $websiteId, string $itemId) : bool
    {
        return $itemId === $this->getGiftCardItemId($websiteId);
    }

    public function isShippingChargeItem(int $websiteId, string $itemId) : bool
    {
        $item = '';
        $tableConfig =  $this->scopeConfig->getValue('erp_core/shipping/shipping_agent', 'website', $websiteId);
        if ($tableConfig) {
            $tableConfigResults = $this->unserializer->unserialize($tableConfig);
            if (is_array($tableConfigResults)) {
                foreach ($tableConfigResults as $tableConfigResult) {
                    if (key_exists('freight_item', $tableConfigResult)) {
                        $item = $tableConfigResult['freight_item'];
                        $parts = explode('#', $item);
                        if ($parts[0] === $itemId) {
                            return true;
                        }
                    }
                }
            }
        }
        return false;
    }

    public function getShippingAgentService(int $websiteId, string $magentoShippingMethod) : string
    {
        $agent = '';
        $tableConfig =  $this->scopeConfig->getValue('erp_core/shipping/shipping_agent', 'website', $websiteId);
        if ($tableConfig) {
            $tableConfigResults = $this->unserializer->unserialize($tableConfig);
            if (is_array($tableConfigResults)) {
                foreach ($tableConfigResults as $tableConfigResult) {
                    if (key_exists('agent_service', $tableConfigResult)) {
                        $agent = $tableConfigResult['agent_service'];
                    }
                    $magentoShipping = $tableConfigResult['magento_shipping'];
                    if (stripos($magentoShippingMethod, $magentoShipping) !== false) {
                        return $agent;
                    }
                }
            }
        }
        return $agent;
    }

    public function getShippingTrackUrl(int $websiteId, string $magentoShippingMethod) : string
    {
        $url = '';
        $tableConfig =  $this->scopeConfig->getValue('erp_core/shipping/shipping_agent', 'website', $websiteId);
        if ($tableConfig) {
            $tableConfigResults = $this->unserializer->unserialize($tableConfig);
            if (is_array($tableConfigResults)) {
                foreach ($tableConfigResults as $tableConfigResult) {
                    if (key_exists('track_url', $tableConfigResult)) {
                        $url = $tableConfigResult['track_url'];
                    }
                    $magentoShipping = $tableConfigResult['magento_shipping'];
                    if (stripos($magentoShippingMethod, $magentoShipping) !== false) {
                        return $url;
                    }
                }
            }
        }
        return '';
    }

    public function getSalesPerson(int $websiteId) : ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/sales_person', 'website', $websiteId);
    }

    public function getProfitCenter(int $websiteId) : ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/profit_center', 'website', $websiteId);
    }

    public function getTermsOfDelivery(int $websiteId) : ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/terms_of_delivery', 'website', $websiteId);
    }

    public function getWayOfDelivery(int $websiteId) : ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/way_of_delivery', 'website', $websiteId);
    }

    public function getCustomerCategory(int $websiteId) : ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/customer_category', 'website', $websiteId);
    }

    public function getCustomerDistrict(int $websiteId) : ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/customer_district', 'website', $websiteId);
    }

    public function getOrderStatusSend($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : string
    {
        $status = $this->scopeConfig->getValue('erp_core/order/order_status_send', $scope);
        return $status === null ? '' : $status;
    }

    public function getOrderStatusSent($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : string
    {
        $status = $this->scopeConfig->getValue('erp_core/order/order_status_sent', $scope);
        return $status === null ? '' : $status;
    }

    public function sendInvoiceConfirmation($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : bool
    {
        return $this->scopeConfig->getValue('erp_core/order/send_invoice_confirmation', $scope) == '1';
    }

    public function getDefaultTaxCode(int $websiteId) : ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/default_tax_code', 'website', $websiteId);
    }

    public function getOrderPostingMode($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/posting_mode', $scope);
    }

    public function getLCY($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : string
    {
        $lcy = $this->scopeConfig->getValue('erp_core/currency/lcy_erp', $scope);

        return $lcy === null ? '' : $lcy;
    }

    public function getCurrency(?int $storeId) : string
    {
        $currency = $this->scopeConfig->getValue('erp_core/currency/currency', 'store', $storeId);
        if ($currency === null) {
            $currency = "";
        }
        return $currency;
    }

    public function releaseCronLock() : void
    {
        $this->configWriter->save(
            'erp_core/cron/locked',
            '0',
            $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT,
            $scopeId = 0
        );
    }

    public function requestCronLock(ErpLogger $logger) : bool
    {
        // IMPORTANT: The lock is read directly from the database to avoid problems with the cache manager.
        $isLocked = '1';
        $configCollection = $this->configCollectionFactory->create()
                                                            ->addFieldToSelect('path')
                                                            ->addFieldToSelect('value');
        $configCollection->addScopeFilter(ScopeConfigInterface::SCOPE_TYPE_DEFAULT, 0, 'erp_core');
        foreach ($configCollection->getData() as $setting) {
            if ($setting['path'] === 'erp_core/cron/locked') {
                $isLocked =   $setting['value'];
                break;
            }
        }

        if ($isLocked == '1') {
            $logger->warning('*** A lock was detected, another job may be running.');
            $logger->warning('*** This can happen if you manually start a job when another job is running.');
            $logger->warning('*** If this is the case, the lock is released automatically when' .
                             ' the current job is finished.');
            $logger->warning('*** It can also happen if a serious error occurred during the previous job,');
            $logger->warning('*** if this is the case, the cause MUST be investigated properly!');
            $logger->warning('*** You can manually unlock the jobs on the configuration screen.');
            return false;
        }

        $this->configWriter->save(
            'erp_core/cron/locked',
            '1',
            $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT,
            $scopeId = 0
        );
        return true;
    }

    public function stopOnError($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : bool
    {
        return $this->scopeConfig->getValue('erp_core/cron/stop_on_error', $scope) == '1';
    }

    public function getLogDirectory($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : ?string
    {
        return $this->scopeConfig->getValue('erp_core/cron/log_directory', $scope);
    }

    public function getNotificationReceiver($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : ?string
    {
        return $this->scopeConfig->getValue('erp_core/cron/receiver_notification', $scope);
    }

    public function getLogRetentionTime($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : ?string
    {
        return $this->scopeConfig->getValue('erp_core/cron/log_retention', $scope);
    }

    public function getCronProductImport($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : string
    {
        return $this->scopeConfig->getValue('erp_core/cron/cron_product_import', $scope) ?? '';
    }

    public function getCronOrderExport($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : string
    {
        return $this->scopeConfig->getValue('erp_core/cron/cron_order_export', $scope) ?? '';
    }

    public function getCronInvoiceExport($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : string
    {
        return $this->scopeConfig->getValue('erp_core/cron/cron_invoice_export', $scope) ?? '';
    }

    public function getCronShipmentImport($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : string
    {
        return $this->scopeConfig->getValue('erp_core/cron/cron_shipment_import', $scope) ?? '';
    }

    public function getCronInvoiceImport($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : string
    {
        return $this->scopeConfig->getValue('erp_core/cron/cron_invoice_import', $scope) ?? '';
    }

    public function getCronCreditMemoExport($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : string
    {
        return $this->scopeConfig->getValue('erp_core/cron/cron_credit_memo_export', $scope) ?? '';
    }

    public function getCronReturnImport($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT) : string
    {
        return $this->scopeConfig->getValue('erp_core/cron/cron_return_import', $scope) ?? '';
    }

    public function getRefundItemPosAdj(int $websiteId) : ?string
    {
        return $this->scopeConfig->getValue('erp_core/refund/adjustment_positive', 'website', $websiteId);
    }

    public function getRefundItemNegAdj(int $websiteId) : ?string
    {
        return $this->scopeConfig->getValue('erp_core/refund/adjustment_negative', 'website', $websiteId);
    }

    public function getDiscountMethod(int $storeId) : string
    {
        return $this->scopeConfig->getValue('erp_core/discount/method', 'store', $storeId) ?? self::DISCOUNT_LINE;
    }

    public function getDiscountErpItem(int $storeId) : string
    {
        $itemId = '';
        $itemConfig = $this->scopeConfig->getValue('erp_core/discount/item_id', 'store', $storeId);
        if ($itemConfig) {
            $itemId = explode('#', $itemConfig)[0];
        }
        return $itemId;
    }
}
