<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\ErpCore\Service;

use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\ErpAdapterInterface;
use Magento\Framework\App\ProductMetadataInterface;
use Magento\Store\Model\StoreManagerInterface;
use Symfony\Component\Console\Output\OutputInterface;

class ReturnService
{
    protected $logger;
    protected $notification;
    protected $erp;
    protected $settings;
    protected $storeManager;
    protected $productMetadata;

    public function __construct(
        ErpLogger $logger,
        Notification $notification,
        ErpAdapterInterface $erp,
        Settings $settings,
        StoreManagerInterface $storeManager,
        ProductMetadataInterface $productMetadata
    ) {
        $this->logger = $logger;
        $this->notification = $notification;
        $this->erp = $erp;
        $this->settings = $settings;
        $this->storeManager = $storeManager;
        $this->productMetadata = $productMetadata;
    }

    public function executeFromShell(OutputInterface $output, bool $forceUnlock): void
    {
        try {
            $this->logger->info('Starting import of returns from command line');
            $this->logger->setOutput($output);
            $this->import($forceUnlock);
        } catch (\Throwable $t) {
            $this->logger->critical($t->getTraceAsString());
        }
    }

    public function executeFromCron(bool $forceUnlock = false): void
    {
        try {
            $this->logger->info('Starting import of returns from cron job');
            $this->import($forceUnlock);
        } catch (\Throwable $t) {
            $this->logger->critical($t->getTraceAsString());
        }
    }

    public function executeFromWeb(bool $forceUnlock = false): void
    {
        try {
            $this->logger->info('Starting import of returns from web');
            $this->import($forceUnlock);
        } catch (\Throwable $t) {
            $this->logger->critical($t->getTraceAsString());
        }
    }

    private function import(bool $forceUnlock = false): void
    {
        if ('Enterprise' !== $this->productMetadata->getEdition()) {
            $this->logger->error('Returns are not available in Magento ' . $this->productMetadata->getEdition() . ' edition.');

            return;
        }

        if ($forceUnlock) {
            $this->settings->releaseCronLock();
        }

        if (false == $this->settings->requestCronLock($this->logger)) {
            return;
        }

        $capabilities = $this->erp->getCapabilities();
        foreach ($capabilities as $capability) {
            if (!$capability->isCapabilitySupported(Capability::RETURN_IMPORT)) {
                $this->logger->warning('Import of returns are not supported by the ' . $capability->getName() . ' integration');
                $this->settings->releaseCronLock();

                return;
            }
        }

        $start_time = microtime(true);

        $errorMessage = '';

        try {
            $this->logger->info('Retrieving returns from ERP');
            $json = $this->erp->getReturns();
            foreach ($json as $item) {
                $this->logger->info('Importing return for order ' . $item->webOrderNumber . ', sku ' . $item->sku . ', qty ' . $item->qty);
                $this->processReturn(
                    $this->logger,
                    $item->erpOrderNumber,
                    $item->webOrderNumber,
                    $item->receivedDate,
                    $item->reasonCode,
                    $item->comment,
                    $item->sku,
                    $item->qty
                );
                $this->erp->setReturnAsProcessed($item->id);
            }
        } catch (\Laminas\Http\Exception\RuntimeException $exception) {
            $errorMessage = $errorMessage . $exception->getMessage();
            $this->logger->error($exception->getMessage());
        } catch (\Magento\Framework\Exception\LocalizedException $exception) {
            $errorMessage = $errorMessage . $exception->getMessage();
            $this->logger->error($exception->getMessage());
        } catch (\RuntimeException $exception) {
            $errorMessage = $errorMessage . $exception->getMessage();
            $this->logger->error($exception->getMessage());
        }

        if ('' !== $errorMessage && $this->settings->stopOnError()) {
            $this->notification->notify($errorMessage);

            return;
        }
        $this->logger->info('Import completed in ' . (microtime(true) - $start_time) . ' sec');
        $this->settings->releaseCronLock();

        if ('' !== $errorMessage) {
            $this->notification->notify($errorMessage);
        }
    }

    public function processReturn(
        ErpLogger $logger,
        string $erpOrderNumber,
        string $webOrderNumber,
        string $receivedDate,
        string $reasonCode,
        string $comment,
        string $sku,
        int $qty
    ) : void {
        // Override this method in plugin
    }
}
