<?php declare(strict_types=1);
/**
 *
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'
 *
 *           Copyright © eComero Management AB, All rights reserved.
 *
 */
namespace Ecomero\ErpCore\Test\Unit;

use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Attribute;
use Ecomero\ErpCore\Model\Erp;
use Ecomero\ErpCore\Model\Item;
use Ecomero\ErpCore\Model\ResourceModel\Item\Collection;
use Ecomero\ErpCore\Model\ResourceModel\Item\CollectionFactory;
use Magento\Catalog\Model\ProductFactory;
use Magento\Catalog\Model\ProductRepository;
use Magento\Catalog\Model\ResourceModel\Product;
use Magento\CatalogInventory\Model\StockRegistry;
use Magento\ConfigurableProduct\Helper\Product\Options\Factory;
use Magento\Store\Model\StoreManager;

class ProductServiceTest extends \PHPUnit\Framework\TestCase
{
    protected $productServiceClass;
    protected $productAttributeMock;
    protected $settingsMock;
    protected $erpMock;
    protected $productRepositoryMock;
    protected $productResourceModelMock;
    protected $productMock;

    private function createErpLoggerMock() : ErpLogger
    {
        /** @var ErpLogger&\PHPUnit\Framework\MockObject\MockObject $loggerMock */
        $loggerMock = $this->getMockBuilder(ErpLogger::class)
        ->disableOriginalConstructor()
        ->getMock();
        return $loggerMock;
    }

    private function createNotificationMock() : Notification
    {
        /** @var Notification&\PHPUnit\Framework\MockObject\MockObject $notificationMock */
        $notificationMock = $this->getMockBuilder(Notification::class)
                ->disableOriginalConstructor()
                ->setMethods(['notify'])
                ->getMock();
        return $notificationMock;
    }

    private function createErpMock() : Erp
    {
        /** @var Erp&\PHPUnit\Framework\MockObject\MockObject $erpMock */
        $erpMock = $this->getMockBuilder(Erp::class)
                ->disableOriginalConstructor()
                ->setMethods(['setErpOrderNo', 'setWebsiteFromStoredId', 'isErpEnabled', 'getWebsite', 'createDocument', 'updateDocument', 'getDocumentFromExternalId'])
                ->getMock();
        $orderResponseMock = new \stdClass();
        $orderResponseMock->id = '1';
        $orderResponseMock->number = '123';
        $orderResponseMock->customerNumber = '1';
        $erpMock->method('isErpEnabled')->willReturn('MockErp');
        $erpMock->method('getWebsite')->willReturn(1);
        $erpMock->method('createDocument')->willReturn($orderResponseMock);
        $erpMock->method('updateDocument')->willReturn($orderResponseMock);

        return $erpMock;
    }

    private function createProductFactoryMock() : ProductFactory
    {
        /** @var ProductFactory&\PHPUnit\Framework\MockObject\MockObject $productFactoryMock */
        $productFactoryMock = $this->getMockBuilder(ProductFactory::class)
                ->setMethods(['create'])
                ->disableOriginalConstructor()
                ->getMock();

        $productFactoryMock->method('create')->willReturn($this->productMock);

        return $productFactoryMock;
    }

    private function createProductRepositoryMock() : ProductRepository
    {
        $typeInstanceMock =  $this->getMockBuilder(\Magento\ConfigurableProduct\Model\Product\Type\Configurable::class)
            ->setMethods(['getUsedProducts'])
            ->disableOriginalConstructor()
            ->getMock();

        $this->productMock = $this->getMockBuilder(\Magento\Catalog\Model\Product::class)
            ->setMethods(['getCustomAttribute', 'setWebsiteIds', 'setStoreId', 'getCost', 'setCost', 'getWeight', 'setWeight', 'setStockData', 'getId', 'getTypeInstance', 'setAssociatedProductIds'])
            ->disableOriginalConstructor()
            ->getMock();

        $this->productMock->method('getCost')->willReturn(10.58);
        $this->productMock->method('getWeight')->willReturn(2.13);
        $this->productMock->method('getId')->willReturn(2);
        $this->productMock->method('getTypeInstance')->willReturn($typeInstanceMock);
        $typeInstanceMock->method('getUsedProducts')->willReturn([$this->productMock, $this->productMock, $this->productMock]);

        /** @var ProductRepository&\PHPUnit\Framework\MockObject\MockObject $productRepositoryMock */
        $productRepositoryMock = $this->getMockBuilder(ProductRepository::class)
            ->setMethods(['get', 'save'])
            ->disableOriginalConstructor()
            ->getMock();

        return $productRepositoryMock;
    }

    private function createProductResourceModelMock() : Product
    {
        /** @var Product&\PHPUnit\Framework\MockObject\MockObject $productResourceModelMock */
        $productResourceModelMock = $this->getMockBuilder(Product::class)
            ->disableOriginalConstructor()
            ->getMock();

        return $productResourceModelMock;
    }

    private function createProductOptionsFactoryMock() : Factory
    {
        /** @var Factory&\PHPUnit\Framework\MockObject\MockObject $productOptionsFactoryMock */
        $productOptionsFactoryMock = $this->getMockBuilder(Factory::class)
            ->disableOriginalConstructor()
            ->getMock();

        return $productOptionsFactoryMock;
    }

    private function createSettingsMock() : Settings
    {
        /** @var Settings&\PHPUnit\Framework\MockObject\MockObject $settingsMock */
        $settingsMock = $this->getMockBuilder(Settings::class)
            ->disableOriginalConstructor()
            ->setMethods(['requestCronLock', 'getCategoryMapping', 'releaseCronLock', 'stopOnError', 'getLCY', 'getCurrency'])
            ->getMock();

        $category = [];
        $category['category_filter'] = 'HY_';
        $category['product_category'] = 'Jackets';
        $category['attribute_set'] = 'Garments';

        $settingsMock->method('getCategoryMapping')->willReturn([$category]);
        $settingsMock->method('getLCY')->willReturn('SEK');
        $settingsMock->method('getCurrency')->willReturn('SEK', 'GBP', 'EUR', 'SEK', 'GBP', 'EUR');

        return $settingsMock;
    }

    private function createStoreManagerMock() : StoreManager
    {
        $websiteMock = $this->getMockBuilder(\Magento\Store\Model\Website::class)
            ->setMethods(['getId', 'getName'])
            ->disableOriginalConstructor()
            ->getMock();

        $websiteMock->method('getId')->willReturn(1);
        $websiteMock->method('getName')->willReturn('testcompany');

        $storeMock = $this->getMockBuilder(\Magento\Store\Model\Store::class)
        ->disableOriginalConstructor()
        ->getMock();

        /** @var StoreManager&\PHPUnit\Framework\MockObject\MockObject $storeManagerMock */
        $storeManagerMock = $this->getMockBuilder(StoreManager::class)
            ->setMethods(['getWebsites', 'getWebsite', 'getStores' , 'getDefaultStoreView'])
            ->disableOriginalConstructor()
            ->getMock();

        $storeManagerMock->method('getWebsites')->willReturn([$websiteMock]);
        $storeManagerMock->method('getWebsite')->willReturn($websiteMock);
        $storeManagerMock->method('getStores')->willReturn([$storeMock, $storeMock, $storeMock]);
        $storeManagerMock->method('getDefaultStoreView')->willReturn($storeMock);

        return $storeManagerMock;
    }

    private function createStockRegistryMock() : StockRegistry
    {
        /** @var StockRegistry&\PHPUnit\Framework\MockObject\MockObject $stockRegistryMock */
        $stockRegistryMock = $this->getMockBuilder(StockRegistry::class)
            ->disableOriginalConstructor()
            ->getMock();

        return $stockRegistryMock;
    }

    private function createItemCollectionFactoryMock() : CollectionFactory
    {
        $attributeMock = $this->getMockBuilder(Attribute::class)
            ->disableOriginalConstructor()
            ->getMock();

        $attributeMock->value = '123434-234234-324432545';

        $itemMock = $this->getMockBuilder(Item::class)
            ->setMethods(['sku', 'getName', 'getPrice', 'attributes', 'getCategoryLabel', 'getWebsiteIds', 'getCategoryIds'])
            ->disableOriginalConstructor()
            ->getMock();

        $itemMock->method('getCategoryLabel')->willReturn('HY_123');
        $itemMock->method('getName')->willReturn('My Test SKU');
        $itemMock->method('getPrice')->willReturn(55.80);
        $itemMock->method('getWebsiteIds')->willReturn([1,2]);
        $itemMock->method('getCategoryIds')->willReturn([]);

        $itemMock->attributes['erp_id'] = $attributeMock;
        $itemMock->sku = 'sku-test-123';
        $itemMock->prices['gbp'] = 20;
        $itemMock->cost = 10;
        $itemMock->weight = 2.0;
        $itemMock->description = 'test-prod';
        $itemMock->magentoId = 2;

        $itemCollectionMock = $this->getMockBuilder(Collection::class)
            ->setMethods(['initItems','initItemsPrices','initItemsAttributes', 'count', 'getItems'])
            ->disableOriginalConstructor()
            ->getMock();
        $itemCollectionMock->method('count')->willReturn(2);
        $itemCollectionMock->method('getItems')->willReturn([$itemMock, $itemMock]);

        /** @var CollectionFactory&\PHPUnit\Framework\MockObject\MockObject $itemCollectionFactoryMock */
        $itemCollectionFactoryMock = $this->getMockBuilder(CollectionFactory::class)
            ->setMethods(['create'])
            ->disableOriginalConstructor()
            ->getMock();

        $itemCollectionFactoryMock->method('create')->willReturn($itemCollectionMock);

        return $itemCollectionFactoryMock;
    }

    public function setUp() : void
    {
        $this->settingsMock = $this->createSettingsMock();
        $this->erpMock = $this->createErpMock();
        $this->productRepositoryMock = $this->createProductRepositoryMock();
        $this->productResourceModelMock = $this->createProductResourceModelMock();

        /** @var \Magento\Catalog\Model\Attribute&\PHPUnit\Framework\MockObject\MockObject $this->productAttributeMock */
        $this->productAttributeMock = $this->getMockBuilder(\Magento\Catalog\Model\Attribute::class)
                ->disableOriginalConstructor()
                ->setMethods(['getValue'])
                ->getMock();

        $this->productServiceClass = new \Ecomero\ErpCore\Service\ProductService(
            $this->createErpLoggerMock(),
            $this->createNotificationMock(),
            $this->erpMock,
            $this->createProductFactoryMock(),
            $this->productRepositoryMock,
            $this->productResourceModelMock,
            $this->createProductOptionsFactoryMock(),
            $this->settingsMock,
            $this->createStoreManagerMock(),
            $this->createStockRegistryMock(),
            $this->createItemCollectionFactoryMock()
        );
    }

    public function testGetErpItemInfoFromSku() : void
    {
        $this->productRepositoryMock->expects($this->once())
            ->method('get')
            ->willReturn($this->productMock);

        $this->productMock->expects($this->at(0))
            ->method('getCustomAttribute')
            ->willReturn($this->productAttributeMock);

        $this->productMock->expects($this->at(1))
            ->method('getCustomAttribute')
            ->willReturn($this->productAttributeMock);

        $this->productMock->expects($this->at(2))
            ->method('getCustomAttribute')
            ->willReturn($this->productAttributeMock);

        $this->productServiceClass->getErpItemInfoFromSKU('SKU123');
    }

    public function testGetErpItemInfoFromSkuNoSku() : void
    {
        $this->productRepositoryMock->expects($this->once())
            ->method('get')
            ->will(
                $this->throwException(
                    new \Magento\Framework\Exception\NoSuchEntityException()
                )
            );

        try {
            $this->productServiceClass->getErpItemInfoFromSKU('SKU123');
            $this->fail("Expected NoSuchEntityException not thrown");
        } catch (\RuntimeException $e) {
            $this->assertEquals("Error can not get (or find) sku SKU123 in Magento", $e->getMessage());
        }
    }

    public function testGetErpItemInfoFromSkuMissingItemId() : void
    {
        $this->productRepositoryMock->expects($this->once())
            ->method('get')
            ->willReturn($this->productMock);

        $this->productMock->expects($this->once())
            ->method('getCustomAttribute')
            ->willReturn(null);

        try {
            $this->productServiceClass->getErpItemInfoFromSKU('SKU123');
            $this->fail("Expected RuntimeException not thrown");
        } catch (\RuntimeException $e) {
            $this->assertEquals("Error can not get Erp Article/Item Id from Magento product, is the attribute missing on the sku? (SKU123)", $e->getMessage());
        }
    }

    public function testGetErpItemInfoFromSkuMissingVatRate() : void
    {
        $this->productRepositoryMock->expects($this->once())
            ->method('get')
            ->willReturn($this->productMock);

        $this->productMock->expects($this->at(0))
            ->method('getCustomAttribute')
            ->willReturn($this->productAttributeMock);

        try {
            $this->productServiceClass->getErpItemInfoFromSKU('SKU123');
            $this->fail("Expected RuntimeException not thrown");
        } catch (\RuntimeException $e) {
            $this->assertEquals("Error can not get Erp Article/Item VAT Rate from Magento product, is the attribute missing on the sku? (SKU123)", $e->getMessage());
        }
    }

    public function testGetErpItemInfoFromSkuMissingVatIncluded() : void
    {
        $this->productRepositoryMock->expects($this->once())
            ->method('get')
            ->willReturn($this->productMock);

        $this->productMock->expects($this->at(0))
            ->method('getCustomAttribute')
            ->willReturn($this->productAttributeMock);

        $this->productMock->expects($this->at(1))
            ->method('getCustomAttribute')
            ->willReturn($this->productAttributeMock);

        try {
            $this->productServiceClass->getErpItemInfoFromSKU('SKU123');
            $this->fail("Expected RuntimeException not thrown");
        } catch (\RuntimeException $e) {
            $this->assertEquals("Error can not get Erp Article/Item VAT Included from Magento product, is the attribute missing on the sku? (SKU123)", $e->getMessage());
        }
    }

    public function testImportCron() : void
    {
        $this->settingsMock->expects($this->once())
            ->method('requestCronLock')
            ->willReturn(true);

        $this->productRepositoryMock->expects($this->at(0))
            ->method('get')
            ->will(
                $this->throwException(
                    new \Magento\Framework\Exception\NoSuchEntityException()
                )
            );

        $this->productRepositoryMock->expects($this->exactly(6))
            ->method('get')
            ->willReturn($this->productMock);

        $this->productServiceClass->importCron();
    }

    public function testImportWeb() : void
    {
        $this->productServiceClass->importWeb(true);
    }

    public function testImportCommandLine() : void
    {
        /** @var \Symfony\Component\Console\Output\ConsoleOutput&\PHPUnit\Framework\MockObject\MockObject $outputMock */
        $outputMock = $this->getMockBuilder(\Symfony\Component\Console\Output\ConsoleOutput::class)
            ->disableOriginalConstructor()
            ->getMock();
        $this->productServiceClass->importCommandLine($outputMock, false);
    }
}
