<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Model\Order;

use Ecomero\ErpCore\Helper\CronLock;
use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\OrderExportHelper;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Helper\TaxItemHelper;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpOrderInterface;
use Magento\Sales\Model\ResourceModel\Order as OrderRepository;

class OrderExport extends \Ecomero\ErpCore\Model\Executor
{
    public const DOCUMENT_TYPE = 'Order';

    protected $orderExportHelper;
    protected $taxItemHelper;
    protected $settings;
    protected $erp;
    protected $orderRepository;

    public function __construct(
        ErpLogger $logger,
        Notification $notification,
        ErpOrderInterface $erp,
        CronLock $cronLock,
        OrderExportHelper $orderExportHelper,
        TaxItemHelper $taxItemHelper,
        Settings $settings,
        OrderRepository $orderRepository
    ) {
        parent::__construct(
            $logger,
            $cronLock,
            $notification,
            $erp
        );

        $this->settings = $settings;
        $this->taxItemHelper = $taxItemHelper;
        $this->orderExportHelper = $orderExportHelper;
        $this->erp = $erp;
        $this->orderRepository = $orderRepository;
    }

    public function processOrder(\Magento\Sales\Model\Order $order): void
    {
        if ($this->orderExportHelper->isDocumentAlreadyInErp($order->getIncrementId())) {
            return;
        }

        // Check if we already got an order number, then never export
        if ($order->getErpOrderNo()) {
            $this->logger->error('Found existing order number, can not export again '.$order->getErpOrderNo());

            return;
        }

        $storeId = (int) $order->getStoreId();
        $this->erp->setWebsiteFromStoredId($storeId);

        $orderResponse = $this->orderExportHelper->createNewDocument(
            $order,
            $order->getOrderCurrencyCode()
        );

        $this->addDiscountLine($order, $orderResponse->getId());

        $numOrderLines = $this->orderExportHelper->addDocumentLines($order, $orderResponse, $order->getId());

        $hasShipping = $this->orderExportHelper->addShippingCharges($order, $orderResponse, $order->getId(), $numOrderLines);

        // Update order number to "tag" order as completely transferred
        $orderResponse = $this->erp->updateDocument(
            self::DOCUMENT_TYPE,
            $orderResponse->getId(),
            $order->getIncrementId()
        );

        // Post order (bokför order)
        if (
            'ALWAYS' === $this->settings->getOrderPostingMode()
            || ('NON-SHIP' === $this->settings->getOrderPostingMode() && !$hasShipping)
        ) {
            $this->erp->postOrder($orderResponse->getId());
        }

        $orderStatusMarkSent = $this->settings->getOrderStatusSent();
        $order->setStatus($orderStatusMarkSent);
        $order->setState(\Magento\Sales\Model\Order::STATE_PROCESSING);

        $order->addStatusToHistory(
            $orderStatusMarkSent,
            self::DOCUMENT_TYPE.' sucessfully transferred to Erp with id '.$orderResponse->getNumber()
        );
        $order->setErpOrderNo($orderResponse->getNumber());
        $order->setErpCustomerNo($orderResponse->getCustomerNumber());

        $this->orderRepository->save($order);
    }

    protected function run(): string
    {
        $this->orderExportHelper->setDocumentType(self::DOCUMENT_TYPE);

        return $this->orderExportHelper->iterateOrders(
            $this->settings->getOrderStatusSend(),
            [$this, 'processOrder']
        );
    }

    protected function getServiceDescription(): string
    {
        return 'order export';
    }

    protected function getCapability(): string
    {
        return Capability::ORDER_EXPORT;
    }

    protected function addDiscountLine(\Magento\Sales\Model\Order $order, string $erpOrderId): void
    {
        $storeId = (int) $order->getStoreId();
        $discountMethod = $this->settings->getDiscountMethod($storeId);
        $discountAmount = (float) $order->getDiscountAmount();
        $locationId = $this->settings->getWarehouseLocation($this->erp->getWebsite());

        if (\Ecomero\ErpCore\Helper\Settings::DISCOUNT_TOTAL === $discountMethod && 0 != $discountAmount) {
            if (!$this->taxItemHelper->getPriceIncludesTaxForFirstProduct($order)) {
                $discountAmount += (float) ($order->getDiscountTaxCompensationAmount());
            }
            $vatCode = $this->taxItemHelper->getTaxCodeForFirstProduct($order->getId());
            $couponCode = $order->getCouponCode() ?? '';
            $this->erp->addDocumentItem(
                self::DOCUMENT_TYPE,
                $erpOrderId,
                $this->settings->getDiscountErpItem($storeId),
                $discountAmount,
                1,
                $vatCode,
                0,
                0,
                $couponCode."\n",
                $locationId
            );
        }
    }
}
