<?php

declare(strict_types=1);

/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Model\Order;

use Ecomero\ErpCore\Helper\CronLock;
use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Helper\TaxItemHelper;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Catalog\CatalogImport;
use Ecomero\ErpCore\Model\Erp\ErpCreditMemoInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Sales\Api\CreditmemoRepositoryInterface;
use Magento\Sales\Model\ResourceModel\Order\Creditmemo\CollectionFactory;

class CreditMemoExport extends \Ecomero\ErpCore\Model\Executor
{
    public const ERROR_MARKER = 'ERROR see history log';
    protected $settings;
    protected $creditMemoCollectionFactory;
    protected $creditMemoRepository;
    protected $searchCriteriaBuilder;
    protected $catalogImport;
    protected $taxItemHelper;
    protected $erp;

    public function __construct(
        ErpLogger $logger,
        Notification $notification,
        ErpCreditMemoInterface $erp,
        Settings $settings,
        CronLock $cronLock,
        CollectionFactory $creditMemoCollectionFactory,
        CreditmemoRepositoryInterface $creditMemoRepository,
        SearchCriteriaBuilder $searchCriteriaBuilder,
        CatalogImport $catalogImport,
        TaxItemHelper $taxItemHelper
    ) {
        parent::__construct(
            $logger,
            $cronLock,
            $notification,
            $erp
        );

        $this->settings = $settings;
        $this->creditMemoCollectionFactory = $creditMemoCollectionFactory;
        $this->creditMemoRepository = $creditMemoRepository;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
        $this->catalogImport = $catalogImport;
        $this->taxItemHelper = $taxItemHelper;
        $this->erp = $erp;
    }

    public function addHandlingFeeLines(\Magento\Sales\Model\Order\Creditmemo $creditMemoDetails, string $creditMemoId)
    {
        $numOrderLines = 0;

        $order = $creditMemoDetails->getOrder();
        $orderNo = $order['entity_id'];
        $vatCode = '';
        if ($this->settings->getUseShippingTaxFromOrder()) {
            $vatCode = $this->taxItemHelper->getTaxCodeForFirstProduct($orderNo);
        } else {
            $vatCode = $this->taxItemHelper->getShippingTaxCodeForOrder($orderNo);
        }

        $amount = (float) $creditMemoDetails->getAdjustmentPositive();
        if (0 != $amount) {
            $this->logger->debug('  - Adding adjustment to refund : '.$amount);

            $posAdjItem = $this->settings->getRefundItemPosAdj($this->erp->getWebsite());
            $this->addAdjustmentLine(
                $posAdjItem,
                $creditMemoId,
                $amount,
                $vatCode
            );
            ++$numOrderLines;
        }

        $amount = 0 - (float) $creditMemoDetails->getAdjustmentNegative();
        if (0 != $amount) {
            $this->logger->debug('  - Adding fee to refund : '.$amount);

            $negAdjItem = $this->settings->getRefundItemNegAdj($this->erp->getWebsite());
            $this->addAdjustmentLine(
                $negAdjItem,
                $creditMemoId,
                $amount,
                $vatCode
            );
            ++$numOrderLines;
        }

        return $numOrderLines;
    }

    public function addCreditLines(\Magento\Sales\Model\Order\Creditmemo $creditMemoDetails, object $creditMemoResponse): int
    {
        $numOrderLines = 0;
        $locationId = $this->settings->getWarehouseLocation($this->erp->getWebsite());

        /** @var \Magento\Sales\Model\Order\Creditmemo\Item $item */
        foreach ($creditMemoDetails->getAllItems() as $item) {
            if ($item->getOrderItem()->getParentItem()) {
                continue;
            }

            $bcItem = $this->catalogImport->getErpItemInfoFromSKU($item->getSku());
            $vatCode = '';
            if ($bcItem['erp_id']) {
                $vatCode = $this->taxItemHelper->getTaxCodeForOrderLine(
                    $creditMemoDetails->getOrder()->getId(),
                    $item->getOrderItem()->getItemId()
                );
                $itemPrice = $item->getPrice();
                if ('1' == $bcItem['erp_tax_included']) {
                    $itemPrice = $item->getOrderItem()->getPriceInclTax();
                }

                $this->logger->debug('  - Processing credit memo '.$creditMemoResponse->number.
                                     ' ('.$creditMemoResponse->id.'), line item '.$item->getName().
                                     ', Price : '.$item->getPriceInclTax().
                                     ', Qty : '.$item->getQty().
                                     ', VAT: '.$vatCode);

                // Correct the discounts, we can not use both percentage and amounts,
                // If we have both types, the use the amount.
                $dicountAmount = (float) $item->getDiscountAmount();
                $dicountPercent = (float) $item->getDiscountPercent();
                if ($dicountAmount > 0 && $dicountPercent > 0) {
                    $dicountPercent = 0;
                }

                $rc = $this->erp->addCreditMemoItem(
                    $creditMemoResponse->number,
                    $bcItem['erp_id'],
                    (float) $itemPrice,
                    (float) $item->getQty(),
                    $vatCode,
                    $dicountAmount,
                    $dicountPercent,
                    $locationId
                );
                ++$numOrderLines;
            }
        }

        $giftCardAmount = $creditMemoDetails->getGiftCardsAmount();
        if ($giftCardAmount && $giftCardAmount > 0) {
            $erpGiftCardArticleId = $this->settings->getGiftCardItemId($this->erp->getWebsite());

            $rc = $this->erp->addCreditMemoItem(
                $creditMemoResponse->number,
                $erpGiftCardArticleId,
                (float) -$giftCardAmount,
                (float) 1,
                $vatCode,
                0,
                0,
                $locationId
            );

            ++$numOrderLines;
        }

        return $numOrderLines;
    }

    public function creditShippingCharges(\Magento\Sales\Model\Order\Creditmemo $creditMemoDetails, object $creditMemoResponse, int $numOrderLines): void
    {
        $order = $creditMemoDetails->getOrder();
        $orderNo = $order['entity_id'];
        $vatCode = '';
        if ($this->settings->getUseShippingTaxFromOrder()) {
            $vatCode = $this->taxItemHelper->getTaxCodeForFirstProduct($orderNo);
        } else {
            $vatCode = $this->taxItemHelper->getShippingTaxCodeForOrder($orderNo);
        }

        $magentoShippingMethod = $order->getShippingDescription();
        if (null === $magentoShippingMethod || $order->getIsVirtual()) {
            $this->logger->debug('  - *** No shipping descripton, virtual product?');

            return;
        }
        $locationId = $this->settings->getWarehouseLocation($this->erp->getWebsite());
        $shippingAgentService = $this->settings->getShippingChargeItem($this->erp->getWebsite(), $magentoShippingMethod);
        $shippingAgentServiceArray = explode('#', $shippingAgentService);
        $shipmentItemId = '';
        $shipmentItemType = '';
        $shipmentItemTaxIncluded = false;
        if (4 === count($shippingAgentServiceArray)) {
            $shipmentItemId = $shippingAgentServiceArray[0];
            $shipmentItemType = $shippingAgentServiceArray[1];
            $shipmentItemTaxIncluded = (bool) ('true' === $shippingAgentServiceArray[2]);
            // NOT USED - $shipmentItemTaxRate = $shippingAgentServiceArray[3];

            $shippingPrice = $creditMemoDetails->getBaseShippingAmount();
            if ($shipmentItemTaxIncluded) {
                $shippingPrice = $creditMemoDetails->getShippingInclTax();
            }

            if (0 == $shippingPrice) {
                $this->logger->debug('  - *** Free shipping ');

                return;
            }

            $this->erp->addCreditMemoItem(
                $creditMemoResponse->number,
                $shipmentItemId,
                (float) $shippingPrice,
                (float) 1,
                $vatCode,
                (float) 0,
                (float) 0,
                $locationId
            );
        } else {
            $this->logger->debug('  - *** No Shipping Agent Mapping found for '.$magentoShippingMethod);
        }
    }

    public function export(): string
    {
        $creditMemoCollecion = $this->creditMemoCollectionFactory->create();

        $creditMemoCollecion->getSelect()
            ->joinLeft(
                ['order' => $creditMemoCollecion->getTable('sales_order')],
                'main_table.order_id = order.entity_id',
                ['erp_customer_no' => 'order.erp_customer_no',
                    'order_number' => 'order.increment_id', ]
            )
            ->joinLeft(
                ['invoice' => $creditMemoCollecion->getTable('sales_invoice')],
                'main_table.order_id = invoice.order_id',
                ['erp_invoice_no' => 'invoice.erp_invoice_no']
            )
            ->where('order.erp_customer_no IS NOT NULL AND '.
                    'main_table.erp_creditmemo_no IS NULL')
        ;

        $creditMemoCollecion
            ->addFieldToSelect('entity_id')
            ->addFieldToSelect('created_at')
            ->addFieldToSelect('increment_id')
            ->addFieldToSelect('order_id')
            ->addFieldToSelect('order_currency_code')
        ;

        $errorMessage = '';
        foreach ($creditMemoCollecion->getData() as $creditMemo) {
            try {
                $creditMemoDetails = $this->creditMemoRepository->get($creditMemo['entity_id']);

                $storeId = (int) $creditMemoDetails->getStoreId();
                $this->erp->setWebsiteFromStoredId($storeId);
                $this->taxItemHelper->setDefaultTaxCode($this->settings->getDefaultTaxCode($this->erp->getWebsite()));
                $activeErp = $this->erp->isErpEnabled();
                if ('' === $activeErp) {
                    $this->logger->debug('Processing Magento credit memo '.$creditMemo['increment_id'].' no ERP integration active for website');

                    continue;
                }

                $this->logger->debug('Processing Magento credit memo '.$creditMemo['increment_id'].' for '.$activeErp.' integration');
                $bcCreditMemo = $this->erp->createCreditMemo(
                    substr($creditMemo['created_at'], 0, 10),
                    $creditMemo['order_number'],
                    $creditMemo['erp_invoice_no'],
                    $creditMemo['erp_customer_no'],
                    $creditMemo['order_currency_code']
                );

                $numOrderLines = $this->addCreditLines($creditMemoDetails, $bcCreditMemo);

                $numOrderLines += $this->addHandlingFeeLines($creditMemoDetails, $bcCreditMemo->id);

                $this->creditShippingCharges($creditMemoDetails, $bcCreditMemo, $numOrderLines);
                if ($this->settings->isPostCreditMemo($this->erp->getWebsite())) {
                    $this->erp->postCreditMemo($bcCreditMemo->id);
                }

                /** @var \Magento\Sales\Model\Order\Creditmemo $creditMemoDetails */
                $creditMemoDetails = $this->creditMemoRepository->get($creditMemo['entity_id']);
                $creditMemoDetails->setErpCreditmemoNo($bcCreditMemo->number);
                $this->creditMemoRepository->save($creditMemoDetails);
            } catch (\Laminas\Http\Exception\RuntimeException $exception) {
                $errorMessage = $errorMessage.$this->handleException($creditMemoDetails, $exception);
            } catch (\Magento\Framework\Exception\LocalizedException $exception) {
                $errorMessage = $errorMessage.$this->handleException($creditMemoDetails, $exception);
            } catch (\RuntimeException $exception) {
                $errorMessage = $errorMessage.$this->handleException($creditMemoDetails, $exception);
            }

            if ('' !== $errorMessage && $this->settings->stopOnError()) {
                $this->notification->notify($errorMessage);

                return '';
            }
        }

        return $errorMessage;
    }

    protected function run(): string
    {
        return $this->export();
    }

    protected function getServiceDescription(): string
    {
        return 'credit memo export';
    }

    protected function getCapability(): string
    {
        return Capability::CREDIT_MEMO_EXPORT;
    }

    private function addAdjustmentLine(string $adjItem, string $creditMemoNumber, float $amount, string $vatCode): ?object
    {
        $rc = null;
        $locationId = $this->settings->getWarehouseLocation($this->erp->getWebsite());
        $adjItemArray = explode('#', $adjItem);
        $itemId = '';
        $taxIncluded = false;
        if (3 === count($adjItemArray)) {
            $itemId = $adjItemArray[0];
//            $taxIncluded = (bool)($adjItemArray[1] === 'true');
//            $taxRate = $adjItemArray[2];

            $rc = $this->erp->addCreditMemoItem(
                $creditMemoNumber,
                $itemId,
                $amount,
                1,
                $vatCode,
                0,
                0,
                $locationId
            );
        }

        return $rc;
    }

    private function handleException($creditMemo, $exception): string
    {
        $errorMessage = $exception->getMessage();

        $this->logger->error($errorMessage);
        $creditMemo->addComment($errorMessage);
        $creditMemo->setErpCreditmemoNo(self::ERROR_MARKER);
        $creditMemo->save();

        return $errorMessage;
    }
}
