<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Model\ResourceModel\Item;

use Ecomero\ErpCore\Model\AttributeFactory;
use Ecomero\ErpCore\Model\ItemFactory;
use Magento\Framework\Stdlib\ArrayManager;

class Collection
{
    protected $collection;
    protected $itemFactory;
    protected $attributeFactory;
    protected $arrayManager;

    public function __construct(
        ItemFactory $itemFactory,
        AttributeFactory $attributeFactory,
        ArrayManager $arrayManager
    ) {
        $this->collection = [];
        $this->itemFactory = $itemFactory;
        $this->attributeFactory = $attributeFactory;
        $this->arrayManager = $arrayManager;
    }

    public function count(): int
    {
        return count($this->collection);
    }

    public function getItems(): array
    {
        return $this->collection;
    }

    public function initItems(array $json, int $websiteId, string $categoryId, string $categoryFilter, string $attribSetId): void
    {
        // Pre process JSON since it may contain duplicates
        foreach ($json as $item) {
            $sku = $this->arrayManager->get('sku', $item);
            if (false == array_key_exists($sku, $this->collection)) {
                $this->collection[$sku] = $this->itemFactory->create();
            }

            $this->collection[$sku]->sku = $sku;
            $this->collection[$sku]->category = $this->arrayManager->get('category', $item);
            $this->collection[$sku]->commonName = $this->arrayManager->get('commonName', $item);
            $this->collection[$sku]->description = $this->arrayManager->get('description', $item);
            $this->collection[$sku]->inventory = $this->arrayManager->get('inventory', $item);
            $this->collection[$sku]->cost = $this->arrayManager->get('cost', $item);
            $this->collection[$sku]->weight = $this->arrayManager->get('weight', $item);
            if ($this->arrayManager->get('isEnabledInErp', $item)) {
                $this->collection[$sku]->isEnabledFromErp = true;
                $this->collection[$sku]->isEnabled = $item->isEnabledInErp;
            } else {
                $this->collection[$sku]->isEnabledFromErp = false;
            }

            $this->collection[$sku]->attributeSet = $attribSetId;
            $this->collection[$sku]->addWebsite($websiteId);
            $this->collection[$sku]->addCategory($categoryId, $categoryFilter);

            // Create special attribute for Erp Internal Id
            if (false == array_key_exists('erp_id', $this->collection[$sku]->attributes)) {
                $this->collection[$sku]->attributes['erp_id'] = $this->attributeFactory->create();
                $this->collection[$sku]->attributes['erp_id']->name = 'erp_id';
                $this->collection[$sku]->attributes['erp_id']->type = 'Text';
                $this->collection[$sku]->attributes['erp_id']->value = $this->arrayManager->get('id', $item);
            }

            if (false == array_key_exists('erp_tax_rate', $this->collection[$sku]->attributes)) {
                $this->collection[$sku]->attributes['erp_tax_rate'] = $this->attributeFactory->create();
                $this->collection[$sku]->attributes['erp_tax_rate']->name = 'erp_tax_rate';
                $this->collection[$sku]->attributes['erp_tax_rate']->type = 'Text';
                $this->collection[$sku]->attributes['erp_tax_rate']->value = $this->arrayManager->get('taxRate', $item);
            }

            if (false == array_key_exists('erp_tax_included', $this->collection[$sku]->attributes)) {
                $this->collection[$sku]->attributes['erp_tax_included'] = $this->attributeFactory->create();
                $this->collection[$sku]->attributes['erp_tax_included']->name = 'erp_tax_included';
                $this->collection[$sku]->attributes['erp_tax_included']->type = 'Text';
                $this->collection[$sku]->attributes['erp_tax_included']->value = $this->arrayManager->get('priceIncludesVAT', $item)
                    ? '1' : '0';
            }

            if (false == array_key_exists('name', $this->collection[$sku]->attributes)) {
                $this->collection[$sku]->attributes['name'] = $this->attributeFactory->create();
                $this->collection[$sku]->attributes['name']->name = 'name';
                $this->collection[$sku]->attributes['name']->type = 'Text';
                $this->collection[$sku]->attributes['name']->value = $this->arrayManager->get('description', $item);
            }

            if (false == array_key_exists('price', $this->collection[$sku]->attributes)) {
                $this->collection[$sku]->attributes['price'] = $this->attributeFactory->create();
                $this->collection[$sku]->attributes['price']->name = 'price';
                $this->collection[$sku]->attributes['price']->type = 'Decimal';
                $this->collection[$sku]->attributes['price']->value = $this->arrayManager->get('price', $item);
            }
        }
    }

    public function initItemsAttributes(array $json): void
    {
        // Pre process JSON since it may contain duplicates
        foreach ($json as $item) {
            $sku = $this->arrayManager->get('sku', $item);
            if (false == array_key_exists($sku, $this->collection)) {
                $this->collection[$sku] = $this->itemFactory->create();
            }

            $name = strtolower($this->arrayManager->get('name', $item));
            if ('' != $name) {
                $this->collection[$sku]->attributes[$name] = $this->attributeFactory->create();
                $this->collection[$sku]->attributes[$name]->name = $name;
                $this->collection[$sku]->attributes[$name]->type = $this->arrayManager->get('type', $item);
                $this->collection[$sku]->attributes[$name]->value = $this->arrayManager->get('value', $item);
            }
        }

        // Update name from attribute values
        foreach ($this->collection as $key => $item) {
            $item->attributes['name']->value = $this->getNameWithAttributes($item);
        }
    }

    public function initItemsPrices(array $json): void
    {
        // Pre process JSON since it may contain duplicates
        foreach ($json as $item) {
            $sku = $this->arrayManager->get('sku', $item);
            if (false == array_key_exists($sku, $this->collection)) {
                $this->collection[$sku] = $this->itemFactory->create();
            }

            if ('' != $item->currency) {
                $this->collection[$sku]->prices[strtolower($item->currency)] = $this->arrayManager->get('priceCurrency', $item);
            }
        }
    }

    private function getNameWithAttributes(\Ecomero\ErpCore\Model\Item $item): string
    {
        $itemName = $item->description;

        foreach ($item->attributes as $attrib) {
            if (\Ecomero\ErpCore\Model\Attribute::YES === $attrib->usedInConfiguration) {
                $itemName = $itemName.'-'.strtoupper($attrib->value);
            }
        }

        return $itemName;
    }
}
