<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Helper;

use Ecomero\ErpCore\Model\Catalog\CatalogImport;
use Ecomero\ErpCore\Model\Erp\ErpOrderInterface;
use Ecomero\ErpCore\Model\Erp\ErpShippingInterface;
use Ecomero\ErpCore\Model\Order\Response as OrderResponse;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order\Status\HistoryFactory;
use Magento\Sales\Model\Order\Status\HistoryRepository;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory;

class OrderExportHelper
{
    protected $logger;
    protected $notification;
    protected $settings;
    protected $cronLock;
    protected $erp;
    protected $erpShipping;
    protected $orderCollectionFactory;
    protected $orderRepository;
    protected $customerHelper;
    protected $giftCardAccountHelper;
    protected $historyRepository;
    protected $historyFactory;
    protected $taxItemHelper;
    protected $catalogImport;
    protected $documentType;

    public function __construct(
        ErpLogger $logger,
        Notification $notification,
        Settings $settings,
        CronLock $cronLock,
        ErpOrderInterface $erp,
        ErpShippingInterface $erpShipping,
        CustomerHelper $customerHelper,
        CollectionFactory $orderCollectionFactory,
        OrderRepositoryInterface $orderRepository,
        HistoryRepository $historyRepository,
        HistoryFactory $historyFactory,
        TaxItemHelper $taxItemHelper,
        CatalogImport $catalogImport
    ) {
        $this->logger = $logger;
        $this->notification = $notification;
        $this->settings = $settings;
        $this->cronLock = $cronLock;
        $this->erp = $erp;
        $this->erpShipping = $erpShipping;
        $this->orderCollectionFactory = $orderCollectionFactory;
        $this->orderRepository = $orderRepository;
        $this->customerHelper = $customerHelper;
        $this->historyRepository = $historyRepository;
        $this->historyFactory = $historyFactory;
        $this->taxItemHelper = $taxItemHelper;
        $this->catalogImport = $catalogImport;
    }

    public function iterateOrders(string $orderStatusFilter, array $orderProcessor): string
    {
        $orderCollecion = $this->orderCollectionFactory->create()
            ->addFieldToSelect('entity_id')
            ->addFieldToSelect('increment_id')
            ->addFieldToSelect('customer_email')
            ->addFieldToSelect('order_currency_code')
            ->addFieldToSelect('erp_order_no')
        ;

        $errorMessage = '';
        $orderCollecion->addAttributeToFilter('status', ['in' => $orderStatusFilter]);
        foreach ($orderCollecion->getData() as $order) {
            $errorMessage = $this->sendOrder((int)$order['entity_id'], $orderProcessor);
        }

        return $errorMessage;
    }

    public function sendOrder(int $entityId, array $orderProcessor): string
    {
        $errorMessage = '';

        try {
            $fullOrderObj = $this->orderRepository->get($entityId);
            $storeId = (int) $fullOrderObj->getStoreId();
            $this->erp->setWebsiteFromStoredId($storeId);
            $this->taxItemHelper->setDefaultTaxCode($this->settings->getDefaultTaxCode($this->erp->getWebsite()));
            $activeErp = $this->erp->isErpEnabled();
            if ('' !== $activeErp) {
                $this->logger->debug('Processing Magento order '.$fullOrderObj->getIncrementId().' for '.$activeErp.' integration');
                call_user_func($orderProcessor, $fullOrderObj);
            } else {
                $this->logger->debug('Processing Magento order '.$fullOrderObj->getIncrementId().' no ERP integration active for website');
            }
        } catch (\Laminas\Http\Exception\RuntimeException $exception) {
            $errorMessage = $errorMessage.$this->handleException($fullOrderObj, $exception);
        } catch (\Magento\Framework\Exception\LocalizedException $exception) {
            $errorMessage = $errorMessage.$this->handleException($fullOrderObj, $exception);
        } catch (\RuntimeException $exception) {
            $errorMessage = $errorMessage.$this->handleException($fullOrderObj, $exception);
        }

        if ('' !== $errorMessage && $this->settings->stopOnError()) {
            $this->notification->notify($errorMessage);

            return '';
        }

        return $errorMessage;
    }

    public function addCommentToOrder(
        \Magento\Sales\Model\Order $order,
        string $comment
    ) {
        /** @var \Magento\Sales\Model\Order\Status\History $history */
        $history = $this->historyFactory->create();
        $history->setOrder($order);
        $history->setComment($comment);
        $history->setIsCustomerNotified(true);
        $this->historyRepository->save($history);
    }

    public function setDocumentType(string $documentType)
    {
        $this->documentType = $documentType;
    }

    public function isDocumentAlreadyInErp(string $magentoOrderId): bool
    {
        $rc = $this->erp->getDocumentFromExternalId(
            $this->documentType,
            'PENDING('.$magentoOrderId.')'
        );
        if (property_exists($rc, 'errorMessage')) {
            if (null !== $rc->errorMessage) {
                $this->logger->error($rc->errorMessage);

                return true;
            }
        }
        if (property_exists($rc, 'number')) {
            $this->logger->error($this->documentType.' with Magento id '.$magentoOrderId.' already exists in Erp.');

            return true;
        }

        return false;
    }

    public function createNewDocument(\Magento\Sales\Model\Order $order, string $orderCurrency): OrderResponse
    {
        /** @var \Magento\Sales\Model\Order\Address $billingAddress */
        $billingAddress = $order->getBillingAddress();
        $customer = $this->customerHelper->getCustomerFromAddress($billingAddress);

        $magentoShippingDescription = $order->getShippingDescription();
        $validShipingAddress = false;
        $shippingAgent = '';
        $shippingService = '';
        $wayOfDelivery = '';
        $shippingPickupLocationId = '';
        if ($magentoShippingDescription && !$order->getIsVirtual()) {
            $shippingAgentService = $this->settings->getShippingAgentService($this->erp->getWebsite(), $magentoShippingDescription);
            $shippingAgentServiceArray = explode('-', $shippingAgentService);
            if (2 == count($shippingAgentServiceArray)) {
                $shippingAgent = trim($shippingAgentServiceArray[0]);
                $shippingService = trim($shippingAgentServiceArray[1]);
            } else {
                $this->logger->debug('  - *** No Shipping Agent Mapping found for '.$magentoShippingDescription);
            }
            $magentoShippingMethod = $order->getShippingMethod(true);
            if ($magentoShippingMethod) {
                $carrier = $magentoShippingMethod->getCarrierCode();
                $method = str_replace('shipping_', '', $magentoShippingMethod->getMethod());
                $wayOfDelivery = $this->erpShipping->getWayOfDelivery($carrier, $method);
                $shippingAddress = $order->getShippingAddress();
                $shippingPickupLocationId = $order->getKssPickupLocationId(); // Klarna Shipping Service, field may not exist for non Klarna integrations

                // Workaround : When pickup location is used by Klarna,
                // the shippingAddress is set to the address of the pickup location (this is not correct)
                $validShipingAddress = !(isset($shippingPickupLocationId) && '' !== $shippingPickupLocationId);
            }
        }

        return $this->erp->createDocument(
            $this->documentType,
            $customer->getCustomerNumber(),
            $customer->getCompanyNumber(),
            $orderCurrency,
            $billingAddress,
            $validShipingAddress ? $shippingAddress : $billingAddress,
            $shippingAgent,
            $shippingService,
            $shippingPickupLocationId,
            'PENDING('.$order['increment_id'].')',
            $this->settings->getSalesPerson($this->erp->getWebsite()),
            $this->settings->getProfitCenter($this->erp->getWebsite()),
            $this->settings->getTermsOfDelivery($this->erp->getWebsite()),
            $wayOfDelivery,
            $this->settings->getCustomerCategory($this->erp->getWebsite()),
            $this->settings->getCustomerDistrict($this->erp->getWebsite())
        );
    }

    public function addDocumentLines(\Magento\Sales\Model\Order $orderDetails, OrderResponse $orderResponse, string $orderNo): int
    {
        $numOrderLines = 0;

        $locationId = $this->settings->getWarehouseLocation($this->erp->getWebsite());

        /** @var \Magento\Sales\Model\Order\Item $item */
        foreach ($orderDetails->getAllItems() as $item) {
            if (\Magento\Catalog\Model\Product\Type::TYPE_BUNDLE === $item->getProductType()) {
                $this->logger->debug("  - Item {$item->getSku()} is a product bundle, it should not be exported");

                continue;
            }

            if (null !== $item->getParentItem() && \Magento\ConfigurableProduct\Model\Product\Type\Configurable::TYPE_CODE === $item->getParentItem()->getProductType()) {
                continue;
            }

            $bcItem = $this->catalogImport->getErpItemInfoFromSKU($item->getSku());
            if ($bcItem['erp_id']) {
                $vatCode = $this->taxItemHelper->getTaxCodeForOrderLine($orderNo, $item->getItemId());
                $itemPrice = (float) $item->getPrice();
                if ('1' == $bcItem['erp_tax_included']) {
                    $itemPrice = $item->getPriceInclTax();
                }

                $this->logger->debug('  - Processing line for document '.$orderResponse->getNumber().
                                     ' ('.$orderResponse->getId().'), line item '.$item->getName().
                                     ', Price : '.$item->getPriceInclTax().
                                     ', Qty : '.$item->getQtyOrdered().
                                     ', VAT: '.$vatCode);

                $itemComment = '';
                if ($options = $item->getData('product_options')) {
                    if (isset($options['options'])) {
                        foreach ($options['options'] as $optionValues) {
                            if ($optionValues['value']) {
                                if ('' !== $itemComment) {
                                    $itemComment .= "\n";
                                }
                                $itemComment .= $optionValues['label']."\n".$optionValues['value'];
                            }
                        }
                    }
                }

                $storeId = (int) $orderDetails->getStoreId();
                if (\Ecomero\ErpCore\Helper\Settings::DISCOUNT_LINE === $this->settings->getDiscountMethod($storeId)) {
                    // Correct the discounts, we can not use both percentage and amounts,
                    // If we have both types, the use the amount.
                    if ('1' == $bcItem['erp_tax_included']) {
                        $discountAmount = (float) $item->getDiscountAmount();
                        $discountPercent = (float) $item->getDiscountPercent();
                    } else {
                        $discountAmount = (float) ($item->getDiscountAmount() - $item->getDiscountTaxCompensationAmount());
                        $discountPercent = (float) $item->getDiscountPercent();
                    }
                    if ($discountAmount > 0 && $discountPercent > 0) {
                        $discountPercent = 0;
                    }
                } else {
                    $discountAmount = 0;
                    $discountPercent = 0;
                }

                $this->erp->addDocumentItem(
                    $this->documentType,
                    $orderResponse->getNumber(),
                    $bcItem['erp_id'],
                    (float) $itemPrice,
                    (float) $item->getQtyOrdered(),
                    $vatCode,
                    $discountAmount,
                    $discountPercent,
                    $itemComment,
                    $locationId
                );
                ++$numOrderLines;
            }
        }

        try {
            $objectManager = \Magento\Framework\App\ObjectManager::getInstance();
            $this->giftCardAccountHelper = $objectManager->create('Magento\GiftCardAccount\Helper\Data');

            if ($this->giftCardAccountHelper) {
                $cards = $this->giftCardAccountHelper->getCards($orderDetails);

                foreach ($cards as $card) {
                    $amount = $card[\Magento\GiftCardAccount\Model\Giftcardaccount::AUTHORIZED];
                    $giftCardCode = $card[\Magento\GiftCardAccount\Model\Giftcardaccount::CODE];
                    $erpGiftCardArticleId = $this->settings->getGiftCardItemId($this->erp->getWebsite());

                    $this->erp->addDocumentItem(
                        $this->documentType,
                        $orderResponse->getNumber(),
                        $erpGiftCardArticleId,
                        (float) -$amount,
                        (float) 1,
                        $vatCode,
                        0,
                        0,
                        'Gift card: '.$giftCardCode,
                        $locationId
                    );
                    ++$numOrderLines;
                }
            }
        } catch (\Exception $e) {
        }

        return $numOrderLines;
    }

    public function addShippingCharges(\Magento\Sales\Model\Order $order, OrderResponse $orderResponse, string $orderNo, int $numOrderLines): bool
    {
        if ($order->getIsVirtual()) {
            $this->logger->debug('  - *** No shipping charges (virtual product)');

            return false;
        }

        $vatCode = '';
        $magentoShippingMethod = $order->getShippingDescription();
        if (!$magentoShippingMethod) {
            $this->logger->debug('  - *** No shipping description (virtual product)');

            return false;
        }

        $locationId = $this->settings->getWarehouseLocation($this->erp->getWebsite());

        if ($this->settings->getUseShippingTaxFromOrder()) {
            $vatCode = $this->taxItemHelper->getTaxCodeForFirstProduct($orderNo);
        } else {
            $vatCode = $this->taxItemHelper->getShippingTaxCodeForOrder($orderNo);
        }

        $shippingAgentService = $this->settings->getShippingChargeItem($this->erp->getWebsite(), $magentoShippingMethod);
        $shippingAgentServiceArray = explode('#', $shippingAgentService);
        $shipmentItemId = '';
        $shipmentItemType = '';
        $shipmentItemTaxIncluded = false;
        if (4 == count($shippingAgentServiceArray)) {
            $shipmentItemId = $shippingAgentServiceArray[0];
            $shipmentItemType = $shippingAgentServiceArray[1];
            $shipmentItemTaxIncluded = (bool) ('true' === $shippingAgentServiceArray[2]);

            $shippingPrice = $order->getBaseShippingAmount();
            if ($shipmentItemTaxIncluded) {
                $shippingPrice = $order->getShippingInclTax();
            }

            if (0 == $shippingPrice) {
                $this->logger->debug('  - *** No shipping charge exported (price = 0), ');
            } else {
                $this->erp->addDocumentItem(
                    $this->documentType,
                    $orderResponse->getNumber(),
                    $shipmentItemId,
                    (float) $shippingPrice,
                    1,
                    $vatCode,
                    0,
                    0,
                    null,
                    $locationId
                );
            }
        } else {
            $this->logger->debug('  - *** No Shipping Charge Mapping found for '.$magentoShippingMethod);
        }

        return true;
    }

    public function handleException($order, $exception): string
    {
        $errorMessage = $exception->getMessage();

        $this->logger->error($errorMessage);

        $orderStatusHistories = $order->getStatusHistories();
        $orderStatusHistory = reset($orderStatusHistories);
        if (false !== $orderStatusHistory
            && $orderStatusHistory->getComment() === $errorMessage) {
            $orderStatusHistory->setCreatedAt(new \DateTime());
            $errorMessage = ''; // Prevent error to be sent as a notification again
        } else {
            $order->addStatusHistoryComment($errorMessage);
        }

        // Change status (not the state) on the order so it is not resent until manually handled
        if ('ERROR' === substr($errorMessage, 0, 5)) {
            $order->setStatus('erp_error');
        }

        $order->save();

        return 'Store : '.$order->getStoreName()."\n".$errorMessage."\n";
    }
}
