<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Model\Order;

use Ecomero\ErpCore\Helper\CronLock;
use Ecomero\ErpCore\Helper\Email;
use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\OrderExportHelper;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpOrderInterface;
use Magento\Sales\Model\ResourceModel\Order as OrderRepository;

class OrderImport extends \Ecomero\ErpCore\Model\Executor
{
    public const DOCUMENT_TYPE = 'Order';

    protected $orderExportHelper;
    protected $emailHelper;
    protected $taxItemHelper;
    protected $settings;
    protected $erp;
    protected $orderRepository;

    public function __construct(
        ErpLogger $logger,
        Notification $notification,
        ErpOrderInterface $erp,
        CronLock $cronLock,
        OrderExportHelper $orderExportHelper,
        Settings $settings,
        Email $emailHelper,
        OrderRepository $orderRepository
    ) {
        parent::__construct(
            $logger,
            $cronLock,
            $notification,
            $erp
        );

        $this->settings = $settings;
        $this->emailHelper = $emailHelper;
        $this->orderExportHelper = $orderExportHelper;
        $this->erp = $erp;
        $this->orderRepository = $orderRepository;
    }

    public function processOrder(\Magento\Sales\Model\Order $order): void
    {
        $rc = $this->erp->getDocumentFromExternalId(
            self::DOCUMENT_TYPE,
            $order->getIncrementId()
        );

        if (property_exists($rc, 'deliveryDate')) {
            $deliveryDate = $rc->deliveryDate;
            $orderDeliveryDate = new \DateTime($order->getErpDeliveryDate());
            if (!is_null($deliveryDate) && ($orderDeliveryDate->format('Y-m-d') !== $deliveryDate->format('Y-m-d'))) {
                $order->setErpDeliveryDate($deliveryDate);

                try {
                    $this->orderRepository->save($order);
                    $this->emailHelper->sendOrderStatus($order);
                    $this->logger->info(".... new delivery date ({$deliveryDate->format('Y-m-d')}) saved on order.");
                } catch (\Throwable $th) {
                    $errorMessage = "{$order->getIncrementId()} : "."\n".$th->getMessage()."\n";
                    $this->logger->error(".... failed delivery date save on order {$errorMessage}.");
                }
            }
        }
    }

    protected function run(): string
    {
        $this->orderExportHelper->setDocumentType(self::DOCUMENT_TYPE);

        return $this->orderExportHelper->iterateOrders(
            $this->settings->getOrderStatusSent(),
            [$this, 'processOrder']
        );
    }

    protected function getServiceDescription(): string
    {
        return 'order import';
    }

    protected function getCapability(): string
    {
        return Capability::ORDER_IMPORT;
    }
}
