<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Helper;

class WebSocket
{
    protected const SERVER = 'ecomnext.ecomero.com';
    protected const MAX_TIMEOUT = 60 * 2;

    protected $app;
    protected $secret;
    protected $socketId;
    protected $messageId;

    public function __construct(
        Settings $settings
    ) {
        $this->app = $settings->getAppName();
        $this->secret = $settings->getAppSecret();
    }

    public function get(string $eventName)
    {
        $rc = [];
        $uri = 'wss://'.self::SERVER.'/app/'.$this->app.'?appKey='.$this->app;

        \Ratchet\Client\connect($uri)->then(function (\Ratchet\Client\WebSocket $conn) use (&$rc, $eventName) {
            $conn->on('message', function (\Ratchet\RFC6455\Messaging\MessageInterface $msg) use (&$conn, &$rc, $eventName) {
                $payload = json_decode($msg->getPayload(), true);
                if ('pusher:connection_established' === $payload['event']) {
                    $data = json_decode($payload['data'], true);
                    $this->socketId = $data['socket_id'];
                    $channel = 'private-erp-service';
                    $conn->send(json_encode([
                        'event' => 'pusher:subscribe',
                        'data' => [
                            'channel' => $channel,
                            'auth' => $this->getAuth($this->socketId, $channel, $this->secret),
                        ],
                    ]));
                }
                if ('pusher_internal:subscription_succeeded' === $payload['event']) {
                    \React\EventLoop\Loop::addTimer(self::MAX_TIMEOUT, function () use (&$conn) {
                        $conn->close();

                        throw new \RuntimeException('No response from server');
                    });
                    $this->messageId = uniqid($this->socketId);
                    $conn->send(json_encode(['event' => 'client-command', 'channel' => 'private-erp-service', 'data' => $eventName, 'msg-id' => $this->messageId]));
                }

                if ('pusher:error' === $payload['event']) {
                    throw new \RuntimeException($payload['data']['message']);
                }

                if ('client-command-response' === $payload['event']) {
                    if ($payload['msg-id'] === $this->messageId) {
                        $data = json_decode($payload['data'], true);
                        $rc = $data;
                        $conn->close();
                    }
                }
            });

            $conn->on('close', function ($code = null, $reason = null) {
                \React\EventLoop\Loop::stop();
            });

            $conn->send(json_encode(['event' => 'client-company', 'channel' => 'private-erp-service', 'data' => 'company']));
        }, function (\Exception $e) {
            echo "Could not connect: {$e->getMessage()}\n";
            \React\EventLoop\Loop::stop();
        });

        \React\EventLoop\Loop::run();

        return $rc;
    }

    private function getAuth(string $socketId, string $channelName, string $secret): string
    {
        $signature = "{$socketId}:{$channelName}";

        return $this->app.':'.hash_hmac('sha256', $signature, $secret);
    }
}
