<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Model\Catalog;

use Ecomero\ErpCore\Helper\CronLock;
use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpCatalogInterface;
use Magento\Catalog\Model\Product\Image\UrlBuilder;
use Magento\Catalog\Model\ProductFactory;
use Magento\Catalog\Model\ResourceModel\Product\CollectionFactory;
use Magento\ConfigurableProduct\Model\ResourceModel\Product\Type\ConfigurableFactory;
use Magento\Framework\App\ObjectManager;

class ImageExport extends \Ecomero\ErpCore\Model\Executor
{
    protected $settings;
    protected $productCollectionFactory;
    protected $mediaUrl;
    protected $imageUrlBuilder;
    protected $configurableFactory;
    protected $productFactory;
    protected $erp;

    public function __construct(
        UrlBuilder $urlBuilder = null,
        Notification $notification,
        ConfigurableFactory $configurableFactory,
        ProductFactory $productFactory,
        ErpLogger $logger,
        ErpCatalogInterface $erp,
        Settings $settings,
        CronLock $cronLock,
        CollectionFactory $productCollectionFactory
    ) {
        parent::__construct(
            $logger,
            $cronLock,
            $notification,
            $erp
        );

        $this->imageUrlBuilder = $urlBuilder ?? ObjectManager::getInstance()->get(UrlBuilder::class);
        $this->configurableFactory = $configurableFactory;
        $this->productFactory = $productFactory;
        $this->settings = $settings;
        $this->productCollectionFactory = $productCollectionFactory;
        $this->erp = $erp;
    }

    protected function run(): string
    {
        return $this->export();
    }

    protected function getServiceDescription(): string
    {
        return 'image export';
    }

    protected function getCapability(): string
    {
        return Capability::PRODUCT_PICTURE_EXPORT;
    }

    private function getImageUrl($product): string
    {
        $imageUrl = $this->getImageUrlForType($product, 'style_image');
        if ('' === $imageUrl) {
            $imageUrl = $this->getImageUrlForType($product, 'small_image');
        }

        return $imageUrl;
    }

    private function getImageUrlForType($product, string $imageType): string
    {
        $imageUrl = '';

        foreach ($product->getMediaGalleryImages() as $mediaGalleryImage) {
            $imageFile = $mediaGalleryImage->getFile();

            if ($product->getData($imageType) === $imageFile) {
                $imageUrl = $this->imageUrlBuilder->getUrl(
                    $imageFile,
                    'pagebuilder_product_image_admin'
                );

                break;
            }
        }

        return $imageUrl;
    }

    private function export(): string
    {
        $errorMessage = '';

        try {
            $productCollection = $this->productCollectionFactory->create();
            $productCollection->addAttributeToSelect('*');
            $productCollection->addAttributeToFilter('erp_id', ['notnull' => true]);

            foreach ($productCollection as $product) {
                $bcAttrib = $product->getCustomAttribute('erp_id');
                $itemId = $bcAttrib->getValue();

                $imageUrl = $this->getImageUrl($product);
                if ('' === $imageUrl) {
                    // Fallback, check image on parent product
                    $configProduct = $this->configurableFactory->create();
                    $parentArray = $configProduct->getParentIdsByChild($product->getId());
                    if (isset($parentArray[0])) {
                        $parentProduct = $this->productFactory->create()->load($parentArray[0]);
                        $imageUrl = $this->getImageUrl($parentProduct);
                    }
                }

                $rc = $this->erp->addPicture($itemId, $imageUrl);
                if ($rc) {
                    $this->logger->debug('  - Exporting image '.$imageUrl.
                                         ' for "'.$product->getName().
                                         '" ('.$itemId.') status code '.$rc);
                } else {
                    $this->logger->debug('  - Failed exporting image '.$imageUrl.
                                         ' for "'.$product->getName().
                                         '" ('.$itemId.')');
                }
            }
        } catch (\Laminas\Http\Exception\RuntimeException $exception) {
            $this->logger->error($exception->getMessage());
            $errorMessage = $exception->getMessage();
        } catch (\Throwable $t) {
            $this->logger->critical((string) $t);
            $errorMessage = $t->getMessage();
        }

        return $errorMessage;
    }
}
