<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Model\Invoice;

use Ecomero\ErpCore\Helper\CronLock;
use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\OrderExportHelper;
use Ecomero\ErpCore\Helper\OrderProcessor;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Helper\TaxItemHelper;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpOrderInterface;

class InvoiceExport extends \Ecomero\ErpCore\Model\Executor
{
    public const DOCUMENT_TYPE = 'Invoice';

    protected $orderExportHelper;
    protected $orderProcessor;
    protected $taxItemHelper;
    protected $settings;
    protected $erp;

    public function __construct(
        ErpLogger $logger,
        CronLock $cronLock,
        Notification $notification,
        ErpOrderInterface $erp,
        OrderExportHelper $orderExportHelper,
        OrderProcessor $orderProcessor,
        TaxItemHelper $taxItemHelper,
        Settings $settings
    ) {
        parent::__construct(
            $logger,
            $cronLock,
            $notification,
            $erp
        );

        $this->taxItemHelper = $taxItemHelper;
        $this->settings = $settings;
        $this->orderExportHelper = $orderExportHelper;
        $this->orderProcessor = $orderProcessor;
        $this->erp = $erp;
    }

    public function processOrder(\Magento\Sales\Model\Order $order): void
    {
        $this->orderExportHelper->initialize(self::DOCUMENT_TYPE, (int) $order->getStoreId());

        if ($this->orderExportHelper->isDocumentAlreadyInErp($order['increment_id'])) {
            return;
        }

        $orderResponse = $this->orderExportHelper->createNewDocument(
            $order,
            $order['order_currency_code']
        );

        $numOrderLines = $this->orderExportHelper->addDocumentLines($order, $orderResponse, $order['entity_id']);

        $this->orderExportHelper->addShippingCharges($order, $orderResponse, $order['entity_id'], $numOrderLines);

        // Update order number to "tag" order as completely transferred
        $this->erp->updateDocument(
            self::DOCUMENT_TYPE,
            $orderResponse->getId(),
            $order['increment_id']
        );

        $orderStatusMarkSent = $this->settings->getOrderStatusSent();
        $order->setStatus($orderStatusMarkSent);
        $order->addStatusToHistory(
            $orderStatusMarkSent,
            self::DOCUMENT_TYPE.' sucessfully transferred to Erp with id '.$orderResponse->getNumber()
        );
        $order->setErpOrderNo($orderResponse->getNumber());
        $order->setErpCustomerNo($orderResponse->getCustomerNumber());
        $order->save();
    }

    protected function run(): string
    {
        return $this->orderProcessor->iterateOrders(
            $this->settings->getOrderStatusSend(),
            [$this, 'processOrder']
        );
    }

    protected function getServiceDescription(): string
    {
        return 'invoice export';
    }

    protected function getCapability(): string
    {
        return Capability::INVOICE_EXPORT;
    }
}
