<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Model\Catalog;

use Ecomero\ErpCore\Helper\CronLock;
use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpCatalogInterface;
use Ecomero\ErpCore\Model\ResourceModel\Item\CollectionFactory;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product\Attribute\Source\Status;
use Magento\Catalog\Model\Product\Type;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Catalog\Model\ProductFactory;
use Magento\Catalog\Model\ResourceModel\Product;
use Magento\CatalogInventory\Api\StockRegistryInterface;
use Magento\ConfigurableProduct\Helper\Product\Options\Factory;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\StoreManagerInterface;

class CatalogImport extends \Ecomero\ErpCore\Model\Executor
{
    protected $productFactory;
    protected $productRepository;
    protected $productResourceModel;
    protected $optionsFactory;
    protected $settings;
    protected $storeManager;
    protected $stockRegistry;
    protected $itemCollectionFactory;
    protected $erp;
    protected $isImportRunning = false;

    public function __construct(
        ErpLogger $logger,
        Notification $notification,
        ErpCatalogInterface $erp,
        ProductFactory $productFactory,
        ProductRepositoryInterface $productRepository,
        Product $productResourceModel,
        Factory $optionsFactory,
        Settings $settings,
        CronLock $cronLock,
        StoreManagerInterface $storeManager,
        StockRegistryInterface $stockRegistry,
        CollectionFactory $itemCollectionFactory
    ) {
        parent::__construct(
            $logger,
            $cronLock,
            $notification,
            $erp
        );

        $this->erp = $erp;
        $this->productFactory = $productFactory;
        $this->productRepository = $productRepository;
        $this->productResourceModel = $productResourceModel;
        $this->optionsFactory = $optionsFactory;
        $this->settings = $settings;
        $this->storeManager = $storeManager;
        $this->stockRegistry = $stockRegistry;
        $this->itemCollectionFactory = $itemCollectionFactory;

        $this->erp->setLogger($logger);
    }

    public function isImportRunning()
    {
        return $this->isImportRunning;
    }

    public function getErpItemInfoFromSKU($sku): array
    {
        $itemInfo = [];

        try {
            $product = $this->productRepository->get($sku);
            $bcAttrib = $product->getCustomAttribute('erp_id');
            if ($bcAttrib) {
                $itemInfo['erp_id'] = $bcAttrib->getValue();
            } else {
                throw new \RuntimeException('Error can not get Erp Article/Item Id from Magento '.
                                            'product, is the attribute missing on the sku? ('.$sku.')');
            }

            $bcAttrib = $product->getCustomAttribute('erp_tax_rate');
            if ($bcAttrib) {
                $itemInfo['erp_tax_rate'] = $bcAttrib->getValue();
            } else {
                throw new \RuntimeException('Error can not get Erp Article/Item VAT Rate from Magento'.
                                            ' product, is the attribute missing on the sku? ('.$sku.')');
            }

            $bcAttrib = $product->getCustomAttribute('erp_tax_included');
            if ($bcAttrib) {
                $itemInfo['erp_tax_included'] = $bcAttrib->getValue();
            } else {
                throw new \RuntimeException('Error can not get Erp Article/Item VAT Included from Magento'
                .' product, is the attribute missing on the sku? ('.$sku.')');
            }

            return $itemInfo;
        } catch (NoSuchEntityException $e) {
            throw new \RuntimeException('Error can not get (or find) sku '.$sku.' in Magento');
        }
    }

    protected function run(): string
    {
        return $this->import();
    }

    protected function getServiceDescription(): string
    {
        return 'product import';
    }

    protected function getCapability(): string
    {
        return Capability::PRODUCT_IMPORT;
    }

    private function initCommonProductAttributes($product, $item, $isNew): string
    {
        $isDirty = '';

        // Set attributes for all web sites
        $product->setWebsiteIds($item->getWebsiteIds());
        $product->setStoreId(0);

        // Only set attribute, product category and metadata on new product.
        // It is possible that this data has been updated manually in Magento for existing products.
        $itemName = $item->getName();
        if ($isNew) {
            $product->setAttributeSetId($item->attributeSet);
            $product->setCategoryIds($item->getCategoryIds());
            if (Configurable::TYPE_CODE === $product->getTypeId()) {
                $itemName = $item->description;
            } else {
                // Do not set price on configurable products
                $product->setPrice($item->getPrice());
            }
            $product->setMetaTitle($itemName);
            $product->setMetaKeyword($itemName);
            $product->setMetaDescription($itemName);
            $product->setName($itemName);
            $isDirty .= 'new';
        }

        if ($product->getName() !== $itemName
            && Configurable::TYPE_CODE !== $product->getTypeId()) {
            $product->setName($itemName);
            $isDirty .= 'name';
        }

        if (((float) $product->getWeight()) !== ((float) $item->weight)) {
            $product->setWeight($item->weight);
            $isDirty .= '-weight';
        }

        return $isDirty;
    }

    private function storeProductWithURLCheck(
        \Magento\Catalog\Model\Product $product,
        \Ecomero\ErpCore\Model\Item $item
    ): void {
        try {
            $this->productRepository->save($product);
        } catch (\Magento\Framework\Exception\LocalizedException $exception) {
            // If two product in the same website ends up with the same URL key
            // then append the SKU and try to save the product again.
            if (false !== strpos($exception->getMessage(), 'URL key')) {
                $url = preg_replace('#[^0-9a-z]+#i', '-', $item->getName().'-'.$product->getSku());
                $this->logger->debug('    ...url key exists, generating new key : '.$url);
                $product->setUrlKey($url);
                $this->productRepository->save($product);
            } else {
                throw $exception;
            }
        }
    }

    private function createOrUpdateSimpleProducts(\Ecomero\ErpCore\Model\ResourceModel\Item\Collection $itemCollection): void
    {
        $currentProduct = 0;
        $maxProduct = $itemCollection->count();
        $stores = $this->storeManager->getStores(false, true);

        /** @var \Ecomero\ErpCore\Model\Item $item */
        foreach ($itemCollection->getItems() as $item) {
            ++$currentProduct;

            $skuDescription = $item->sku.', '.
                                $item->getName().', '.
                                $item->getPrice().', '.
                                $item->attributes['erp_id']->value;
            $stockItem = null;

            try {
                $product = $this->productRepository->get($item->sku);
                $stockItem = $this->stockRegistry->getStockItemBySku($item->sku);
                $isNew = false;
                $this->logger->debug('  - '.$currentProduct.'/'.$maxProduct.
                                     ' ['.$item->getCategoryLabel().']'.
                                     ' Product already imported, updating ('.$skuDescription.')');
            } catch (NoSuchEntityException $e) {
                $product = $this->productFactory->create();
                $product->setSku($item->sku);
                $product->setTypeId(Type::TYPE_SIMPLE);
                if ($item->isSimpleProduct()) {
                    $product->setVisibility(Visibility::VISIBILITY_BOTH);
                } else {
                    $product->setVisibility(Visibility::VISIBILITY_NOT_VISIBLE);
                }
                $product->setStatus(Status::STATUS_DISABLED);
                $isNew = true;
                $this->logger->debug('  - '.$currentProduct.'/'.$maxProduct.
                                     ' ['.$item->getCategoryLabel().']'.
                                     ' New product ('.$skuDescription.')');
            }

            $isDirty = $this->initCommonProductAttributes($product, $item, $isNew);

            if ($item->isEnabledFromErp) {
                $newStatus = $item->isEnabled ? Status::STATUS_ENABLED : Status::STATUS_DISABLED;
                if ($product->getStatus() != $newStatus) {
                    $product->setStatus($newStatus);
                    $isDirty .= '-enabled';
                }
            }

            // The cost attribute is not applicable to "configurable" product type
            if (((float) $product->getCost()) !== ((float) $item->cost)) {
                $product->setCost($item->cost);
                $isDirty .= '-cost';
            }

            if (null == $stockItem || $stockItem->getQty() != $item->inventory) {
                $product->setQuantityAndStockStatus([
                    'qty' => $item->inventory,
                    'is_in_stock' => $item->inventory > 0 ? 1 : 0,
                ]);
                $isDirty .= '-qty';
            }

            if ('' !== $isDirty) {
                $this->logger->debug('    ...found changes, saving product ('.$isDirty.')');
                $this->storeProductWithURLCheck($product, $item);
            }

            /** @var \Magento\Catalog\Model\Product $product */
            $product = $this->productRepository->get($item->sku);
            $item->magentoId = $product->getId();
            $item->associatedProductIds = [];

            foreach ($item->attributes as $attrib) {
                if ($attrib->assignAttributeToAttributeSet($product, 'Product Details')) {
                    $attrib->setAttribute($product);
                }
            }

            // Set product price for each website
            $lcy = strtolower($this->settings->getLCY());
            foreach ($stores as $storeName => $store) {
                $storeCurrency = $this->settings->getCurrency((int) $store->getId());
                $currency = strtolower($storeCurrency);
                if ('' == $currency || $currency == $lcy) {
                    $product->setPrice($item->getPrice());
                    $product->setStoreId($store->getId());
                    $this->productResourceModel->saveAttribute((object) $product, 'price');
                } elseif (array_key_exists($currency, $item->prices)) {
                    $product->setPrice($item->prices[$currency]);
                    $product->setStoreId($store->getId());
                    $this->productResourceModel->saveAttribute((object) $product, 'price');
                } else {
                    $this->logger->error('    ...no price in '.$currency.' found on item in Erp');
                }
            }
        }
    }

    private function createOrUpdateConfigurableProduct(\Ecomero\ErpCore\Model\Item $item): void
    {
        $configuarble_sku = $item->getCommmonName();

        if (null == $item->associatedProductIds) {
            $this->logger->error('  - No associated products found! ('.$configuarble_sku.')');

            return;
        }

        if (1 == count($item->associatedProductIds)) {
            $this->logger->debug("  - Single product found '".$item->getCommmonName().
                                 "', do not create a matching configurable product");

            return;
        }

        try {
            $product = $this->productRepository->get($configuarble_sku);
            $isNew = false;
            $this->logger->debug('  - Configurable product already exists, updating ('.$configuarble_sku.')');
        } catch (NoSuchEntityException $e) {
            $product = $this->productFactory->create();
            $product->setSku($configuarble_sku);
            $product->setTypeId(Configurable::TYPE_CODE);
            $product->setVisibility(Visibility::VISIBILITY_BOTH);
            $product->setStatus(Status::STATUS_ENABLED);
            $isNew = true;
            $this->logger->debug('  - New configurable product must be created ('.$configuarble_sku.')');
        }

        if (\Magento\ConfigurableProduct\Model\Product\Type\Configurable::TYPE_CODE !== $product->getTypeId()) {
            $this->logger->error('An existing simple product was found in Magento for '.$item->getCommmonName().".\n".
            "If you want to create a configurable product, then please delete the simple product in Magento and also in the ERP.\n".
            'After the product is deleted, please re-run the import.');

            return;
        }
        /** @var \Magento\ConfigurableProduct\Model\Product\Type\Configurable $configurableProduct */
        $configurableProduct = $product->getTypeInstance();

        $isDirty = $this->initCommonProductAttributes($product, $item, $isNew);

        if ('' !== $isDirty) {
            $this->logger->debug('    ...found changes, saving product ('.$isDirty.')');

            $product->setStockData(['use_config_manage_stock' => 1,
                'manage_stock' => 1,
                'is_qty_decimal' => 0,
                'is_in_stock' => 1, ]);

            // Super attribute
            $usedProductAttribArray = [];
            foreach ($item->attributes as $key => $value) {
                if (\Ecomero\ErpCore\Model\Attribute::YES === $value->usedInConfiguration) {
                    $attrib = $this->productResourceModel->getAttribute($key);
                    if ($attrib) {
                        $usedProductAttribArray[] = $attrib->getId();
                    }
                }
            }
            $configurableProduct->setUsedProductAttributeIds($usedProductAttribArray, $product);
            $configurableAttributesData = $configurableProduct->getConfigurableAttributesAsArray($product);
            $product->setCanSaveConfigurableAttributes(true);
            $product->setConfigurableAttributesData($configurableAttributesData);
            $configurableProductsData = [];
            $product->setConfigurableProductsData($configurableProductsData);

            $this->storeProductWithURLCheck($product, $item);
        }

        // Link simple products
        $usedProducts = $configurableProduct->getUsedProducts($product);
        $existingLinks = [];
        foreach ($usedProducts as $child) {
            $existingLinks[] = $child->getId();
        }

        // Only update if links are changed
        array_multisort($existingLinks);
        array_multisort($item->associatedProductIds);
        if (serialize($existingLinks) !== serialize($item->associatedProductIds)) {
            // Add missing products
            foreach ($usedProducts as $child) {
                if (!in_array($child->getId(), $item->associatedProductIds)) {
                    $item->associatedProductIds[] = $child->getId();
                }
            }

            $this->logger->debug('    ...found link changes, saving product');

            /** @var \Magento\Catalog\Model\Product $configurableProduct */
            $configurableProduct = $this->productRepository->get($configuarble_sku);
            $configurableProduct->setAssociatedProductIds($item->associatedProductIds);
            $configurableProduct->setCanSaveConfigurableAttributes(true);
            $this->storeProductWithURLCheck($product, $item);
        }
    }

    private function getUniqueProducts(object $itemCollection): array
    {
        $uniqueProdArray = [];

        foreach ($itemCollection->getItems() as $item) {
            if ('' == $item->getCommmonName()) {
                continue;
            }
            if (false == array_key_exists($item->getCommmonName(), $uniqueProdArray)) {
                $uniqueProdArray[$item->getCommmonName()] = $item;
            }
            if (null != $item->magentoId) {
                $uniqueProdArray[$item->getCommmonName()]->associatedProductIds[] = $item->magentoId;
            }
        }

        return $uniqueProdArray;
    }

    private function createProducts(\Ecomero\ErpCore\Model\ResourceModel\Item\Collection $itemCollection): void
    {
        if ($itemCollection->count() > 0) {
            $this->createOrUpdateSimpleProducts($itemCollection);

            $uniqueProducts = $this->getUniqueProducts($itemCollection);
            foreach ($uniqueProducts as $item) {
                $this->createOrUpdateConfigurableProduct($item);
            }
        }
    }

    private function import(): string
    {
        $this->isImportRunning = true;

        $websiteName = '';
        $errorMessage = '';
        $forceUnlock = false;

        try {
            $this->storeManager->setCurrentStore(0);
            $websiteName = null;
            $allWebsites = $this->storeManager->getWebsites(false, true);
            $itemCollection = $this->itemCollectionFactory->create();
            foreach ($allWebsites as $website) {
                $websiteId = (int) $website->getId();

                if (!$this->settings->isEnabled($websiteId)) {
                    continue;
                }

                $websiteName = $this->storeManager->getWebsite($websiteId)->getName();

                $itemCategories = $this->settings->getCategoryMapping($websiteId);
                if (null === $itemCategories) {
                    continue;
                }
                $this->erp->setWebsite($websiteId);
                foreach ($itemCategories as $category) {
                    $categoryFilter = $category['category_filter'];
                    $categoryId = $category['product_category'];
                    $attribSetId = $category['attribute_set'];

                    $this->logger->info('Retrieving products for category "'.$categoryFilter.
                                        '" (Website : '.$websiteName.')');

                    $json = $this->erp->getItemList($categoryFilter, $forceUnlock);
                    $itemCollection->initItems($json, $websiteId, $categoryId, $categoryFilter, $attribSetId);

                    $json = $this->erp->getItemPrices($categoryFilter);
                    $itemCollection->initItemsPrices($json);

                    $json = $this->erp->getItemAttributes($categoryFilter);
                    $itemCollection->initItemsAttributes($json);
                }
                $this->erp->setItemImportCompleted();
            }
            if ($websiteName) {
                $this->erp->createCommonName($itemCollection);
                $this->createProducts($itemCollection);
            }
        } catch (\Laminas\Http\Exception\RuntimeException $exception) {
            $errorMessage = 'Website name : '.$websiteName."\n".$errorMessage.$exception->getMessage()."\n";
            $this->logger->error($errorMessage);
        } catch (\Magento\Framework\Exception\LocalizedException $exception) {
            $errorMessage = 'Website name : '.$websiteName."\n".$errorMessage.$exception->getMessage()."\n";
            $this->logger->error($errorMessage);
        } catch (\RuntimeException $exception) {
            $errorMessage = 'Website name : '.$websiteName."\n".$errorMessage.$exception->getMessage()."\n";
            $this->logger->error($errorMessage);
        }

        $this->isImportRunning = false;

        return $errorMessage;
    }
}
