<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Helper;

use Ecomero\ErpCore\Model\CustomerFactory;
use Ecomero\ErpCore\Model\Erp\ErpCustomerInterface;

class CustomerHelper
{
    protected $logger;
    protected $erp;
    protected $settings;
    protected $customerFactory;

    public function __construct(
        ErpLogger $logger,
        ErpCustomerInterface $erp,
        Settings $settings,
        CustomerFactory $customerFactory
    ) {
        $this->logger = $logger;
        $this->erp = $erp;
        $this->settings = $settings;
        $this->customerFactory = $customerFactory;
    }

    public function getCustomerFingerPrint(
        string $name,
        string $street,
        string $city,
        string $postalcode,
        string $email
    ): string {
        $customerInfo = $name.$street.$city.$postalcode.$email;
        $customerInfo = str_replace(' ', '', $customerInfo);
        $customerInfo = strtolower($customerInfo);

        return sha1($customerInfo);
    }

    public function getCustomerFromAddress(
        \Magento\Sales\Model\Order\Address $address
    ): \Ecomero\ErpCore\Model\Customer {
        if ($address->getCompany()) {
            $orderFingerPrint = $this->getCustomerFingerprint(
                $address->getCompany(),
                $address->getStreetLine(1),
                $address->getCity(),
                $address->getPostcode() ?? '',
                $address->getEmail()
            );
        } else {
            $orderFingerPrint = $this->getCustomerFingerprint(
                $address->getFirstname().$address->getLastname(),
                $address->getStreetLine(1),
                $address->getCity(),
                $address->getPostcode() ?? '',
                $address->getEmail()
            );
        }

        $json = $this->erp->getCustomerFromEmail($address->getEmail());
        if ($json) {
            // We found a customer, let us check if it is the same customer
            foreach ($json as $item) {
                $customer = $this->customerFactory->create(['customerData' => $item]);

                $bcFingerPrint = $this->getCustomerFingerprint(
                    $customer->getFirstName().' '.$customer->getLastName(),
                    $customer->getStreet(),
                    $customer->getCity(),
                    $customer->getPostalcode(),
                    $customer->getEmail()
                );

                if ('fingerprint' === $this->settings->getCustomerIdentificationMethod($this->erp->getWebsite())
                    && $orderFingerPrint == $bcFingerPrint) {
                    $this->logger->debug('  - Found customer in Erp with id '.$customer->getId().
                                         ' and fingerprint '.$bcFingerPrint);

                    return $customer;
                }

                if ('email' === $this->settings->getCustomerIdentificationMethod($this->erp->getWebsite())
                    && $customer->getEmail() === $address->getEmail()) {
                    $this->logger->debug('  - Found customer in Erp with id '.$customer->getId().
                                         ' and email '.$address->getEmail());

                    return $customer;
                }
            }
        }

        if ($this->settings->getAutoCreateCustomers($this->erp->getWebsite())) {
            $json = $this->erp->createCustomer(
                $address->getCompany(),
                $address->getFirstname(),
                $address->getLastname(),
                $address->getStreetLine(1),
                $address->getCity(),
                $address->getPostcode(),
                $address->getEmail(),
                $address->getTelephone(),
                $address->getCountryId(),
                $address->getRegion(),
                $address->getVatId()
            );

            if ($json) {
                $customer = $this->customerFactory->create(['customerData' => $json]);
                $this->logger->debug('  - Creating new customer in Erp with id '.$customer->getId());

                return $customer;
            }

            throw new \RuntimeException('Error while creating customer in Erp');
        }

        throw new \RuntimeException('Customer not found in Erp (auto create customer is disabled in the store setup)');
    }
}
