<?php

declare(strict_types=1);

namespace Ecomero\ErpCore\Helper;

use Magento\Framework\Mail\Template\TransportBuilder;
use Magento\Sales\Model\Order;

class Email
{
    protected $transportBuilder;
    protected $scopeConfig;
    protected $settings;

    public function __construct(
        TransportBuilder $transportBuilder,
        Settings $settings,
        \Magento\Framework\App\Config\ScopeConfigInterface $scopeConfig
    ) {
        $this->transportBuilder = $transportBuilder;
        $this->scopeConfig = $scopeConfig;
        $this->settings = $settings;
    }

    public function sendOrderStatus(Order $order): void
    {
        if (!$this->settings->isSendOrderStatusUpdate()) {
            return;
        }

        $vars = $this->getEmailVars($order);

        $this->transportBuilder
            ->setTemplateIdentifier($this->settings->getOrderStatusEmailTemplate())
            ->setTemplateOptions(
                [
                    'area' => \Magento\Framework\App\Area::AREA_FRONTEND,
                    'store' => $order->getStoreId(),
                ]
            )
            ->setTemplateVars(
                $vars
            )
            ->setFrom(
                [
                    'name' => $this->settings->getSalesRepresentativeName(),
                    'email' => $this->settings->getSalesRepresentativeEmail(),
                ]
            )
            ->addTo($order->getCustomerEmail())
            ->getTransport()
            ->sendMessage()
        ;
    }

    public function getEmailVars(Order $order): array
    {
        $vars = [];

        if ($order->getErpOrderNo()) {
            $vars['erp_order_id'] = $order->getErpOrderNo();
        }

        if ($order->getIncrementId()) {
            $vars['order_id'] = $order->getIncrementId();
        }

        if ($order->getId()) {
            $vars['entity_id'] = $order->getId();
        }

        if ($email = $this->getStoreEmail()) {
            $vars['store_email'] = $email;
        }

        if ($phone = $this->getStorePhoneNumber()) {
            $vars['store_phone'] = $phone;
        }

        /**
         * @var \DateTime $date
         */
        if ($date = $order->getErpDeliveryDate()) {
            if ($fullDate = $date->format('Y-m-d')) {
                $vars['full_date'] = $fullDate;
            }

            if ($year = $date->format('Y')) {
                $vars['delivery_year'] = $year;
            }
            if ($week = $date->format('W')) {
                $vars['delivery_week'] = $week;
            }
            if ($day = $date->format('d')) {
                $vars['delivery_day'] = $day;
            }
        }

        return $vars;
    }

    private function getStoreEmail()
    {
        return $this->scopeConfig->getValue('trans_email/ident_support/email') ?? '';
    }

    private function getStorePhoneNumber()
    {
        return $this->scopeConfig->getValue('general/store_information/phone') ?? '';
    }
}
