<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Model;

use Ecomero\ErpCore\Helper\ErpLogger;
use Magento\Catalog\Model\Config;
use Magento\Eav\Api\AttributeManagementInterface;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\Store\Model\StoreManagerInterface;

class Attribute
{
    public const NO = 0;
    public const DEFAULT = 1;
    public const YES = 2;

    public $type;
    public $name;
    public $value;
    public $locale;
    public $usedInConfiguration;    // This attribute is used in a configurable product

    protected $logger;
    protected $config;
    protected $attributeManagement;
    protected $storeManager;
    protected $arrayManager;

    public function __construct(
        ErpLogger $logger,
        Config $config,
        AttributeManagementInterface $attributeManagement,
        StoreManagerInterface $storeManager,
        ArrayManager $arrayManager
    ) {
        $this->logger = $logger;
        $this->config = $config;
        $this->attributeManagement = $attributeManagement;
        $this->storeManager = $storeManager;
        $this->arrayManager = $arrayManager;
        $this->usedInConfiguration = self::DEFAULT;
    }

    public function isUsedInConfiguration(): bool
    {
        return ('Option' == $this->type && self::DEFAULT == $this->usedInConfiguration)
                || ('Option' == $this->type && $this->usedInConfiguration = self::YES);
    }

    public function setAttribute(\Magento\Catalog\Model\Product $product): void
    {
        $attr = $product->getResource()->getAttribute($this->name)->setStoreId(0);
        if (null == $attr) {
            $this->logger->error('The attribute '.$this->name.' can not be found in Magento!');
        } else {
            $attributeValue = $this->value;
            $existingValue = $product->getData($this->name);

            if ('Option' == $this->type) {
                // If we have an attribute of type option, we need to make a lookup
                if (self::DEFAULT == $this->usedInConfiguration) {
                    $this->usedInConfiguration = self::YES;
                }
                $attributeValue = $attr->getSource()->getOptionId($this->value);
                if (null == $attributeValue) {
                    $this->logger->error("The attribute option '".$this->value.
                                         "' cannot be found in Magento for attribute ".$this->name);

                    return;
                }
            }

            if ('Decimal' == $this->type) {
                $attributeValue = (float) str_replace(',', '.', (string) $attributeValue);
                $existingValue = (float) str_replace(',', '.', (string) $existingValue);
            }

            if ($attributeValue !== $existingValue) {
                $storeIds = $this->getStoreFromLocale($this->locale);
                foreach ($storeIds as $storeId) {
                    $product->addAttributeUpdate($this->name, $attributeValue, $storeId);
                }
            }
        }
    }

    public function assignAttributeToAttributeSet($product, string $attributeGroup): bool
    {
        if (array_key_exists($this->name, $product->getAttributes())) {
            return true;
        }

        $attributeSet = $product->getAttributeSetId();

        $group_id = $this->config->getAttributeGroupId($attributeSet, $attributeGroup);

        try {
            $this->attributeManagement->assign(
                'catalog_product',
                $attributeSet,
                $group_id,
                $this->name,
                100
            );
        } catch (NoSuchEntityException $ex) {
            return false;
        }

        return true;
    }

    private function getStoreFromLocale(?string $locale): array
    {
        $storeIds = [];
        $locale2store = [
            'EN' => ['EUR', 'sek_en', 'eur_en'],
            'SE' => ['SEK', 'sek_sv', 'eur_sv'],
            'SV' => ['SEK', 'sek_sv', 'eur_sv']
        ];
        $codes = $this->arrayManager->get($locale, $locale2store);
        if ($codes) {
            foreach ($codes as $code) {
                foreach ($this->storeManager->getStores() as $store) {
                    if (strtolower($store->getCode()) === strtolower($code)) {
                        $storeIds[] = (int) $store->getId();
                    }
                }
            }
        }

        return $storeIds ?: [0];
    }
}
