<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Helper;

use Magento\Config\Model\ResourceModel\Config\Data\CollectionFactory;
use Magento\Framework\App\Config\ScopeConfigInterface;
use Magento\Framework\App\Config\Storage\WriterInterface;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\Encryption\EncryptorInterface;
use Magento\Framework\Serialize\Serializer\Json;

class Settings extends AbstractHelper
{
    public const DISCOUNT_LINE = 'LINE';
    public const DISCOUNT_TOTAL = 'TOTAL';

    protected $encryptor;
    protected $configWriter;
    protected $configCollectionFactory;
    protected $unserializer;

    public function __construct(
        Context $context,
        EncryptorInterface $encryptor,
        WriterInterface $configWriter,
        CollectionFactory $configCollectionFactory,
        Json $unserializer
    ) {
        parent::__construct($context);
        $this->encryptor = $encryptor;
        $this->configWriter = $configWriter;
        $this->configCollectionFactory = $configCollectionFactory;
        $this->unserializer = $unserializer;
    }

    public function isEnabled(int $websiteId): bool
    {
        return '1' == $this->scopeConfig->getValue('erp_core/general/enabled', 'website', $websiteId);
    }

    public function getCustomerIdentificationMethod(int $websiteId): string
    {
        return $this->scopeConfig->getValue('erp_core/order/customer_id_method', 'website', $websiteId);
    }

    public function getAutoCreateCustomers(int $websiteId): bool
    {
        return '1' == $this->scopeConfig->getValue('erp_core/order/auto_create_customers', 'website', $websiteId);
    }

    public function getAllowNewProducts($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): bool
    {
        return '1' == $this->scopeConfig->getValue('erp_core/catalog/allow_new_products', $scope);
    }

    public function getAllowNameUpdate($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): bool
    {
        return '1' == $this->scopeConfig->getValue('erp_core/catalog/allow_name_update', $scope);
    }

    public function getCategoryMapping(int $websiteId): ?array
    {
        $tableConfig = $this->scopeConfig->getValue('erp_core/catalog/category_mapping', 'website', $websiteId);
        if ($tableConfig) {
            $tableConfigResults = $this->unserializer->unserialize($tableConfig);
            if (is_array($tableConfigResults)) {
                return $tableConfigResults;
            }
        }

        return null;
    }

    public function sendShipmentConfirmation($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): bool
    {
        return '1' == $this->scopeConfig->getValue('erp_core/shipping/send_shipment_confirmation', $scope);
    }

    public function getUseShippingTaxFromOrder($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): bool
    {
        return '1' == $this->scopeConfig->getValue('erp_core/shipping/shipping_tax_order', $scope);
    }

    public function getWarehouseLocation(int $websiteId): string
    {
        return $this->scopeConfig->getValue('erp_core/shipping/warehouse_location', 'website', $websiteId) ?? '';
    }

    public function getShippingChargeItem(int $websiteId, string $magentoShippingMethod): string
    {
        $item = '';
        $tableConfig = $this->scopeConfig->getValue('erp_core/shipping/shipping_agent', 'website', $websiteId);
        if ($tableConfig) {
            $tableConfigResults = $this->unserializer->unserialize($tableConfig);
            if (is_array($tableConfigResults)) {
                foreach ($tableConfigResults as $tableConfigResult) {
                    if (key_exists('freight_item', $tableConfigResult)) {
                        $item = $tableConfigResult['freight_item'];
                    }
                    $magentoShipping = $tableConfigResult['magento_shipping'];
                    if (false !== stripos($magentoShippingMethod, $magentoShipping)) {
                        return $item;
                    }
                }
            }
        }

        return $item;
    }

    public function getSalesRepresentativeName(): string
    {
        return $this->scopeConfig->getValue('trans_email/ident_sales/name');
    }

    public function getSalesRepresentativeEmail(): string
    {
        return $this->scopeConfig->getValue('trans_email/ident_support/email');
    }

    public function getGiftCardItemId(int $websiteId): string
    {
        $itemId = '';
        $itemConfig = $this->scopeConfig->getValue('erp_core/giftcard/item_id', 'website', $websiteId);
        if ($itemConfig) {
            $itemId = explode('#', $itemConfig)[0];
        }

        return $itemId;
    }

    public function isGiftCardItem(int $websiteId, string $itemId): bool
    {
        return $itemId === $this->getGiftCardItemId($websiteId);
    }

    public function isShippingChargeItem(int $websiteId, string $itemId): bool
    {
        $item = '';
        $tableConfig = $this->scopeConfig->getValue('erp_core/shipping/shipping_agent', 'website', $websiteId);
        if ($tableConfig) {
            $tableConfigResults = $this->unserializer->unserialize($tableConfig);
            if (is_array($tableConfigResults)) {
                foreach ($tableConfigResults as $tableConfigResult) {
                    if (key_exists('freight_item', $tableConfigResult)) {
                        $item = $tableConfigResult['freight_item'];
                        $parts = explode('#', $item);
                        if ($parts[0] === $itemId) {
                            return true;
                        }
                    }
                }
            }
        }

        return false;
    }

    public function getShippingAgentService(int $websiteId, string $magentoShippingMethod): string
    {
        $agent = '';
        $tableConfig = $this->scopeConfig->getValue('erp_core/shipping/shipping_agent', 'website', $websiteId);
        if ($tableConfig) {
            $tableConfigResults = $this->unserializer->unserialize($tableConfig);
            if (is_array($tableConfigResults)) {
                foreach ($tableConfigResults as $tableConfigResult) {
                    if (key_exists('agent_service', $tableConfigResult)) {
                        $agent = $tableConfigResult['agent_service'];
                    }
                    $magentoShipping = $tableConfigResult['magento_shipping'];
                    if (false !== stripos($magentoShippingMethod, $magentoShipping)) {
                        return $agent;
                    }
                }
            }
        }

        return $agent;
    }

    public function getShippingTrackUrl(int $websiteId, string $magentoShippingMethod): string
    {
        $url = '';
        $tableConfig = $this->scopeConfig->getValue('erp_core/shipping/shipping_agent', 'website', $websiteId);
        if ($tableConfig) {
            $tableConfigResults = $this->unserializer->unserialize($tableConfig);
            if (is_array($tableConfigResults)) {
                foreach ($tableConfigResults as $tableConfigResult) {
                    if (key_exists('track_url', $tableConfigResult)) {
                        $url = $tableConfigResult['track_url'];
                    }
                    $magentoShipping = $tableConfigResult['magento_shipping'];
                    if (false !== stripos($magentoShippingMethod, $magentoShipping)) {
                        return $url;
                    }
                }
            }
        }

        return '';
    }

    public function isSendOrderStatusUpdate(): bool
    {
        return (bool) $this->scopeConfig->getValue('erp_core/order/send_order_status');
    }

    public function getOrderStatusEmailTemplate(): string
    {
        return $this->scopeConfig->getValue('erp_core/order/order_status_email_template') ?? 'erp_core_order_order_status_email_template';
    }

    public function getSalesPerson(int $websiteId): ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/sales_person', 'website', $websiteId);
    }

    public function getProfitCenter(int $websiteId): ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/profit_center', 'website', $websiteId);
    }

    public function getPaymentMethod(int $websiteId): ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/payment_method', 'website', $websiteId);
    }

    public function getPaymentTerms(int $websiteId): ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/payment_terms', 'website', $websiteId);
    }

    public function getTermsOfDelivery(int $websiteId): ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/terms_of_delivery', 'website', $websiteId);
    }

    public function getWayOfDelivery(int $websiteId): ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/way_of_delivery', 'website', $websiteId);
    }

    public function getCustomerCategory(int $websiteId): ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/customer_category', 'website', $websiteId);
    }

    public function getCustomerDistrict(int $websiteId): ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/customer_district', 'website', $websiteId);
    }

    public function getOrderStatusSend($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): string
    {
        $status = $this->scopeConfig->getValue('erp_core/order/order_status_send', $scope);

        return null === $status ? '' : $status;
    }

    public function getOrderStatusSent($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): string
    {
        $status = $this->scopeConfig->getValue('erp_core/order/order_status_sent', $scope);

        return null === $status ? '' : $status;
    }

    public function sendInvoiceConfirmation($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): bool
    {
        return '1' == $this->scopeConfig->getValue('erp_core/order/send_invoice_confirmation', $scope);
    }

    public function getDefaultTaxCode(int $websiteId): ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/default_tax_code', 'website', $websiteId);
    }

    public function getUseTaxClass(): bool
    {
        return '1' == $this->scopeConfig->getValue('erp_core/order/use_tax_class');
    }

    public function getOrderPostingMode($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): ?string
    {
        return $this->scopeConfig->getValue('erp_core/order/posting_mode', $scope);
    }

    public function getLCY($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): string
    {
        $lcy = $this->scopeConfig->getValue('erp_core/currency/lcy_erp', $scope);

        return null === $lcy ? '' : $lcy;
    }

    public function getCurrency(?int $storeId): string
    {
        $currency = $this->scopeConfig->getValue('erp_core/currency/currency', 'store', $storeId);
        if (null === $currency) {
            $currency = '';
        }

        return $currency;
    }

    public function stopOnError($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): bool
    {
        return '1' == $this->scopeConfig->getValue('erp_core/cron/stop_on_error', $scope);
    }

    public function getLogDirectory($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): ?string
    {
        return $this->scopeConfig->getValue('erp_core/cron/log_directory', $scope);
    }

    public function getNotificationReceiver($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): ?string
    {
        return $this->scopeConfig->getValue('erp_core/cron/receiver_notification', $scope);
    }

    public function getLogRetentionTime($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): ?string
    {
        return $this->scopeConfig->getValue('erp_core/cron/log_retention', $scope);
    }

    public function getCronProductImport($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): string
    {
        return $this->scopeConfig->getValue('erp_core/cron/cron_product_import', $scope) ?? '';
    }

    public function getCronOrderExport($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): string
    {
        return $this->scopeConfig->getValue('erp_core/cron/cron_order_export', $scope) ?? '';
    }

    public function getCronOrderImport($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): string
    {
        return $this->scopeConfig->getValue('erp_core/cron/cron_order_import', $scope) ?? '';
    }

    public function getCronInvoiceExport($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): string
    {
        return $this->scopeConfig->getValue('erp_core/cron/cron_invoice_export', $scope) ?? '';
    }

    public function getCronShipmentImport($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): string
    {
        return $this->scopeConfig->getValue('erp_core/cron/cron_shipment_import', $scope) ?? '';
    }

    public function getCronInvoiceImport($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): string
    {
        return $this->scopeConfig->getValue('erp_core/cron/cron_invoice_import', $scope) ?? '';
    }

    public function getCronCreditMemoExport($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): string
    {
        return $this->scopeConfig->getValue('erp_core/cron/cron_credit_memo_export', $scope) ?? '';
    }

    public function getCronReturnImport($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): string
    {
        return $this->scopeConfig->getValue('erp_core/cron/cron_return_import', $scope) ?? '';
    }

    public function getCronCompanyImport($scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): string
    {
        return $this->scopeConfig->getValue('erp_core/cron/cron_company_import', $scope) ?? '';
    }

    public function getRefundItemPosAdj(int $websiteId): ?string
    {
        return $this->scopeConfig->getValue('erp_core/refund/adjustment_positive', 'website', $websiteId);
    }

    public function getRefundItemNegAdj(int $websiteId): ?string
    {
        return $this->scopeConfig->getValue('erp_core/refund/adjustment_negative', 'website', $websiteId);
    }

    public function getDiscountMethod(int $storeId): string
    {
        return $this->scopeConfig->getValue('erp_core/discount/method', 'store', $storeId) ?? self::DISCOUNT_LINE;
    }

    public function getDiscountErpItem(int $storeId): string
    {
        $itemId = '';
        $itemConfig = $this->scopeConfig->getValue('erp_core/discount/item_id', 'store', $storeId);
        if ($itemConfig) {
            $itemId = explode('#', $itemConfig)[0];
        }

        return $itemId;
    }

    // @return string
    public function getAPIEndPoint(string $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): ?string
    {
        return $this->scopeConfig->getValue('erp_core/general/api_end_point', $scope);
    }

    // @return string
    public function getAPIKey(string $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): ?string
    {
        $secret = $this->scopeConfig->getValue('erp_core/general/api_key', $scope);

        return $this->encryptor->decrypt($secret);
    }

    public function getAppName(string $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): ?string
    {
        return $this->scopeConfig->getValue('erp_core/general/ws_app', $scope);
    }

    // @return string
    public function getAppSecret(string $scope = ScopeConfigInterface::SCOPE_TYPE_DEFAULT): ?string
    {
        $secret = $this->scopeConfig->getValue('erp_core/general/ws_secret', $scope);

        return $this->encryptor->decrypt($secret);
    }

    // @return string
    public function getCommonNameAttributes(int $websiteId): ?string
    {
        return $this->scopeConfig->getValue('erp_core/catalog/product_attributes_common_name', 'website', $websiteId);
    }

    // @return string
    public function getConfigurableAttributes(int $websiteId): ?string
    {
        return $this->scopeConfig->getValue('erp_core/catalog/product_attributes_configurable', 'website', $websiteId);
    }

    public function isPostCreditMemo(int $websiteId): bool
    {
        return '1' == $this->scopeConfig->getValue('erp_core/refund/post_credit_memo', 'website', $websiteId);
    }

    public function getTaxClassForShipping(): string
    {
        return $this->scopeConfig->getValue('tax/classes/shipping_tax_class');
    }

}
