<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Model;

use Ecomero\ErpCore\Helper\AttributeHelper;
use Ecomero\ErpCore\Helper\ErpLogger;
use Magento\Catalog\Model\Config;
use Magento\Eav\Api\AttributeManagementInterface;
use Magento\Framework\App\Filesystem\DirectoryList;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Framework\Filesystem\Io\File;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\Store\Model\StoreManagerInterface;

class Attribute
{
    public const NO = 0;
    public const DEFAULT = 1;
    public const YES = 2;

    public $type;
    public $name;
    public $value;
    public $locale;
    public $fileName;
    public $usedInConfiguration;    // This attribute is used in a configurable product

    protected $logger;
    protected $config;
    protected $attributeManagement;
    protected $storeManager;
    protected $arrayManager;
    protected $directoryList;
    protected $file;
    protected $attributeHelper;

    protected static $imagePathCache = [];

    public function __construct(
        ErpLogger $logger,
        Config $config,
        AttributeManagementInterface $attributeManagement,
        StoreManagerInterface $storeManager,
        ArrayManager $arrayManager,
        DirectoryList $directoryList,
        File $file,
        AttributeHelper $attributeHelper
    ) {
        $this->logger = $logger;
        $this->config = $config;
        $this->attributeManagement = $attributeManagement;
        $this->storeManager = $storeManager;
        $this->arrayManager = $arrayManager;
        $this->directoryList = $directoryList;
        $this->usedInConfiguration = self::DEFAULT;
        $this->file = $file;
        $this->attributeHelper = $attributeHelper;
    }

    public function isUsedInConfiguration(): bool
    {
        return ('Option' == $this->type && self::DEFAULT == $this->usedInConfiguration)
                || ('Option' == $this->type && $this->usedInConfiguration = self::YES);
    }

    public function setAttribute(\Magento\Catalog\Model\Product $product): void
    {
        $attr = $product->getResource()->getAttribute($this->name)->setStoreId(0);
        if (null == $attr) {
            $this->logger->error('The attribute '.$this->name.' can not be found in Magento!');
        } else {
            $attributeValue = $this->value;
            $existingValue = $product->getData($this->name);

            if ('Image' == $this->type
                && $this->fileName
                && ('no_selection' === $existingValue || null === $existingValue)) {
                $product->setMediaGallery(['images' => [], 'values' => []]);

                if (array_key_exists($this->fileName, Attribute::$imagePathCache)) {
                    $this->logger->info("          Importing image {$this->fileName}");
                    $localImgPath = Attribute::$imagePathCache[$this->fileName];
                } else {
                    $this->logger->info("          Downloading image {$this->fileName}");
                    $localImgPath = $this->downloadImage($this->fileName, $attributeValue);
                    Attribute::$imagePathCache[$this->fileName] = $localImgPath;
                }

                if ($localImgPath) {
                    try {
                        $product->addImageToMediaGallery($localImgPath, ['image', 'thumbnail', 'small_image'], false, false);
                        $product->save();
                    } catch (\Exception $e) {
                        $this->logger->debug($e->getMessage());
                    }
                }
            }

            if ('Option' == $this->type) {
                // If we have an attribute of type option, we need to make a lookup
                if (self::DEFAULT == $this->usedInConfiguration) {
                    $this->usedInConfiguration = self::YES;
                }
                $attributeValue = $attr->getSource()->getOptionId($this->value);
                if (null == $attributeValue) {
                    $attributeValue = $this->attributeHelper->createOrGetId($this->name, $this->value);
                    if (!$attributeValue) {
                        $this->logger->error("The attribute option '".$this->value.
                        "' cannot be found in Magento for attribute ".$this->name);

                        return;
                    }
                }
            }

            if ('Decimal' == $this->type) {
                $attributeValue = (float) str_replace(',', '.', (string) $attributeValue);
                $existingValue = (float) str_replace(',', '.', (string) $existingValue);
            }

            if ($attributeValue !== $existingValue) {
                $storeIds = $this->getStoreFromLocale($this->locale);
                foreach ($storeIds as $storeId) {
                    $product->addAttributeUpdate($this->name, $attributeValue, $storeId);
                }
            }
        }
    }

    public function assignAttributeToAttributeSet($product, string $attributeGroup): bool
    {
        if (array_key_exists($this->name, $product->getAttributes())) {
            return true;
        }

        $attributeSet = $product->getAttributeSetId();

        $group_id = $this->config->getAttributeGroupId($attributeSet, $attributeGroup);

        try {
            $this->attributeManagement->assign(
                'catalog_product',
                $attributeSet,
                $group_id,
                $this->name,
                100
            );
        } catch (NoSuchEntityException $ex) {
            return false;
        }

        return true;
    }

    private function downloadImage(string $fileName, string $imageUrl): ?string
    {
        $tmpDir = $this->directoryList->getPath(DirectoryList::MEDIA).DIRECTORY_SEPARATOR.'tmp';
        $this->file->checkAndCreateFolder($tmpDir);
        $newFileName = $tmpDir.'/'.$fileName;
        $downloadUrl = $this->urlEncode($imageUrl);
        $result = $this->file->read($downloadUrl, $newFileName);

        return $result ? $newFileName : null;
    }

    private function urlEncode(string $url): string
    {
        $url = str_replace(' ', '%20', $url);
        $url = str_replace("'", '%27', $url);
        $url = str_replace('å', '%C3%A5', $url);
        $url = str_replace('ä', '%C3%A4', $url);
        $url = str_replace('ö', '%C3%B6', $url);
        $url = str_replace('Å', '%C3%85', $url);
        $url = str_replace('Ä', '%C3%84', $url);

        return str_replace('Ö', '%C3%96', $url);
    }

    private function getStoreFromLocale(?string $locale): array
    {
        $storeIds = [];
        $locale2store = [
            'EN' => ['EUR', 'sek_en', 'eur_en'],
            'SE' => ['SEK', 'sek_sv', 'eur_sv'],
            'SV' => ['SEK', 'sek_sv', 'eur_sv'],
        ];
        $codes = $this->arrayManager->get($locale, $locale2store);
        if ($codes) {
            foreach ($codes as $code) {
                foreach ($this->storeManager->getStores() as $store) {
                    if (strtolower($store->getCode()) === strtolower($code)) {
                        $storeIds[] = (int) $store->getId();
                    }
                }
            }
        }

        return $storeIds ?: [0];
    }
}
