<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Model\Invoice;

use Ecomero\ErpCore\Helper\CronLock;
use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\OrderExportHelper;
use Ecomero\ErpCore\Helper\OrderProcessor;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Helper\TaxItemHelper;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpInvoiceInterface;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order\Email\Sender\InvoiceSenderFactory;
use Magento\Sales\Model\ResourceModel\Order\Invoice as InvoiceRepository;
use Magento\Sales\Model\ResourceModel\Order\Invoice\CollectionFactory as InvoiceCollectionFactory;
use Magento\Sales\Model\Service\InvoiceServiceFactory;

class InvoiceImport extends \Ecomero\ErpCore\Model\Executor
{
    public const DOCUMENT_TYPE = 'Invoice';

    protected $orderRepository;
    protected $invoiceServiceFactory;
    protected $invoiceCollectionFactory;
    protected $invoiceRepository;
    protected $invoiceSenderFactory;
    protected $taxItemHelper;
    protected $orderExportHelper;
    protected $orderProcessor;
    protected $settings;
    protected $erp;
    protected $arrayManager;

    public function __construct(
        ErpLogger $logger,
        CronLock $cronLock,
        Notification $notification,
        ErpInvoiceInterface $erp,
        OrderRepositoryInterface $orderRepository,
        InvoiceServiceFactory $invoiceServiceFactory,
        InvoiceRepository $invoiceRepository,
        InvoiceCollectionFactory $invoiceCollectionFactory,
        InvoiceSenderFactory $invoiceSenderFactory,
        TaxItemHelper $taxItemHelper,
        OrderExportHelper $orderExportHelper,
        OrderProcessor $orderProcessor,
        Settings $settings,
        ArrayManager $arrayManager
    ) {
        parent::__construct(
            $logger,
            $cronLock,
            $notification,
            $erp
        );

        $this->orderRepository = $orderRepository;
        $this->invoiceServiceFactory = $invoiceServiceFactory;
        $this->invoiceRepository = $invoiceRepository;
        $this->invoiceCollectionFactory = $invoiceCollectionFactory;
        $this->invoiceSenderFactory = $invoiceSenderFactory;
        $this->taxItemHelper = $taxItemHelper;
        $this->settings = $settings;
        $this->orderExportHelper = $orderExportHelper;
        $this->orderProcessor = $orderProcessor;
        $this->erp = $erp;
        $this->arrayManager = $arrayManager;
    }

    public function processOrder(\Magento\Sales\Model\Order $order): void
    {
        $this->orderExportHelper->initialize(self::DOCUMENT_TYPE, (int) $order->getStoreId());

        if (!$order->getErpOrderNo() || false !== strpos($order->getErpOrderNo(), 'ERROR')) {
            return;
        }

        $erpInvoices = $this->erp->getInvoiceFromOrder($order->getIncrementId());
        foreach ($erpInvoices as $erpInvoice) {
            $this->createInvoice($order, $erpInvoice);
        }
    }

    public function createInvoice(\Magento\Sales\Model\Order $order, array $erpInvoice)
    {
        try {
            $erpInvoiceNumber = $this->arrayManager->get('number', $erpInvoice);
            if ($order->canInvoice() && $erpInvoiceNumber) {
                if ($this->isOrderAlreadyInMagento($erpInvoiceNumber)) {
                    $this->logger->debug('  - Invoice already imported for invoice : '.$erpInvoiceNumber);

                    return;
                }

                if ($order->canShip()) {
                    $this->logger->debug('Found order that is not shipped '.$order->getErpOrderNo());

                    return;
                }

                $this->logger->debug('  - Creating invoice document for invoice : '.$erpInvoiceNumber);

                $bcInvoiceLines = $this->arrayManager->get('invoiceItems', $erpInvoice);

                $itemsArray = [];
                $shippingAmount = 0;
                $total = 0;
                $giftCardAmount = 0;
                foreach ($bcInvoiceLines as $bcInvoiceLine) {
                    $items = $this->getOrderItemIdFromSku(
                        $order,
                        $bcInvoiceLine['sku'],
                        $bcInvoiceLine['qty']
                    );
                    foreach ($items as $key => $value) {
                        $itemsArray[$key] = $value;
                    }

                    if ($items) {
                        $total += $bcInvoiceLine['amount'];
                    }
                }

                $invoice = $this->invoiceServiceFactory->create()->prepareInvoice($order, $itemsArray);

                $shippingAmount = $this->arrayManager->get('shippingFee', $erpInvoice);
                $invoice->setShippingAmount($shippingAmount);
                if (0 == $order->getShippingAmount() && 0 != $shippingAmount) {
                    $order->setShippingAmount($shippingAmount);
                }

                $giftCardAmount = $order->getBaseGiftCardsAmount() ?? (float) 0;
                $invoice->setBaseGiftCardsAmount($giftCardAmount);
                $invoice->setGiftCardsAmount($giftCardAmount);
                $invoice->setSubtotal($total);
                $invoice->setBaseSubtotal($total);
                $invoice->setGrandTotal($total + $shippingAmount);
                $invoice->setBaseGrandTotal($total + $shippingAmount);

                $invoice->setRequestedCaptureCase(\Magento\Sales\Model\Order\Invoice::CAPTURE_ONLINE);
                $invoice->setErpInvoiceNo($erpInvoiceNumber);
                $invoice->register();
                $this->invoiceRepository->save($invoice);
                $this->orderRepository->save($order);  // Order must be saved to update its status to closed

                if ($this->settings->sendInvoiceConfirmation()) {
                    $this->invoiceSenderFactory->create()->send($invoice);
                    $this->orderExportHelper->addCommentToOrder(
                        $order,
                        "Notified customer about invoice {$invoice->getId()}."
                    );
                }
            }
        } catch (\Throwable  $e) {
            $this->logger->critical($e->getMessage());
        }
    }

    protected function run(bool $force): string
    {
        return $this->orderProcessor->iterateOrders(
            $this->settings->getOrderStatusSent(),
            [$this, 'processOrder']
        );
    }

    protected function getServiceDescription(): string
    {
        return 'invoice import';
    }

    protected function getCapability(): string
    {
        return Capability::INVOICE_IMPORT;
    }

    private function getOrderItemIdFromSku(object $order, string $sku, float $qty): array
    {
        $itemsArray = [];
        foreach ($order->getAllItems() as $orderItem) {
            if ($orderItem->getSku() == $sku) {
                $itemsArray[$orderItem->getId()] = (float) $qty;
            }
        }

        return $itemsArray;
    }

    private function isOrderAlreadyInMagento(string $invoiceNo): bool
    {
        $invoiceCollecion = $this->invoiceCollectionFactory->create();
        $invoiceCollecion->addAttributeToFilter('erp_invoice_no', ['eq' => $invoiceNo]);
        foreach ($invoiceCollecion->getData() as $order) {
            return true;
        }

        return false;
    }
}
