<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Helper;

use Ecomero\ErpCore\Model\Catalog\ProductBase;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\Sales\Model\ResourceModel\Order\Tax\Item;
use Magento\Tax\Model\TaxClass\Repository;

class TaxItemHelper
{
    protected $taxItem;
    protected $productBase;
    protected $arrayManager;
    protected $defaultTaxCode;
    protected $taxClassRepository;
    protected $settings;

    public function __construct(
        Item $taxItem,
        ProductBase $productBase,
        ArrayManager $arrayManager,
        Repository $taxClassRepository,
        Settings $settings
    ) {
        $this->taxItem = $taxItem;
        $this->productBase = $productBase;
        $this->arrayManager = $arrayManager;
        $this->taxClassRepository = $taxClassRepository;
        $this->settings = $settings;
    }

    public function setDefaultTaxCode(?string $defaultTaxCode)
    {
        $this->defaultTaxCode = $defaultTaxCode ? $defaultTaxCode : '';
    }

    public function getDefaultTaxCode()
    {
        return $this->defaultTaxCode;
    }

    public function getTaxCodeForOrderLine(string $orderId, \Magento\Sales\Model\Order\Item $item): string
    {
        if ($this->settings->getUseTaxClass()) {
            $product = $item->getProduct();
            $taxClassId = $product->getTaxClassId();
            $taxClass = $this->taxClassRepository->get($taxClassId);
            return $taxClass->getClassName();
        } else {
            $itemId = $item->getItemId();
            $taxItems = $this->taxItem->getTaxItemsByOrderId($orderId);
            foreach ($taxItems as $taxItem) {
                if ($taxItem['item_id'] == $itemId) {
                    return $taxItem['code'];
                }
            }
        }

        return $this->defaultTaxCode;
    }

    public function getTaxCodeForFirstProduct(string $orderId): string
    {
        if ($this->settings->getUseTaxClass()) {
            $taxClassId = $this->settings->getTaxClassForShipping();
            $taxClass = $this->taxClassRepository->get($taxClassId);
            return $taxClass->getClassName();
        } else {
            $taxItems = $this->taxItem->getTaxItemsByOrderId($orderId);
            foreach ($taxItems as $taxItem) {
                if ('product' === $taxItem['taxable_item_type']) {
                    return $taxItem['code'];
                }
            }
        }

        return $this->defaultTaxCode;
    }

    public function getShippingTaxCodeForOrder(string $orderId): string
    {
        if ($this->settings->getUseTaxClass()) {
            $taxClassId = $this->settings->getTaxClassForShipping();
            $taxClass = $this->taxClassRepository->get($taxClassId);
            return $taxClass->getClassName();
        } else {
            $taxItems = $this->taxItem->getTaxItemsByOrderId($orderId);
            foreach ($taxItems as $taxItem) {
                if ('shipping' === $taxItem['taxable_item_type']) {
                    return $taxItem['code'];
                }
            }
        }

        return $this->defaultTaxCode;
    }

    public function getPriceIncludesTaxForFirstProduct(\Magento\Sales\Model\Order $order): bool
    {
        $orderItems = $order->getAllItems();
        foreach ($orderItems as $item) {
            if (\Magento\Catalog\Model\Product\Type::TYPE_SIMPLE !== $item->getProductType()) {
                continue;
            }
            $bcItem = $this->productBase->getErpItemInfoFromSKU($item->getSku());
            $isTaxIncluded = $this->arrayManager->get('erp_tax_included', $bcItem);
            if ($isTaxIncluded && '1' == $isTaxIncluded) {
                return true;
            }
        }

        return false;
    }
}
