<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Model\Shipping;

use Ecomero\ErpCore\Helper\CronLock;
use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Helper\OrderProcessor;
use Ecomero\ErpCore\Helper\Settings;
use Ecomero\ErpCore\Model\Capability;
use Ecomero\ErpCore\Model\Erp\ErpShippingInterface;
use Magento\Framework\Stdlib\ArrayManager;
use Magento\Sales\Api\Data\ShipmentCommentCreationInterfaceFactory;
use Magento\Sales\Api\Data\ShipmentItemCreationInterfaceFactory;
use Magento\Sales\Api\Data\ShipmentTrackInterfaceFactory;
use Magento\Sales\Api\ShipmentRepositoryInterface;
use Magento\Sales\Api\ShipOrderInterfaceFactory;
use Magento\Sales\Model\Order\ShipmentFactory;
use Magento\Sales\Model\ResourceModel\Order\Shipment\CollectionFactory as ShipmentCollectionFactory;

class ShippingImport extends \Ecomero\ErpCore\Model\Executor
{
    public const DOCUMENT_TYPE = 'Shipment';

    protected $shipmentFactory;
    protected $shipmentTrackFactory;
    protected $shipmentDocFactory;
    protected $shipmentRepository;
    protected $erp;
    protected $orderProcessor;
    protected $shipOrderFactory;
    protected $shipmentItemCreationFactory;
    protected $shipmentCommentCreationFactory;
    protected $settings;
    protected $arrayManager;

    public function __construct(
        ErpLogger $logger,
        Notification $notification,
        CronLock $cronLock,
        ErpShippingInterface $erp,
        ShipmentFactory $shipmentFactory,
        ShipmentTrackInterfaceFactory $shipmentTrackFactory,
        ShipmentCollectionFactory $shipmentDocFactory,
        ShipmentRepositoryInterface $shipmentRepository,
        OrderProcessor $orderProcessor,
        ShipOrderInterfaceFactory $shipOrderFactory,
        ShipmentItemCreationInterfaceFactory $shipmentItemCreationFactory,
        ShipmentCommentCreationInterfaceFactory $shipmentCommentCreationFactory,
        Settings $settings,
        ArrayManager $arrayManager
    ) {
        parent::__construct(
            $logger,
            $cronLock,
            $notification,
            $erp
        );

        $this->shipmentFactory = $shipmentFactory;
        $this->shipmentTrackFactory = $shipmentTrackFactory;
        $this->shipmentDocFactory = $shipmentDocFactory;
        $this->shipmentRepository = $shipmentRepository;
        $this->settings = $settings;
        $this->orderProcessor = $orderProcessor;
        $this->shipOrderFactory = $shipOrderFactory;
        $this->shipmentItemCreationFactory = $shipmentItemCreationFactory;
        $this->shipmentCommentCreationFactory = $shipmentCommentCreationFactory;
        $this->erp = $erp;
        $this->arrayManager = $arrayManager;
    }

    public function processOrder(\Magento\Sales\Model\Order $order): void
    {
        $shipments = $this->erp->getShippment($order->getIncrementId());

        foreach ($shipments as $shipment) {
            $this->createShippingDoc($shipment, $order);
        }
    }

    protected function run(bool $force): string
    {
        return $this->orderProcessor->iterateOrders(
            $this->settings->getOrderStatusSent(),
            [$this, 'processOrder']
        );
    }

    protected function getServiceDescription(): string
    {
        return 'shipment import';
    }

    protected function getCapability(): string
    {
        return Capability::SHIPMENT_IMPORT;
    }

    private function getQtyForSku(array $shippingItems, string $sku): int
    {
        $qty = 0;
        foreach ($shippingItems as $item) {
            if ($item['sku'] == $sku) {
                $qty += $item['qty'];
            }
        }

        return (int) $qty;
    }

    private function existShippingDoc(string $shippingNo): bool
    {
        $shipmentDocCollecion = $this->shipmentDocFactory->create();
        $shipmentDocCollecion->addAttributeToFilter('erp_sales_shipment_no', ['eq' => $shippingNo]);

        return $shipmentDocCollecion->getTotalCount() > 0;
    }

    private function createShippingDoc(array $erpShipment, \Magento\Sales\Model\Order $order): void
    {
        // Check if shipping document already is created
        $externalShipmentId = $this->arrayManager->get('id', $erpShipment);
        if ($this->existShippingDoc($externalShipmentId)) {
            return;
        }

        if ($order->canShip()) {
            $isAnythingToShip = false;
            $items = [];
            foreach ($order->getAllItems() as $orderItem) {
                // Check if order item has qty to ship or is virtual
                $shippingItems = $this->arrayManager->get('items', $erpShipment);
                $qtyShipped = $this->getQtyForSku($shippingItems, $orderItem->getSku());
                if (!$orderItem->getQtyToShip() || 0 == $qtyShipped || $orderItem->getIsVirtual()) {
                    continue;
                }
                $isAnythingToShip = true;
                $item = $this->shipmentItemCreationFactory->create();
                $item->setOrderItemId($orderItem->getId());
                $item->setQty($qtyShipped);
                $items[] = $item;
            }

            if ($isAnythingToShip) {
                $this->logger->debug("  - Creating shipping document for shipment : {$externalShipmentId}");
                $tracks = $this->getTrackingNumbers($order, $erpShipment);

                /** @var \Magento\Sales\Api\Data\ShipmentCommentCreationInterface $comment */
                $comment = $this->shipmentCommentCreationFactory->create();
                $comment->setComment('Shippment automatically created by ERP integration');
                $comment->setIsVisibleOnFront(false);

                $shipmentId = $this->shipOrderFactory->create()->execute(
                    $order->getId(),
                    $items,
                    $this->settings->sendShipmentConfirmation(),
                    true,
                    $comment,
                    $tracks
                );

                // Store external ref on shipment
                $shipment = $this->shipmentRepository->get($shipmentId);
                $shipment->setErpSalesShipmentNo($externalShipmentId);
                $this->shipmentRepository->save($shipment);
            }
        }
    }

    private function getTrackingNumbers(\Magento\Sales\Model\Order $order, array $erpShipment): array
    {
        $tracking = [];
        $packages = $this->arrayManager->get('packages', $erpShipment);
        $erpShippingAgent = $this->arrayManager->get('shippingAgent', $erpShipment);
        $carrierCode = $order->getShippingMethod();

//        $trackingUrl = $this->arrayManager->get('trackingUrl', $erpShipment);

        foreach ($packages as $package) {
            $trackingNumber = $this->arrayManager->get('trackingNumber', $package);
            $weight = $this->arrayManager->get('weight', $package);

            if ($trackingNumber && $erpShippingAgent) {
                $track = $this->shipmentTrackFactory->create();
                $track->setTrackNumber($trackingNumber);
                $track->setCarrierCode($carrierCode);
                $track->setTitle('Tracking Number');
                $track->setDescription($erpShippingAgent);
                $track->setWeight($weight);

                // If the warehouse is setting a carrier we can match, then use it
                // otherwise use the carrier description from the Magento order
                $url = $this->settings->getShippingTrackUrl(
                    $this->erp->getWebsite(),
                    $erpShippingAgent
                );
                if ('' === $url) {
                    $url = $this->settings->getShippingTrackUrl(
                        $this->erp->getWebsite(),
                        $order->getShippingDescription()
                    );
                    if ('' !== $url) {
                        $track->setDescription($order->getShippingDescription());
                    }
                }

                $tracking[] = $track;
            }
        }

        return $tracking;
    }
}
