<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Helper;

use Ecomero\ErpCore\Model\Erp\ErpAdapterInterface;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory;

class OrderProcessor
{
    protected $logger;
    protected $notification;
    protected $settings;
    protected $erp;
    protected $orderCollectionFactory;
    protected $orderRepository;

    public function __construct(
        ErpLogger $logger,
        Notification $notification,
        Settings $settings,
        ErpAdapterInterface $erp,
        CollectionFactory $orderCollectionFactory,
        OrderRepositoryInterface $orderRepository
    ) {
        $this->logger = $logger;
        $this->notification = $notification;
        $this->settings = $settings;
        $this->erp = $erp;
        $this->orderCollectionFactory = $orderCollectionFactory;
        $this->orderRepository = $orderRepository;
    }

    public function iterateOrders(string $orderStatusFilter, array $orderProcessor): string
    {
        $orderCollecion = $this->orderCollectionFactory->create()
            ->addFieldToSelect('entity_id')
            ->addFieldToSelect('increment_id')
            ->addFieldToSelect('customer_email')
            ->addFieldToSelect('order_currency_code')
            ->addFieldToSelect('erp_order_no')
        ;

        $errorMessage = '';
        $orderCollecion->addAttributeToFilter('status', ['in' => $orderStatusFilter]);

        $orderNumbers = [];
        foreach ($orderCollecion->getData() as $order) {
            $orderNumbers[] = $order['increment_id'];
        }

        $this->erp->preFetch($orderNumbers);

        foreach ($orderCollecion->getData() as $order) {
            $errorMessage = $this->processOrder((int) $order['entity_id'], $orderProcessor);
        }
        
        return $errorMessage;
    }

    public function processOrder(int $entityId, array $orderProcessor): string
    {
        $errorMessage = '';

        try {
            $fullOrderObj = $this->orderRepository->get($entityId);
            $storeId = (int) $fullOrderObj->getStoreId();
            $this->erp->setWebsiteFromStoredId($storeId);
            $activeErp = $this->erp->isErpEnabled();
            if ('' !== $activeErp) {
                $this->logger->debug('Processing Magento order '.$fullOrderObj->getIncrementId().' for '.$activeErp.' integration');
                call_user_func($orderProcessor, $fullOrderObj);
            } else {
                $this->logger->debug('Processing Magento order '.$fullOrderObj->getIncrementId().' no ERP integration active for website');
            }
        } catch (\Throwable $exception) {
            $errorMessage = $errorMessage.$this->handleException($fullOrderObj, $exception);
        }

        if ('' !== $errorMessage && $this->settings->stopOnError()) {
            $this->notification->notify($errorMessage);

            return '';
        }

        return $errorMessage;
    }

    public function handleException(\Magento\Sales\Model\Order $order, \Throwable $exception): string
    {
        $errorMessage = $exception->getMessage();

        $this->logger->error($errorMessage);

        $orderStatusHistories = $order->getStatusHistories();
        $orderStatusHistory = reset($orderStatusHistories);
        if (false !== $orderStatusHistory
            && $orderStatusHistory->getComment() === $errorMessage) {
            $orderStatusHistory->setCreatedAt(new \DateTime());
            $errorMessage = ''; // Prevent error to be sent as a notification again
        } else {
            $order->addStatusHistoryComment($errorMessage);
        }

        // Change status (not the state) on the order so it is not resent until manually handled
        if ('ERROR' === substr($errorMessage, 0, 5) &&
            !preg_match('/The tenant .* is not accessible./m', $errorMessage)
            && !preg_match('/Read timed out after [\w]+ seconds/m', $errorMessage)
            && !preg_match('/Too many requests reached./m', $errorMessage)
        ) {
            $order->setStatus('erp_error');
        }

        $order->save();

        return 'Store : '.$order->getStoreName()."\n".$errorMessage."\n";
    }
}
