<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Model\Catalog;

use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Settings;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Model\Product\Type;
use Magento\ConfigurableProduct\Model\Product\Type\Configurable;
use Magento\Framework\Exception\NoSuchEntityException;

class ProductBase
{
    protected $productRepository;
    protected $settings;
    protected $logger;

    public function __construct(
        ProductRepositoryInterface $productRepository,
        Settings $settings,
        ErpLogger $logger
    ) {
        $this->productRepository = $productRepository;
        $this->settings = $settings;
        $this->logger = $logger;
    }

    protected function initCommonProductAttributes(
        \Magento\Catalog\Model\Product $product,
        \Ecomero\ErpCore\Model\Item $item,
        bool $isNew
    ): string {
        $isDirty = '';

        // Set attributes for all web sites
        $product->setWebsiteIds($item->getWebsiteIds());
        $product->setStoreId(0);

        // Only set attribute, product category and metadata on new product.
        // It is possible that this data has been updated manually in Magento for existing products.
        $itemName = $item->getName();
        if ($isNew) {
            $product->setAttributeSetId($item->attributeSet);
            $product->setCategoryIds($item->getCategoryIds());
            if (Configurable::TYPE_CODE === $product->getTypeId()) {
                $itemName = $item->description;
            }
            $product->setMetaTitle($itemName);
            $product->setMetaKeyword($itemName);
            $product->setMetaDescription($itemName);
            $product->setName($itemName);
            $isDirty .= 'new';
        }

        if ((null === $product->getPrice() || ((float) $product->getPrice()) !== $item->getPrice())
            && Type::TYPE_SIMPLE === $product->getTypeId()) {
            $product->setPrice($item->getPrice());
            $isDirty .= 'price';
        }

        if ($product->getName() !== $itemName
            && Configurable::TYPE_CODE !== $product->getTypeId()
            && $this->settings->getAllowNameUpdate()) {
            $product->setName($itemName);
            $isDirty .= 'name';
        }

        if (null !== $item->weight
            && ((float) $product->getWeight()) !== ((float) $item->weight)) {
            $product->setWeight($item->weight);
            $isDirty .= '-weight';
        }

        return $isDirty;
    }

    protected function storeProductWithURLCheck(
        \Magento\Catalog\Model\Product $product,
        \Ecomero\ErpCore\Model\Item $item
    ): void {
        try {
            // This is a "work around" for a bug in Mageto Core.
            //
            // Save may throw a "The product was unable to be saved. Please try again." exception.
            //
            // If attributes are not read prior to a save operation when the product got tier prices it will fail.
            // The issue is that the attribute set is not cached and then the tier_prices are not loaded before the save.
            // This causes the save operation to insert new tier_prices instead of updating (getOrigData('tier_price) return null).
            //
            // Yes, this is a really confusing bug and it only occurs if the attribute set has not been accessed prior the save operation
            // (it must be in the cache)
            //
            // The problem is in \Magento\Catalog\Model\ResourceModel\_isApplicableAttribute() when it calls
            // $attribute->isInSet($object->getAttributeSetId()); then the attribute "tier_price" is not in the attribute set.
            // This is because isInSet relies on cached information and if this is not updated then "afterLoad" is not called for the tier_prices.
            //
            if($product->getId()) {
                $product->getAttributes();
            }
            $this->productRepository->save($product);
        } catch (\Magento\Framework\Exception\LocalizedException $exception) {
            // If two product in the same website ends up with the same URL key
            // then append the SKU and try to save the product again.
            if (false !== strpos($exception->getMessage(), 'URL key')) {
                $url = preg_replace('#[^0-9a-z]+#i', '-', $item->getName() . '-' . $product->getSku());
                $this->logger->debug('    ...url key exists, generating new key : ' . $url);
                $product->setUrlKey($url);
                $this->productRepository->save($product);
            } else {
                throw $exception;
            }
        }
    }

    public function getErpItemInfoFromSKU($sku): array
    {
        $itemInfo = [];

        try {
            $product = $this->productRepository->get($sku);
            $bcAttrib = $product->getCustomAttribute('erp_id');
            if ($bcAttrib) {
                $itemInfo['erp_id'] = $bcAttrib->getValue();
            } else {
                throw new \RuntimeException('Error can not get Erp Article/Item Id from Magento ' .
                                            'product, is the attribute missing on the sku? (' . $sku . ')');
            }

            $bcAttrib = $product->getCustomAttribute('erp_tax_rate');
            if ($bcAttrib) {
                $itemInfo['erp_tax_rate'] = $bcAttrib->getValue();
            } else {
                throw new \RuntimeException('Error can not get Erp Article/Item VAT Rate from Magento' .
                                            ' product, is the attribute missing on the sku? (' . $sku . ')');
            }

            $bcAttrib = $product->getCustomAttribute('erp_tax_included');
            if ($bcAttrib) {
                $itemInfo['erp_tax_included'] = $bcAttrib->getValue();
            } else {
                throw new \RuntimeException('Error can not get Erp Article/Item VAT Included from Magento'
                . ' product, is the attribute missing on the sku? (' . $sku . ')');
            }

            return $itemInfo;
        } catch (NoSuchEntityException $e) {
            throw new \RuntimeException('Error can not get (or find) sku ' . $sku . ' in Magento');
        }
    }
}
