<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Model\Catalog;

use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Settings;
use Magento\Catalog\Api\ProductRepositoryInterface;
use Magento\Catalog\Helper\Data as CatalogData;
use Magento\Catalog\Model\Product\Attribute\Source\Status;
use Magento\Catalog\Model\Product\Type;
use Magento\Catalog\Model\Product\Visibility;
use Magento\Catalog\Model\ProductFactory;
use Magento\Catalog\Model\ResourceModel\Product;
use Magento\CatalogInventory\Api\StockRegistryInterface;
use Magento\Framework\Api\SearchCriteriaBuilder;
use Magento\Framework\Exception\NoSuchEntityException;
use Magento\Store\Model\StoreManagerInterface;

class SimpleProduct extends ProductBase
{
    protected $productFactory;
    protected $productRepository;
    protected $productResourceModel;
    protected $settings;
    protected $storeManager;
    protected $stockRegistry;
    protected $catalogData;
    protected $searchCriteriaBuilder;

    public function __construct(
        ProductRepositoryInterface $productRepository,
        Settings $settings,
        ErpLogger $logger,
        StoreManagerInterface $storeManager,
        ProductFactory $productFactory,
        Product $productResourceModel,
        StockRegistryInterface $stockRegistry,
        CatalogData $catalogData,
        SearchCriteriaBuilder $searchCriteriaBuilder
    ) {
        parent::__construct(
            $productRepository,
            $settings,
            $logger
        );

        $this->productFactory = $productFactory;
        $this->productRepository = $productRepository;
        $this->productResourceModel = $productResourceModel;
        $this->storeManager = $storeManager;
        $this->stockRegistry = $stockRegistry;
        $this->catalogData = $catalogData;
        $this->searchCriteriaBuilder = $searchCriteriaBuilder;
    }

    public function createOrUpdateSimpleProducts(\Ecomero\ErpCore\Model\ResourceModel\Item\Collection $itemCollection): void
    {
        $currentProduct = 0;
        $maxProduct = $itemCollection->count();
        $websites = $this->storeManager->getWebsites(false, true);

        /** @var \Ecomero\ErpCore\Model\Item $item */
        foreach ($itemCollection->getItems() as $item) {
            ++$currentProduct;

            $skuDescription = $item->sku . ', ' .
                                $item->getName() . ', ' .
                                $item->getPrice() . ', ' .
                                $item->attributes['erp_id']->value;
            $stockItem = null;

            try {
                $product = $this->productRepository->get($item->sku);
                $stockItem = $this->stockRegistry->getStockItemBySku($item->sku);
                $isNew = false;
                $this->logger->debug('  - ' . $currentProduct . '/' . $maxProduct .
                                     ' [' . $item->getCategoryLabel() . ']' .
                                     ' Product already imported, updating (' . $skuDescription . ')');
            } catch (NoSuchEntityException $e) {
                $product = $this->productFactory->create();
                $product->setSku($item->sku);
                $product->setTypeId(Type::TYPE_SIMPLE);
                if ($item->isSimpleProduct()) {
                    $product->setVisibility(Visibility::VISIBILITY_BOTH);
                } else {
                    $product->setVisibility(Visibility::VISIBILITY_NOT_VISIBLE);
                }
                $product->setStatus(Status::STATUS_DISABLED);
                $isNew = true;
                $this->logger->debug('  - ' . $currentProduct . '/' . $maxProduct .
                                     ' [' . $item->getCategoryLabel() . ']' .
                                     ' New product (' . $skuDescription . ')');
            }

            $isDirty = $this->initCommonProductAttributes($product, $item, $isNew);

            if ($item->isEnabledFromErp) {
                $newStatus = $item->isEnabled ? Status::STATUS_ENABLED : Status::STATUS_DISABLED;
                if ($product->getStatus() != $newStatus) {
                    $product->setStatus($newStatus);
                    $isDirty .= '-enabled';
                }
            }

            // The cost attribute is not applicable to "configurable" product type
            if (((float) $product->getCost()) !== ((float) $item->cost)) {
                $product->setCost($item->cost);
                $isDirty .= '-cost';
            }

            if (null == $stockItem || $stockItem->getQty() != $item->inventory) {
                $product->setQuantityAndStockStatus([
                    'qty' => $item->inventory,
                    'is_in_stock' => $item->inventory > 0 ? 1 : 0,
                ]);
                $isDirty .= '-qty';
            }

            try {
                if ('' !== $isDirty) {
                    $this->logger->debug('    ...found changes, saving product (' . $isDirty . ')');
                    $this->storeProductWithURLCheck($product, $item);
                }
            } catch(\Throwable $t) {
                $this->logger->warning('    ...error cannot save product');
                continue;
            }

            /** @var \Magento\Catalog\Model\Product $product */
            $product = $this->productRepository->get($item->sku);
            $item->magentoId = $product->getId();
            $item->associatedProductIds = [];

            foreach ($item->attributes as $code => $attrib) {
                if ($attrib->assignAttributeToAttributeSet($product, 'Product Details')) {
                    if ('name' !== $code || $this->settings->getAllowNameUpdate()) {
                        $attrib->setAttribute($product);
                    }
                }
            }

            // Set product price for each website
            if (false === $this->catalogData->isPriceGlobal()) {
                foreach ($websites as $website) {
                    $websiteId = $website->getId();
                    if (array_key_exists($websiteId, $item->prices)) {
                        $product->setPrice($item->prices[$websiteId]);
                        $product->setStoreId($website->getDefaultStore()->getId());
                        $this->productResourceModel->saveAttribute((object) $product, 'price');
                    } else {
                        $this->logger->info('    ...no price for store ' . $website->getName() . ' found on item in Erp');
                    }
                }
            }
        }
    }

    public function disableRemovedProducts(\Ecomero\ErpCore\Model\ResourceModel\Item\Collection $itemCollection)
    {
        $criteria = $this->searchCriteriaBuilder
            ->addFilter('type_id', 'simple', 'eq')
            ->addFilter('erp_id', '', 'notnull')
            ->addFilter('status', Status::STATUS_ENABLED, 'eq')
            ->create();

        $allTimeImportedProducts = $this->productRepository->getList($criteria);
        foreach ($allTimeImportedProducts->getItems() as $product) {
            $sku = $product->getSku();
            $lastImportedItems = $itemCollection->getItems();
            if (!isset($lastImportedItems[$sku])) {
                $this->logger->info("Disabling imported product {$sku}, not found in ERP during last import");
                $product->setStatus(Status::STATUS_DISABLED);
                try {
                    $this->productResourceModel->saveAttribute((object) $product, 'status');
                } catch (\Throwable $t) {
                    $this->logger->error($t->getMessage());
                }
            }
        }
    }
}
