<?php

declare(strict_types=1);
/**
 *           a88888P8
 *          d8'
 * .d8888b. 88        .d8888b. 88d8b.d8b. .d8888b. .dd888b. .d8888b.
 * 88ooood8 88        88'  `88 88'`88'`88 88ooood8 88'    ` 88'  `88
 * 88.  ... Y8.       88.  .88 88  88  88 88.  ... 88       88.  .88
 * `8888P'   Y88888P8 `88888P' dP  dP  dP `8888P'  dP       `88888P'.
 *
 *           Copyright © eComero Management AB, All rights reserved.
 */

namespace Ecomero\ErpCore\Model;

use Ecomero\ErpCore\Helper\CronLock;
use Ecomero\ErpCore\Helper\ErpLogger;
use Ecomero\ErpCore\Helper\Notification;
use Ecomero\ErpCore\Model\Erp\ErpAdapterInterface;

abstract class Executor
{
    protected $logger;
    protected $cronLock;
    protected $notification;
    protected $erp;

    public function __construct(
        ErpLogger $logger,
        CronLock $cronLock,
        Notification $notification,
        ErpAdapterInterface $erp
    ) {
        $this->logger = $logger;
        $this->cronLock = $cronLock;
        $this->notification = $notification;
        $this->erp = $erp;
    }

    public function executeFromShell(\Symfony\Component\Console\Output\OutputInterface $output, bool $force): void
    {
        try {
            $this->logger->setOutput($output);
            $this->logger->info('Starting '.$this->getServiceDescription().' from command line');
            $this->execute($force);
        } catch (\Throwable $t) {
            $this->logger->critical((string) $t);
        }
    }

    public function executeFromCron(): void
    {
        try {
            $this->logger->info('Starting '.$this->getServiceDescription().' from cron job');
            $this->execute(false);
        } catch (\Throwable $t) {
            $this->logger->critical((string) $t);
        }
    }

    public function executeFromWeb(): void
    {
        try {
            $this->logger->info('Starting '.$this->getServiceDescription().' from web');
            $this->execute(false);
        } catch (\Throwable $t) {
            $this->logger->critical((string) $t);
        }
    }

    public function execute(bool $force): void
    {
        if (false == $this->cronLock->requestLock($this->logger)) {
            return;
        }

        $capabilities = $this->erp->getCapabilities();
        foreach ($capabilities as $capability) {
            if (!$capability->isCapabilitySupported($this->getCapability())) {
                $this->logger->warning('This operation is not supported by the '.$capability->getName().' integration');
                $this->cronLock->releaseLock();

                return;
            }
        }

        $start_time = microtime(true);

        $errorMessage = $this->run($force);

        $this->logger->info('Process completed in '.(microtime(true) - $start_time).' sec');
        $this->cronLock->releaseLock();

        if ('' !== $errorMessage) {
            $this->notification->notify($errorMessage);
        }
    }

    abstract protected function getServiceDescription(): string;

    abstract protected function getCapability(): string;

    abstract protected function run(bool $force): string;
}
